package org.unitedfront2.web;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.unitedfront2.domain.Account;
import org.unitedfront2.domain.Account.Role;
import org.unitedfront2.domain.Account.Status;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.Installer;
import org.unitedfront2.domain.MailAddrUsedByOtherException;
import org.unitedfront2.domain.SimpleUser;
import org.unitedfront2.domain.SimpleUserTable;
import org.unitedfront2.domain.UserCodeUsedByOtherException;
import org.unitedfront2.domain.UserNameUsedByOtherException;
import org.unitedfront2.domain.accesscontrol.OwnerOnly;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.domain.accesscontrol.UserOnly;
import org.unitedfront2.domain.communication.Blog;
import org.unitedfront2.domain.communication.BlogCodeUsedByOtherException;
import org.unitedfront2.domain.communication.Message;
import org.unitedfront2.domain.communication.MessageCodeUsedByOtherException;
import org.unitedfront2.domain.communication.MessageEntry;

/**
 * United Front ̃CXg[s郊XiNXłB
 *
 * @author kurokkie
 *
 */
public class ApplicationInstallListener implements ServletContextListener {

    /** O */
    protected final Log logger = LogFactory.getLog(getClass());

    /** {@link ApplicationContext} */
    private ApplicationContext context;

    /** hCt@Ng */
    private DomainFactory domainFactory;

    @Override
    public void contextDestroyed(ServletContextEvent sce) {
        // nothing
    }

    @Override
    public void contextInitialized(ServletContextEvent sce) {
        this.context = WebApplicationContextUtils
            .getRequiredWebApplicationContext(sce.getServletContext());
        this.domainFactory = (DomainFactory) context.getBean("domainFactory");
        SimpleUserTable simpleUserTable = (SimpleUserTable)
            context.getBean("simpleUserTable");
        if (simpleUserTable.findByCode(WebUtils.ADMIN_USER_CODE) == null) {
            logger.info("Installing United Front ...");
            DataSource dataSource = (DataSource) context.getBean("dataSource");
            TransactionTemplate transactionTemplate = new TransactionTemplate(
                    new DataSourceTransactionManager(dataSource));
            transactionTemplate.execute(new TransactionCallback() {

                @Override
                public Object doInTransaction(TransactionStatus status) {
                    SimpleUser admin = registerAdministrator();
                    registerSystemInformation(admin.getId());
                    registerTopPage(admin.getId());
                    registerAdministratorBlog(admin.getId());
                    return null;
                }
            });
            logger.info("Complete Installing United Front.");
        }
    }

    private SimpleUser registerAdministrator() {
        logger.info("Installing Administrator ...");

        // Account
        Account admin = new Account("administrator@example.com", "password",
                Status.AVAILABLE, Role.ROLE_ADMIN);
        admin.encrypt();
        admin = domainFactory.prototype(admin, Account.class);
        try {
            admin.store();
        } catch (MailAddrUsedByOtherException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalStateException(e);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("admin account: '" + admin + "'");
        }

        // User
        SimpleUserTable simpleUsers = (SimpleUserTable) context.getBeansOfType(
                SimpleUserTable.class).values().iterator().next();
        SimpleUser user = domainFactory.prototype(new SimpleUser(admin.getId(),
                WebUtils.ADMIN_USER_CODE, "Administrator"), SimpleUser.class);
        try {
            simpleUsers.install(user);
        } catch (UserCodeUsedByOtherException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalStateException(e);
        } catch (UserNameUsedByOtherException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalStateException(e);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("admin user: '" + user + "'");
        }

        // Installers for admin
        for (Object o : context.getBeansOfType(Installer.class).values()) {
            ((Installer) o).install(user.getId());
        }

        logger.info("Complete Installing Administrator.");

        return user;
    }

    private void registerSystemInformation(int userId) {
        logger.info("Installing System Information ...");

        // Empty Message
        Message message = domainFactory.prototype(new Message(
                WebUtils.SYSTEM_INFORMATION_MESSAGE_CODE, null, userId, userId,
                domainFactory.prototype(Public.class),
                domainFactory.prototype(OwnerOnly.class)), Message.class);
        try {
            message.store();
        } catch (MessageCodeUsedByOtherException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalStateException(e);
        }

        logger.info("Complete Installing System Information.");
    }

    private void registerTopPage(int userId) {
        logger.info("Installing Top Page ...");

        // Top Page Message
        Map<Locale, MessageEntry> m = new HashMap<Locale, MessageEntry>();
        m.put(Locale.JAPANESE, new MessageEntry(
                context.getMessage("Index", null, Locale.JAPANESE),
                context.getMessage("Index.description", null,
                        Locale.JAPANESE)));
        m.put(Locale.ENGLISH, new MessageEntry(
                context.getMessage("Index", null, Locale.ENGLISH),
                context.getMessage("Index.description", null,
                        Locale.ENGLISH)));
        Message message = domainFactory.prototype(new Message(
                WebUtils.TOP_PAGE_MESSAGE_CODE, m, userId, userId,
                domainFactory.prototype(Public.class),
                domainFactory.prototype(OwnerOnly.class)), Message.class);
        try {
            message.store();
        } catch (MessageCodeUsedByOtherException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalStateException(e);
        }

        logger.info("Complete Installing Top Page.");
    }

    private void registerAdministratorBlog(int adminId) {
        logger.info("Installing Administrator Blog ...");

        Message overview = domainFactory.prototype(new Message(adminId, adminId,
                domainFactory.prototype(Public.class),
                domainFactory.prototype(OwnerOnly.class)), Message.class);
        overview.setSubject(context.getMessage("Index.adminBlog.subject", null,
                Locale.JAPANESE), Locale.JAPANESE);
        overview.setBody(context.getMessage("Index.adminBlog.body", null,
                Locale.JAPANESE), Locale.JAPANESE);
        overview.setSubject(context.getMessage("Index.adminBlog.subject", null,
                Locale.ENGLISH), Locale.ENGLISH);
        overview.setBody(context.getMessage("Index.adminBlog.body", null,
                Locale.ENGLISH), Locale.ENGLISH);
        Blog blog = domainFactory.prototype(new Blog("admin", overview,
                adminId, domainFactory.prototype(Public.class),
                domainFactory.prototype(OwnerOnly.class),
                domainFactory.prototype(UserOnly.class)), Blog.class);
        try {
            blog.store();
        } catch (BlogCodeUsedByOtherException e) {
            // N肦Ȃ
            throw new IllegalStateException(e);
        }

        logger.info("Complete Installing Administrator Blog.");
    }
}
