package org.unitedfront2.domain;

import java.io.Serializable;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.unitedfront2.dao.AccountDao;

/**
 * AJEg\NXłB{@link #getPassword()} ͕̃pX[h擾łȂ_ɒӂĂ
 * BhCt@Ngň̃vg^Cv\bhgpꍇ͎̂悤ɃR[fBO܂B
 *
 * <pre>
 * account = new Account("account@example.com", "password", Status.AVAILABLE,
 *     Role.ROLE_USER);
 * account.encrypt();
 * account = domainFactory.prototype(account);
 * </pre>
 *
 * @author kurokkie
 */
public class Account implements Identifiable<Account>, Storable, Domain,
    Serializable {

    /** [̎ނ` */
    public static enum Role {
        /** ʗp */
        ROLE_USER,

        /** Ǘ */
        ROLE_ADMIN;
    }

    /** AJEg̏Ԃ̎ނ` */
    public static enum Status {
        /** L */
        AVAILABLE,

        /** ~ */
        LOCKED,

        /** L؂ */
        EXPIRED,

        /** pX[h̗L؂ */
        CREDENTIALS_EXPIRED,

        /** 폜 */
        DISABLED;
    }

    /** ÍÕpX[h̕\ (********) */
    public static final String HIDDEN_PASSWORD = "********";

    /** pX[hÍ (MD5) */
    public static final String PASSWORD_ENCRYPTION_ALGORITHM = "MD5";

    /** ꎞIɔsꂽF؃L[̗L̃ftHg (P) */
    public static final int DEFAULT_AUTH_KEY_EXPIRATION_SECONDS = 60 * 60 * 24;

    /** VAԍ */
    private static final long serialVersionUID = -2451333384583974061L;

    /**
     * ppō\pX[h_ɐ܂B
     *
     * @param minLength ŏ
     * @param maxLength ő啶
     * @require ${minLength} < ${maxLength}
     * @return ꂽpX[h
     * @ensure ${minLength} <= ${return.length} <= ${maxLength}
     */
    public static String createRandomPassword(int minLength, int maxLength) {
        int length = minLength
            + ((int) (Math.random() * (maxLength - minLength)));
        char[] password = new char[length];
        SecureRandom random = new SecureRandom();
        int maxNumberIndex = '9' - '0';
        int maxUpperAlphabetIndex = maxNumberIndex + ('Z' - 'A' + 1);
        int maxLowerAlphabetIndex = maxUpperAlphabetIndex + ('z' - 'a' + 1);
        for (int i = 0; i < password.length; i++) {
            int index = random.nextInt(maxLowerAlphabetIndex + 1);
            if (index <= maxNumberIndex) {
                password[i] = (char) ('0' + index);
            } else if (index <= maxUpperAlphabetIndex) {
                password[i] = (char) (('A' - 1) + (index - maxNumberIndex));
            } else {
                password[i]
                    = (char) (('a' - 1) + (index - maxUpperAlphabetIndex));
            }
        }
        return new String(password);
    }

    /** O */
    protected final transient Log logger = LogFactory.getLog(getClass());

    /** ID */
    private Integer id;

    /**
     * [AhX
     *
     * @invariant ӂȒl
     */
    private String mailAddr;

    /**
     * pX[h
     *
     *@@invariant ̃pX[h͎QƂłȂB
     */
    private String password;

    /** [ */
    private final Set<Role> roles = new HashSet<Role>();

    /** AJEg̏ */
    private Status status;

    /** pX[h͈Íς݂ */
    private Boolean encrypted = false;

    /** ꎞIɔsꂽF؃L[ */
    private transient String temporaryAuthKey;

    /** ꎞIɔsꂽF؃L[̗L (b) */
    private transient int authKeyExpirationSeconds
        = DEFAULT_AUTH_KEY_EXPIRATION_SECONDS;

    /** AJEgf[^ANZXIuWFNg */
    private transient AccountDao accountDao;

    public Account() {
        super();
    }

    /**
     * ${password} ͕̃pX[hnĂB
     *
     * @param password ̃pX[h
     */
    public Account(String mailAddr, String password, Status status, Role role) {
        super();
        this.mailAddr = mailAddr;
        this.password = password;
        this.status = status;
        roles.add(role);
    }

    /**
     * ${password} ͈Íς݂̃pX[hnĂB
     *
     * @param password Íς݂̃pX[h
     */
    public Account(Integer id, String mailAddr, String password, Status status,
            Set<Role> roles) {
        this();
        this.id = id;
        this.mailAddr = mailAddr;
        this.password = password;
        this.status = status;
        roles.addAll(roles);
    }

    /**
     * ̃pX[h͕\܂B
     *
     * @return \
     */
    @Override
    public String toString() {
        return new ToStringBuilder(this)
            .append("id", id)
            .append("mailAddr", mailAddr)
            .append("password", getPassword())
            .append("encrypted", encrypted)
            .append("roles", roles)
            .append("status", status).toString();
    }

    @Override
    public boolean equals(final Object other) {
        if (!(other instanceof Account)) {
            return false;
        }
        Account castOther = (Account) other;
        return new EqualsBuilder()
            .append(id, castOther.id)
            .append(mailAddr, castOther.mailAddr)
            .append(password, castOther.password)
            .append(encrypted, castOther.encrypted)
            .append(roles, castOther.roles)
            .append(status, castOther.status).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder()
            .append(id)
            .append(mailAddr)
            .append(password)
            .append(encrypted)
            .append(roles)
            .append(status).toHashCode();
    }

    @Override
    public boolean identify(Account other) {
        if (id == null || other.getId() == null) {
            return false;
        }
        return id.equals(other.getId());
    }

    /**
     * pX[ḧÍs܂BÍASÝA
     * {@link #PASSWORD_ENCRYPTION_ALGORITHM} łB̃\bh́ApX[hÍ
     * ȂƂɈxĂяoƂł܂BēxpX[hɕݒ肷ƁÃ\bh͌Ăяo
     * \ɂȂ܂B
     *
     * @require ${this.encrypted} is true
     * @ensure ${this.encrypted} is false
     * @ensure ${return}  ${old.password} ÍASY
     * {@link #PASSWORD_ENCRYPTION_ALGORITHM} ňÍłB
     * @throws IllegalStateException ɈÍς
     * @see #PASSWORD_ENCRYPTION_ALGORITHM
     */
    public void encrypt() throws IllegalStateException {
        if (isEncrypted()) {
            String errorMessage = "This account already encrypted.";
            throw new IllegalStateException(errorMessage);
        }
        setEncryptedPassword(encrypt(password));
    }

    private String encrypt(String password) {

        MessageDigest md;
        try {
            md = MessageDigest.getInstance(PASSWORD_ENCRYPTION_ALGORITHM);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e);
        }
        byte[] digest = md.digest(password.getBytes());
        StringBuffer sb = new StringBuffer(digest.length * 2);
        for (byte b : digest) {
            String s = Integer.toHexString(b & 0xff);
            if (s.length() == 1) {
                sb.append('0');
            }
            sb.append(s);
        }
        return sb.toString();
    }

    /**
     * ̃AJEgɁApX[h]܂B
     *
     * @param other ̃AJEg
     * @ensure ${this.password} = ${other.password}
     * @ensure ${this.encrypted} = ${other.encrypted}
     */
    public void copyPasswordTo(Account other) {
        if (isEncrypted()) {
            other.setEncryptedPassword(this.password);
        } else {
            other.setPlainPassword(this.password);
        }
    }

    /**
     * @require ${this.encrypted} is true
     * @throws MailAddrUsedByOtherException AJEg̃[AhXɓo^ς
     */
    @Override
    public void store() throws MailAddrUsedByOtherException {
        if (!isEncrypted()) {
            String message = "The account must be encripted before store.";
            logger.error(message);
            throw new IllegalStateException(message);
        }
        if (id == null) {
            if (accountDao.findByMailAddr(mailAddr) != null) {
                if (logger.isWarnEnabled()) {
                    logger.warn("MailAddr '" + mailAddr + "' already exists.");
                }
                throw new MailAddrUsedByOtherException(mailAddr);
            }
            accountDao.register(this);
        } else {
            Account foundAccount = accountDao.findByMailAddr(mailAddr);
            if (foundAccount != null && !foundAccount.getId().equals(id)) {
                if (logger.isWarnEnabled()) {
                    logger.warn("MailAddr '" + mailAddr + "' already exists.");
                }
                throw new MailAddrUsedByOtherException(mailAddr);
            }
            accountDao.update(this);
        }
    }

    /**
     * ꎞIȔF؃L[ݒ肵܂BF؃L[sĂȂꍇ͔s܂BÂF؃L[͍폜
     * B
     *
     * @require ${this} exists.
     * @ensure ${this.temporaryAuthKey} is retrieved.
     * @ensure ${this.temporaryAuthKey} is not null.
     */
    public void retrieveTemporaryAuthKey() {
        accountDao.deleteOldTemporaryAuthKey(authKeyExpirationSeconds);
        String temporaryAuthKey = accountDao.generateTemporaryAuthKey(id);
        if (logger.isDebugEnabled()) {
            logger.debug("The temporary auth key is '" + temporaryAuthKey
                    + "'");
        }
        this.temporaryAuthKey = temporaryAuthKey;
    }

    /**
     * ꎞIȔF؃L[폜܂B
     *
     * @require ${this} exists.
     * @ensure F؃L[f[^x[X폜
     * @ensure ${account.temporaryAuthKey} is null
     */
    public void deleteTemporaryAuthKey() {
        accountDao.deleteTemporaryAuthKey(id);
        this.temporaryAuthKey = null;
    }

    public boolean addRole(Role e) {
        return roles.add(e);
    }

    public boolean addRoles(Collection<? extends Role> c) {
        return roles.addAll(c);
    }

    public void clearRoles() {
        roles.clear();
    }

    public boolean removeRole(Object o) {
        return roles.remove(o);
    }

    public boolean removeRoles(Collection<?> c) {
        return roles.removeAll(c);
    }

    public boolean retainRoles(Collection<?> c) {
        return roles.retainAll(c);
    }

    public Integer getId() {
        return id;
    }

    public void setId(Integer id) {
        this.id = id;
    }

    public String getMailAddr() {
        return mailAddr;
    }

    public void setMailAddr(String mailAddr) {
        this.mailAddr = mailAddr;
    }

    /**
     * pX[hԂ܂BÍĂȂꍇ́ApX[hBԂ܂B
     *
     * @return pX[h
     * @ensure  ${this.encrypted}  <code>true</code> ȂA
     * {@link #HIDDEN_PASSWORD} Ԃ
     * @ensure  ${this.encrypted}  <code>false</code> ȂAÍASY
     * {@link #PASSWORD_ENCRYPTION_ALGORITHM} ňÍꂽpX[hԂ
     */
    public String getPassword() {
        if (isEncrypted()) {
            return this.password;
        } else {
            return HIDDEN_PASSWORD;
        }
    }

    /**
     * pX[hݒ肵܂Bݒ肷pX[hÍς݂Ȃ̂͂ł͎w肵Ȃ߁Aʏ
     * ̃\bh̗p͐܂BIɐݒ肷邽߂ɂ́A
     * {@link #setPlainPassword(String)} ܂
     * {@link #setEncryptedPassword(String)} 𗘗pĂB
     *
     * @param password ÍÕpX[h
     * @see #setPlainPassword(String)
     * @see #setEncryptedPassword(String)
     */
    public void setPassword(String password) {
        this.password = password;
    }

    /**
     * ÍÕpX[hԂ܂B<p>
     *
     *@O: ̃pX[hݒ肳Ă
     *
     * @return ̃pX[h
     * @throws IllegalStateException pX[hÍς
     */
    protected String getPlainPassword() throws IllegalStateException {
        if (isEncrypted()) {
            String message = "Password already encrypted.";
            throw new IllegalStateException(message);
        } else {
            return this.password;
        }
    }

    /**
     * ̃pX[hݒ肵܂B
     *
     * @param password ̃pX[h
     */
    public void setPlainPassword(String password) {
        this.encrypted = false;
        this.password = password;
    }

    /**
     * Í̃pX[hݒ肵܂B<p>
     *
     * 
     * <ul>
     *   <li>pX[h <code>password</code> ݒ肳</li>
     *   <li>{@link #isEncrypted()}  <code>true</code> Ԃ悤ɂȂ</li>
     * </ul>
     *
     * @param password Í̃pX[h
     * @ensure ${this.password} = ${password}
     * @ensure ${this.encrypted} is true
     */
    public void setEncryptedPassword(String password) {
        this.encrypted = true;
        this.password = password;
    }

    public boolean isEncrypted() {
        return encrypted;
    }

    public void setEncrypted(boolean encrypted) {
        this.encrypted = encrypted;
    }

    public Set<Role> getRoles() {
        return Collections.unmodifiableSet(roles);
    }

    public void setRoles(Set<Role> roles) {
        this.roles.clear();
        this.roles.addAll(roles);
    }

    public Status getStatus() {
        return status;
    }

    public void setStatus(Status status) {
        this.status = status;
    }

    public void setStatus(String status) {
        this.status = Status.valueOf(status);
    }

    /**
     * ꎞIɔsꂽF؃L[擾܂BF؃L[̔s
     * {@link #retrieveTemporaryAuthKey()} 𗘗pĂB
     *
     * @return ꎞIɔsꂽF؃L[
     */
    public String getTemporaryAuthKey() {
        return temporaryAuthKey;
    }

    public void setTemporaryAuthKey(String temporaryAuthKey) {
        this.temporaryAuthKey = temporaryAuthKey;
    }

    public void setAuthKeyExpirationSeconds(int authKeyExpirationSeconds) {
        this.authKeyExpirationSeconds = authKeyExpirationSeconds;
    }

    public void setAccountDao(AccountDao accountDao) {
        this.accountDao = accountDao;
    }
}
