package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.ThreadDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.communication.Message;
import org.unitedfront2.domain.communication.MessageCodeUsedByOtherException;
import org.unitedfront2.domain.communication.MessageTable;
import org.unitedfront2.domain.communication.Thread;
import org.unitedfront2.domain.communication.Thread.Status;

/**
 * {@link ThreadDao} ̎NXłB
 *
 * @author kurokkie
 *
 */
@Repository(value = "threadDao")
public class ThreadDaoImpl extends SimpleDaoSupport<Thread>
    implements ThreadDao {

    /** Xbh̃}bsONX */
    private static class ThreadRowMapper
        implements ParameterizedRowMapper<Thread> {

        /** hCt@Ng */
        private DomainFactory domainFactory;

        /** ANZXe[u */
        private AccessControlTable accessControlTable;

        /** bZ[We[u */
        private MessageTable messageTable;

        public ThreadRowMapper(DomainFactory domainFactory,
                AccessControlTable accessControlTable,
                MessageTable messageTable) {
            this.domainFactory = domainFactory;
            this.accessControlTable = accessControlTable;
            this.messageTable = messageTable;
        }

        @Override
        public Thread mapRow(ResultSet rs, int rowNum)
            throws SQLException {
            Thread t = domainFactory.prototype(Thread.class);
            t.setId(rs.getInt("Id"));
            t.setStatus(Status.valueOf(rs.getString("Status")));
            t.setOverview(messageTable.find(rs.getInt("OverviewId")));
            t.setOwnerId(rs.getInt("OwnerId"));
            t.setReadAccessControl(
                    accessControlTable.find(rs.getInt("ReadAccessControlId")));
            t.setWriteAccessControl(
                    accessControlTable.find(rs.getInt("WriteAccessControlId")));
            t.setPostAccessControl(
                    accessControlTable.find(rs.getInt("PostAccessControlId")));
            return t;
        }
    }

    /** ANZXe[u */
    @Autowired
    private AccessControlTable accessControlTable;

    /** bZ[We[u */
    @Autowired
    private MessageTable messageTable;

    /** bZ[W}bp[t@Ng */
    @Resource(name = "messageDao")
    private RowMapperFactory<Message> messageRowMapperFactory;

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<Thread> createRowMapper() {
        return new ThreadRowMapper(getDomainFactory(), accessControlTable,
                messageTable);
    }

    @Override
    protected Map<String, Object> toColumnValueMap(Thread thread) {
        Map<String, Object> args = super.toColumnValueMap(thread);
        args.put("OverviewId", thread.getOverview().getId());
        args.put("ReadAccessControlId",
                thread.getReadAccessControl().getId());
        args.put("WriteAccessControlId",
                thread.getWriteAccessControl().getId());
        args.put("PostAccessControlId",
                thread.getPostAccessControl().getId());
        return args;
    }

    @Override
    protected Class<Thread> getDomainClass() {
        return Thread.class;
    }

    //**************************************************************************
    // ThreadDao
    //**************************************************************************
    @Override
    public void register(Thread thread) {
        storeProperties(thread);
        super.register(thread);
    }

    @Override
    public void registerEntry(int id, int entryId) {
        getSimpleJdbcTemplate().update(
                "INSERT INTO Entry(Id, ThreadId) VALUES(?, ?)", entryId,
                id);
    }

    @Override
    public Message findEntry(int threadId, int entryId) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                    "SELECT m.* FROM Message m, Entry e WHERE e.Id = ? "
                    + "AND e.ThreadId = ? AND e.Id = m.Id",
                    messageRowMapperFactory.createRowMapper(), entryId,
                    threadId);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
    }

    @Override
    public List<Message> findEntries(int id) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM Message, Entry WHERE Entry.ThreadId = ? AND "
                + "Entry.Id = Message.Id ORDER BY Message.Id DESC",
                messageRowMapperFactory.createRowMapper(), id);
    }

    @Override
    public List<Message> findEntries(int id, int no, int num) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM Message, Entry WHERE Entry.ThreadId = ? AND "
                + "Entry.Id = Message.Id ORDER BY Message.Id DESC "
                + "LIMIT ?, ?",
                messageRowMapperFactory.createRowMapper(), id, no, num);
    }

    @Override
    public int countEntry(int id) {
        return getSimpleJdbcTemplate().queryForInt(
                "SELECT COUNT(Entry.Id) FROM Message, Entry WHERE "
                + "Entry.ThreadId = ? AND Entry.Id = Message.Id", id);
    }

    @Override
    public void update(Thread thread) {
        storeProperties(thread);
        super.update(thread);
    }

    @Override
    public void delete(int id) {
        Thread t = find(id);
        super.delete(id);
        try {
            t.getReadAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            t.getWriteAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            t.getPostAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
    }

    private void storeProperties(Thread thread) {
        thread.getReadAccessControl().store();
        thread.getWriteAccessControl().store();
        thread.getPostAccessControl().store();

        thread.getOverview().setOwnerId(thread.getOwnerId());
        thread.getOverview().setAuthorId(thread.getOwnerId());
        thread.getOverview().setReadAccessControl(
                thread.getReadAccessControl());
        thread.getOverview().setWriteAccessControl(
                thread.getWriteAccessControl());
        try {
            thread.getOverview().store();
        } catch (MessageCodeUsedByOtherException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalArgumentException(e);
        }
    }
}
