package org.unitedfront2.dao;

import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.unitedfront2.domain.Account;
import org.unitedfront2.domain.MailAddrUsedByOtherException;
import org.unitedfront2.domain.Account.Role;
import org.unitedfront2.domain.Account.Status;
import org.unitedfront2.test.DaoTestCaseSupport;
import org.unitedfront2.util.PropertyUtils;

public class AccountDaoTest extends DaoTestCaseSupport<Account> {

    @Autowired private AccountDao accountDao;

    @Override
    protected Account createTestDomain() {
        Account account = new Account();
        account.setMailAddr("accountDao@exmaple.com");
        account.setPlainPassword("password");
        account.addRole(Role.ROLE_USER);
        account.setStatus(Status.AVAILABLE);
        account.encrypt();
        return account;
    }

    @Override
    protected void editTestDomain(Account account) {
        account.setMailAddr("new_" + account.getMailAddr());
        account.setStatus(Status.LOCKED);
    }

    @Override
    protected SimpleFindable<Account> getDao() {
        return accountDao;
    }

    @Test
    @Override
    public void testRegister() {
        Account expected = createTestDomain();

        Assert.assertNull(PropertyUtils.getProperty(expected,
                getPrimaryKeyPropertyName()));
        Assert.assertTrue(expected.isEncrypted());
        accountDao.register(expected);
        Assert.assertNotNull(PropertyUtils.getProperty(expected,
            getPrimaryKeyPropertyName()));
        Assert.assertEquals(expected, accountDao.find(expected.getId()));
    }

    @Test
    @Override
    public void testUpdate() {
        Account expected = createTestDomain();
        accountDao.register(expected);
        editTestDomain(expected);

        Assert.assertNotNull(PropertyUtils.getProperty(expected,
                getPrimaryKeyPropertyName()));
        Assert.assertNotNull(accountDao.find(expected.getId()));
        Assert.assertTrue(expected.isEncrypted());
        accountDao.update(expected);
        Assert.assertEquals(expected, accountDao.find(expected.getId()));
    }

    @Test
    public void testFindByTemporaryAuthKey() {
        String key = accountDao.generateTemporaryAuthKey(account1.getId());
        Account found = accountDao.findByTemporaryAuthKey(key);
        Assert.assertEquals(account1, found);
        Assert.assertNotNull(found.getTemporaryAuthKey());
    }

    @Test
    public void testFindByTemporaryAuthKeyȂ() {
        Account found = accountDao.findByTemporaryAuthKey("---");
        Assert.assertNull(found);
    }

    @Test
    public void testGenerateTemporaryAuthKeys() {
        String key = accountDao.generateTemporaryAuthKey(account1.getId());
        Assert.assertNotNull(key);
    }

    @Test
    public void testGenerateTemporaryAuthKeyėp() {
        String key1 = accountDao.generateTemporaryAuthKey(account1.getId());
        String key2 = accountDao.generateTemporaryAuthKey(account1.getId());
        Assert.assertEquals(key1, key2);
    }

    @Test
    public void testDeleteOldTemporaryAuthKey() throws InterruptedException {
        String key1 = accountDao.generateTemporaryAuthKey(account1.getId());
        Thread.sleep(1000);
        String key2 = accountDao.generateTemporaryAuthKey(account2.getId());
        int seconds = 1;

        Assert.assertTrue(seconds >= 0);
        accountDao.deleteOldTemporaryAuthKey(seconds);
        Assert.assertNull(accountDao.findByTemporaryAuthKey(key1));
        Assert.assertNotNull(accountDao.findByTemporaryAuthKey(key2));
    }

    @Test
    public void testDeleteTemporarayAuthKey() {
        String key = accountDao.generateTemporaryAuthKey(account1.getId());
        Account found = accountDao.findByTemporaryAuthKey(key);
        Assert.assertEquals(account1, found);
        accountDao.deleteTemporaryAuthKey(account1.getId());
        found = accountDao.findByTemporaryAuthKey(key);
        Assert.assertNull(found);
    }

    @Test
    public void testFindByMailAddr() throws MailAddrUsedByOtherException {
        Account foundAccount = accountDao.findByMailAddr(account1.getMailAddr());
        Assert.assertEquals(account1, foundAccount);
    }

    @Test
    public void testFindByMailAddrȂ() {
        Assert.assertNull(accountDao.findByMailAddr("_not_found_"));
    }

    @Override
    @Ignore
    public void testDelete() {
        // ignore
    }
}
