package org.unitedfront2.domain;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.unitedfront2.dao.SimpleUserDao;
import org.unitedfront2.domain.communication.Profile;
import org.unitedfront2.domain.communication.ProfileTable;

/**
 * Vvȃ[U\NXłB
 *
 * @author kurokkie
 *
 */
public class SimpleUser implements Storable, Serializable, User, Domain {

    /** ƔfłsԊũftHg (15) */
    public static final long DEFAULT_ACTIVE_INTERSPACE = 15 * 60 * 1000;

    /** VAԍ */
    private static final long serialVersionUID = 2995463513876662631L;

    /**
     * [UXg̒ɁAw肵[Uƈv郆[U܂܂Ă邩ǂ肵܂B
     *
     * @param users [UXg
     * @param user [U
     * @return ܂܂Ă <code>true</code> A܂܂ĂȂ <code>false</code>
     */
    public static boolean exists(Collection<? extends User> users, User user) {
        for (User f : users) {
            if (f.identify(user)) {
                return true;
            }
        }
        return false;
    }

    /**
     * {@link SimpleUser} Xg {@link User} Xgɕϊ܂B
     *
     * @param simpleUsers {@link SimpleUser} Xg
     * @return {@link User} Xg
     */
    public static List<User> toUserList(List<SimpleUser> simpleUsers) {
        List<User> list = new ArrayList<User>(simpleUsers.size());
        for (SimpleUser u : simpleUsers) {
            list.add(u);
        }
        return list;
    }

    /**
     * œnSẴ[ŨvtB[𕜌܂B
     *
     * @param simpleUsers [UXg
     */
    public static void retrieveProfiles(List<SimpleUser> simpleUsers) {
        for (SimpleUser u : simpleUsers) {
            u.retrieveProfile();
        }
    }

    /** O */
    protected final transient Log logger = LogFactory.getLog(getClass());

    /** [U ID iAJEg ID@j */
    private Integer id;

    /** [UR[h */
    private String code;

    /** [U */
    private String name;

    /** ŏIANZX */
    private Date lastAccessDate;

    /** vtB[ */
    private transient Profile profile;

    /** ƔfłsԊu */
    private transient long activeInterSpace = DEFAULT_ACTIVE_INTERSPACE;

    /** [Uf[^ANZXIuWFNg */
    private transient SimpleUserDao simpleUserDao;

    /** vtB[e[u */
    private transient ProfileTable profileTable;

    public SimpleUser() {
        super();
    }

    public SimpleUser(Integer id, String code, String name) {
        super();
        this.id = id;
        this.code = code;
        this.name = name;
    }

    public SimpleUser(Integer id, String code, String name,
            Date lastAccessDate) {
        this(id, code, name);
        this.lastAccessDate = (Date) lastAccessDate.clone();
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this)
            .append("id", id)
            .append("code", code)
            .append("name", name)
            .append("lastAccessDate", lastAccessDate).toString();
    }

    @Override
    public boolean equals(final Object other) {
        if (!(other instanceof SimpleUser)) {
            return false;
        }
        SimpleUser castOther = (SimpleUser) other;
        return new EqualsBuilder()
            .append(id, castOther.id)
            .append(code, castOther.code)
            .append(name, castOther.name)
            .append(lastAccessDate, castOther.lastAccessDate).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder()
            .append(id)
            .append(code)
            .append(name)
            .append(lastAccessDate).toHashCode();
    }

    /**
     * @require ${this.id} not null.
     * @require ${this.code} not null.
     * @require ${this.name} not null.
     * @require ${this.id} exists.
     * @ensure ${this} f[^x[XƓ
     * @throws UserCodeUsedByOtherException [UR[hdĂ
     * @throws UserNameUsedByOtherException [UdĂ
     */
    @Override
    public void store() throws UserCodeUsedByOtherException,
        UserNameUsedByOtherException {
        validateIdForStore();
        validateCodeForStore();
        validateNameForStore();
        simpleUserDao.update(this);
    }

    private void validateIdForStore() {
        if (id == null) {
            throw new IllegalStateException("The ID must not be null.");
        }
        SimpleUser foundUser = simpleUserDao.find(id);
        if (foundUser == null) {
            throw new IllegalStateException("The user '" + this
                    + "' not found in database.");
        }
    }

    private void validateCodeForStore() throws UserCodeUsedByOtherException {
        if (code == null) {
            throw new IllegalStateException("The code must not be null.");
        }
        SimpleUser foundUser = simpleUserDao.findByCode(code);
        if (foundUser != null && !foundUser.getId().equals(id)) {
            if (logger.isWarnEnabled()) {
                logger.warn("User code '" + code + "' already exists.");
            }
            throw new UserCodeUsedByOtherException(code);
        }
    }

    private void validateNameForStore() throws UserNameUsedByOtherException {
        if (name == null) {
            throw new IllegalStateException("The name must not be null.");
        }
        SimpleUser foundUser = simpleUserDao.findByName(name);
        if (foundUser != null && !foundUser.getId().equals(id)) {
            if (logger.isWarnEnabled()) {
                logger.warn("User name '" + name + "' already exists.");
            }
            throw new UserNameUsedByOtherException(name);
        }
    }

    @Override
    public boolean identify(User other) {
        if (id == null || other == null) {
            return false;
        }
        return id.equals(other.getId());
    }

    /**
     * ł邩ʂ܂BݓƍŏIANZXrAƔfłsԊuȓł
     * ΊƔf܂B
     *
     * @return Ȃ true AłȂ false
     * @see #activeInterSpace
     */
    public boolean isActive() {
        if (lastAccessDate == null) {
            return false;
        }
        Date currentDate = simpleUserDao.getCurrentDate();
        if (logger.isDebugEnabled()) {
            logger.debug("CurrentDate[" + currentDate + "] - LastAccessDate["
                + lastAccessDate + "] = " + (currentDate.getTime()
                - lastAccessDate.getTime()) + " , ActiveInterSpace="
                + activeInterSpace);
        }
        return (currentDate.getTime() - lastAccessDate.getTime())
            < activeInterSpace;
    }

    /**
     * ŏIANZXݓōXV܂B
     *
     * @throws IllegalStateException ${this.dateDao}  null
     */
    public void touch() throws IllegalStateException {
        if (simpleUserDao == null) {
            throw new IllegalStateException("The dateDao is null.");
        }
        Date currentDate = simpleUserDao.getCurrentDate();
        simpleUserDao.updateLastAccessDate(id, currentDate);
    }

    /**
     * vtB[𕜌܂B
     *
     */
    public void retrieveProfile() {
        this.profile = profileTable.find(this.id);
    }

    @Override
    public Integer getId() {
        return id;
    }

    public void setId(Integer id) {
        this.id = id;
    }

    @Override
    public String getCode() {
        return code;
    }

    public void setCode(String code) {
        this.code = code;
    }

    @Override
    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public Date getLastAccessDate() {
        if (lastAccessDate != null) {
            return (Date) lastAccessDate.clone();
        } else {
            return null;
        }
    }

    public void setLastAccessDate(Date lastAccessDate) {
        if (lastAccessDate != null) {
            this.lastAccessDate = (Date) lastAccessDate.clone();
        } else {
            this.lastAccessDate = null;
        }
    }

    public Profile getProfile() {
        return profile;
    }

    public void setProfile(Profile profile) {
        this.profile = profile;
    }

    public void setActiveInterSpace(long activeInterSpace) {
        this.activeInterSpace = activeInterSpace;
    }

    public void setSimpleUserDao(SimpleUserDao simpleUserDao) {
        this.simpleUserDao = simpleUserDao;
    }

    public void setProfileTable(ProfileTable profileTable) {
        this.profileTable = profileTable;
    }
}
