package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.CommunityDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.domain.communication.Community;
import org.unitedfront2.domain.communication.MessageCodeUsedByOtherException;
import org.unitedfront2.domain.communication.MessageTable;
import org.unitedfront2.domain.communication.Thread;

/**
 * {@link CommunityDao}  JDBC łB
 *
 * @author kurokkie
 *
 */
@Repository(value = "communityDao")
public class CommunityDaoImpl extends SimpleDaoSupport<Community>
    implements CommunityDao {

    /** R~jeB̃}bsONX */
    private static class CommunityRowMapper
        implements ParameterizedRowMapper<Community> {

        /** O */
        protected final Log logger = LogFactory.getLog(getClass());

        /** hCt@Ng */
        private DomainFactory domainFactory;

        /** ANZXe[u */
        private AccessControlTable accessControlTable;

        /** bZ[We[u */
        private MessageTable messageTable;

        public CommunityRowMapper(DomainFactory domainFactory,
                AccessControlTable accessControlTable,
                MessageTable messageTable) {
            this.domainFactory = domainFactory;
            this.accessControlTable = accessControlTable;
            this.messageTable = messageTable;
        }

        @Override
        public Community mapRow(ResultSet rs, int rowNum)
            throws SQLException {
            Community c;
            try {
                c = domainFactory.prototype((Class<? extends Community>)
                        Class.forName(rs.getString("Class")));
            } catch (ClassNotFoundException e) {
                throw new IllegalStateException(e);
            }
            c.setId(rs.getInt("Id"));
            c.setCode(rs.getString("Code"));
            c.setOverview(messageTable.get(rs.getInt("OverviewId")));
            c.setOwnerId(rs.getInt("OwnerId"));
            c.setReadAccessControl(
                    accessControlTable.get(rs.getInt("ReadAccessControlId")));
            c.setWriteAccessControl(
                    accessControlTable.get(rs.getInt("WriteAccessControlId")));
            c.setPostAccessControl(
                    accessControlTable.get(rs.getInt("PostAccessControlId")));
            try {
                c.setLastUpdateDate(rs.getTimestamp("LastUpdateDate"));
            } catch (SQLException e) {
                // 
                logger.debug(e.getMessage());
            }
            return c;
        }
    }

    /** ANZXe[u */
    @Autowired
    private AccessControlTable accessControlTable;

    /** bZ[We[u */
    @Autowired
    private MessageTable messageTable;

    /** Xbh}bp[t@Ng */
    @Resource(name = "threadDao")
    private RowMapperFactory<Thread> threadRowMapperFactory;

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<Community> createRowMapper() {
        return new CommunityRowMapper(getDomainFactory(), accessControlTable,
                messageTable);
    }

    @Override
    protected Class<Community> getDomainClass() {
        return Community.class;
    }

    @Override
    protected Map<String, Object> toColumnValueMap(Community community) {
        Map<String, Object> args = super.toColumnValueMap(community);
        args.put("Class", community.getClass().getName());
        args.put("OverviewId", community.getOverview().getId());
        args.put("ReadAccessControlId",
                community.getReadAccessControl().getId());
        args.put("WriteAccessControlId",
                community.getWriteAccessControl().getId());
        args.put("PostAccessControlId",
                community.getPostAccessControl().getId());
        return args;
    }

    //**************************************************************************
    // CommunityDao
    //**************************************************************************
    @Override
    public void register(Community community) {
        storeProperties(community);
        super.register(community);
    }

    @Override
    public void registerThread(int id, int threadId) {
        getSimpleJdbcTemplate().update(
                "INSERT INTO CommunityThread(CommunityId, ThreadId) "
                + "VALUES(?, ?)", id, threadId);
    }

    @Override
    public List<Community> findOrderByLastUpdateDateDesc(int no,
            int num) {
        return getSimpleJdbcTemplate().query(
                "SELECT c.*, MAX(m.RegistrationDate) LastUpdateDate FROM "
                + "CommunityThread ct, Thread t, Entry e, Message m,"
                + "Community c WHERE ct.ThreadId = t.Id AND t.Id = e.ThreadId "
                + "AND e.Id = m.Id AND ct.CommunityId = c.Id "
                + "GROUP BY ct.CommunityId ORDER BY LastUpdateDate DESC "
                + "LIMIT ?, ?",
                createRowMapper(), no, num);
    }

    @Override
    public List<Community> findPublicCommunities() {
        return getSimpleJdbcTemplate().query("SELECT c.* "
                + "FROM Community c, AccessControl a "
                + "WHERE c.ReadAccessControlId = a.Id AND a.Class = '"
                + Public.class.getName() + "' " + "ORDER BY c.Id DESC",
                createRowMapper());
    }

    @Override
    public List<Community> findPublicCommunitiesRandomly(int max) {
        return getSimpleJdbcTemplate().query("SELECT c.* "
                + "FROM Community c, AccessControl a "
                + "WHERE c.ReadAccessControlId = a.Id AND a.Class = '"
                + Public.class.getName() + "' ORDER BY RAND() LIMIT 0, ?",
                createRowMapper(), max);
    }

    @Override
    public Community findByCode(String code) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                    "SELECT * FROM Community WHERE Code = ?", createRowMapper(),
                    code);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
    }

    @Override
    public Thread findThread(int communityId, int threadId) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                    "SELECT * FROM Thread t, CommunityThread ct WHERE "
                    + "ct.ThreadId = ? AND ct.CommunityId = ? AND "
                    + "ct.ThreadId = t.Id",
                    threadRowMapperFactory.createRowMapper(), threadId,
                    communityId);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
    }

    @Override
    public List<Thread> findThreads(int id) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM Thread, CommunityThread WHERE "
                + "CommunityThread.CommunityId = ? AND "
                + "CommunityThread.ThreadId = Thread.Id",
                threadRowMapperFactory.createRowMapper(), id);
    }

    @Override
    public void update(Community community) {
        storeProperties(community);
        super.update(community);
    }

    @Override
    public void registerMyCommunity(int userId, int communityId) {
        getSimpleJdbcTemplate().update(
                "INSERT INTO MyCommunity(UserId, CommunityId) VALUES(?, ?)",
                userId, communityId);
    }

    @Override
    public List<Community> findMyCommunies(int userId) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM Community, MyCommunity WHERE "
                + "MyCommunity.UserId = ? AND "
                + "MyCommunity.CommunityId = Community.Id", createRowMapper(),
                userId);
    }

    @Override
    public List<Integer> findCommunityUserIds(int communityId) {
        return getSimpleJdbcTemplate().query(
                "SELECT UserId FROM MyCommunity WHERE CommunityId = ?",
                new ParameterizedRowMapper<Integer>() {
                    @Override
                    public Integer mapRow(ResultSet rs, int rowNum)
                        throws SQLException {
                        return rs.getInt("UserId");
                    }
                }, communityId);
    }

    @Override
    public void delete(int id) {
        Community c = find(id);
        super.delete(id);
        try {
            c.getOverview().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            c.getReadAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            c.getWriteAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            c.getPostAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
    }

    @Override
    public void deleteMyCommunity(int userId, int communityId) {
        getSimpleJdbcTemplate().update(
                "DELETE FROM MyCommunity WHERE UserId = ? AND CommunityId = ?",
                userId, communityId);
    }

    private void storeProperties(Community community) {
        community.getReadAccessControl().store();
        community.getWriteAccessControl().store();
        community.getPostAccessControl().store();

        community.getOverview().setOwnerId(community.getOwnerId());
        community.getOverview().setAuthorId(community.getOwnerId());
        community.getOverview().setReadAccessControl(
                community.getReadAccessControl());
        community.getOverview().setWriteAccessControl(
                community.getWriteAccessControl());
        try {
            community.getOverview().store();
        } catch (MessageCodeUsedByOtherException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalStateException(e);
        }
    }
}
