package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import org.apache.commons.lang.RandomStringUtils;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcTemplate;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.AccountDao;
import org.unitedfront2.domain.Account;
import org.unitedfront2.domain.Account.Role;
import org.unitedfront2.domain.DomainFactory;

/**
 * AJEg̃f[^ANZXJDBCNXłB
 *
 * @author kurokkie
 *
 */
@Repository(value = "accountDao")
public class AccountDaoImpl extends SimpleDaoSupport<Account>
    implements AccountDao {

    /** ꎞIɔsF؃L[̕ (32) */
    public static final int TEMPORARY_AUTH_KEY_LENGTH = 32;

    /** _R[hdۂɁAĐčă`W (100) */
    public static final int MAX_CHALLENGE_COUNT = 100;

    /** AJEg̃}bsONX */
    private static class AccountRowMapper
        implements ParameterizedRowMapper<Account> {

        /** hCt@Ng */
        private DomainFactory domainFactory;

        /** {@link SimpleJdbcTemplate} */
        private SimpleJdbcTemplate simpleJdbcTemplate;

        public AccountRowMapper(DomainFactory domainFactory,
                SimpleJdbcTemplate simpleJdbcTemplate) {
            super();
            this.domainFactory = domainFactory;
            this.simpleJdbcTemplate = simpleJdbcTemplate;
        }

        public Account mapRow(ResultSet rs, int rowNum) throws SQLException {
            Account account = domainFactory.prototype(Account.class);
            account.setId(rs.getInt("Id"));
            account.setMailAddr(rs.getString("MailAddr"));
            account.setEncryptedPassword(rs.getString("Password"));
            account.setStatus(Account.Status.valueOf(rs.getString("Status")));
            account.addRoles(findRoles(account.getId()));
            return account;
        }

        private Set<Account.Role> findRoles(int accountId) {
            List<Role> list = simpleJdbcTemplate.query(
                "SELECT * FROM AccountRole WHERE AccountId = ?",
                new ParameterizedRowMapper<Account.Role>() {
                    public Account.Role mapRow(ResultSet rs, int rowNum)
                        throws SQLException {

                        return Account.Role.valueOf(rs.getString("Role"));
                    }
                }, accountId);
            Set<Account.Role> roleSet = new HashSet<Account.Role>(list.size());
            for (Role role : list) {
                roleSet.add(role);
            }
            return roleSet;
        }
    }

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<Account> createRowMapper() {
        return new AccountRowMapper(getDomainFactory(),
                getSimpleJdbcTemplate());
    }

    @Override
    protected Class<Account> getDomainClass() {
        return Account.class;
    }

    //**************************************************************************
    // AccountDao
    //**************************************************************************
    @Override
    public void register(final Account account) {
        super.register(account);
        registerRoles(account);
    }

    @Override
    public String generateTemporaryAuthKey(int accountId) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                "SELECT AuthKey FROM TemporaryAuthKey WHERE AccountId = ?",
                String.class, accountId);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
        }
        String key = generateTemporaryAuthKey();
        getJdbcTemplate().update(
            "REPLACE INTO TemporaryAuthKey(AccountId, AuthKey, GeneratedDate) "
            + "VALUES(?, ?, NOW())", new Object[] {accountId, key});
        return key;
    }

    private String generateTemporaryAuthKey() {
        for (int i = 0; i < MAX_CHALLENGE_COUNT; i++) {
            String key = RandomStringUtils.randomAlphanumeric(
                    TEMPORARY_AUTH_KEY_LENGTH).toLowerCase(Locale.ENGLISH);
            Account account = findByTemporaryAuthKey(key);
            if (account != null) {
                logger.warn("The generated temporary auth key '" + key
                    + "' already exists.");
                continue;
            }
            return key;
        }
        String errorMessage = "Faled to generate temporary auth key.";
        logger.error(errorMessage);
        throw new IllegalStateException(errorMessage);
    }

    @Override
    public Account findByMailAddr(String mailAddr) {
        Account account;
        try {
            account = getSimpleJdbcTemplate().queryForObject(
                "SELECT * FROM Account WHERE MailAddr = ?",
                createRowMapper(), mailAddr);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
        account.setAccountDao(this);
        return account;
    }

    @Override
    public Account findByTemporaryAuthKey(String temporaryAuthKey) {
        try {
            Account account = getSimpleJdbcTemplate().queryForObject(
                "SELECT * FROM Account, TemporaryAuthKey "
                + "WHERE TemporaryAuthKey.AuthKey = ? AND "
                + "Account.Id = TemporaryAuthKey.AccountId",
                createRowMapper(), temporaryAuthKey);
            account.setTemporaryAuthKey(temporaryAuthKey);
            return account;
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
    }

    @Override
    public void update(Account account) {
        super.update(account);
        deleteRoles(account.getId());
        registerRoles(account);
    }

    @Override
    public void deleteTemporaryAuthKey(int accountId) {
        getJdbcTemplate().update(
            "DELETE FROM TemporaryAuthKey WHERE AccountId = ?",
            new Object[] {accountId});
    }

    @Override
    public void deleteOldTemporaryAuthKey(int seconds) {
        getSimpleJdbcTemplate().update("DELETE FROM TemporaryAuthKey WHERE "
            + "UNIX_TIMESTAMP(NOW()) - UNIX_TIMESTAMP(GeneratedDate) >= ?",
            seconds);
    }

    private void registerRoles(Account account) {
        StringBuffer sb = new StringBuffer("INSERT INTO AccountRole(AccountId, "
                + "Role) VALUES");
        for (int i = 0; i < account.getRoles().size(); i++) {
            if (i != 0) {
                sb.append(", ");
            }
            sb.append("(?, ?)");
        }
        String sql = sb.toString();
        List<Object> args = new ArrayList<Object>();
        for (Role role : account.getRoles()) {
            args.add(account.getId());
            args.add(role.toString());
        }
        getJdbcTemplate().update(sql, args.toArray());
    }

    private void deleteRoles(int accountId) {
        getJdbcTemplate().update("DELETE FROM AccountRole WHERE AccountId = ?",
            new Object[] {accountId});
    }
}
