package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcInsert;
import org.unitedfront2.dao.Registerable;
import org.unitedfront2.dao.SimpleDeletable;
import org.unitedfront2.dao.SimpleFindable;
import org.unitedfront2.dao.Updatable;
import org.unitedfront2.domain.Domain;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.util.PropertyUtils;
import org.unitedfront2.util.UnsupportedBeanOperationException;

/**
 * JDBC p̒ۓI DAO łBЂƂ̎L[e[uɑ΂f[^ANZXɂẮA
 * ̒ۃNXp邱ƂŁA{Iȃ\bhėpł܂BɁAe[ũJƃhCIuWF
 * Ng̃vpeBvꍇɂ͂قƂǃJX^}CYKv܂BvȂꍇłĂA
 * <code>protected</code> \bhI[o[Ch邱ƂŃJX^}CYł܂B<p>
 *
 * L[ȂꍇATuNXł {@link #createRowMapper()} I[o[ChA
 * {@link #register(Object)} ̑ {@link #registerNoReturnKey(Domain)}
 * 𗘗p܂B<p>
 *
 * o^эXV̑ΏۂƂȂ񖼂JX^}CYꍇ́A{@link #toColumnValueMap(Domain)}
 * I[o[Ch܂B
 *
 * @author kurokkie
 *
 * @param <D> hCNX
 */
abstract class SimpleDaoSupport<D extends Domain> extends DaoSupport
    implements Registerable<D>, SimpleFindable<D>, Updatable<D>,
    SimpleDeletable, RowMapperFactory<D> {

    /** vC}[L[̃ftHgl (Id) */
    public static final String DEFAULT_PRIMARY_KEY_COLUMN_NAME = "Id";

    /** hCNX */
    @Autowired
    private DomainFactory domainFactory;

    /** 񖼃Xg */
    private List<String> columnNames;

    @Override
    public void afterPropertiesSet() throws Exception {
        super.afterPropertiesSet();
        columnNames = getSimpleJdbcTemplate().query(
                "SHOW COLUMNS FROM " + getTableName(),
                new ParameterizedRowMapper<String>() {
                    @Override
                    public String mapRow(ResultSet rs, int rowNum)
                        throws SQLException {
                        return rs.getString(1);
                    }
                });
    }

    /**
     * e[u {@link #getTableName()} AL[
     * {@link #getPrimaryKeyColumnName()} 擾Ă܂B
     *
     * @see #getTableName()
     * @see #createRowMapper()
     * @see #getPrimaryKeyColumnName()
     */
    @Override
    public D find(int id) {
        D domainObject;
        try {
            domainObject = getSimpleJdbcTemplate().queryForObject(
                "SELECT * FROM " + getTableName() + " WHERE "
                + getPrimaryKeyColumnName() + " = ?", createRowMapper(), id);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
        return domainObject;
    }

    /**
     * ̃\bh́AL[邱ƂOƂĂ܂BL[ȂꍇÃ\bh
     * I[o[ChA{@link #registerNoReturnKey(Domain)} 𗘗pĂB
     *
     * @see #createSimpleJdbcInsert(DataSource)
     * @see #getPrimaryKeyColumnName()
     * @see #toColumnValueMap(Domain)
     */
    @Override
    public void register(final D domain) {
        Map<String, Object> args = toColumnValueMap(domain);
        args.remove(getPrimaryKeyColumnName());
        int id = getSimpleJdbcInsert().executeAndReturnKey(args).intValue();
        String keyPropertyName = toPropertyName(getPrimaryKeyColumnName());
        PropertyUtils.setProperty(domain, keyPropertyName, id);
    }

    /**
     * SȃhCIuWFNgƂēnAVK̃f[^Ƃēo^܂B̃\bh͎L[
     * ܂B̃\bh𗘗pꍇ͍킹 {@link #createRowMapper()} I[o[C
     * hKv܂B
     *
     * @see #createSimpleJdbcInsert(DataSource)
     * @see #toColumnValueMap(Domain)
     */
    protected void registerNoReturnKey(final D domain) {
        Map<String, Object> args = toColumnValueMap(domain);
        getSimpleJdbcInsert().execute(args);
    }

    /**
     * ̎́AhCIuWFNg̎L[sAhCIuWFNgێvpeB̒lōXV
     * ܂BhCIuWFNg̃vpeB SQL ^ŕ\łȂ^łꍇA̎͐ɓ
     * 삵܂BvpeB {@link Enum} ^łꍇAIɕ^ƂĈ܂B
     *
     * @see #toColumnValueMap(Domain)
     * @see #getPrimaryKeyColumnName()
     * @see #getTableName()
     */
    @Override
    public void update(D domain) {

        Map<String, Object> beanMap = toColumnValueMap(domain);

        String primaryKeyPropertyName
            = toPropertyName(getPrimaryKeyColumnName());
        beanMap.put(primaryKeyPropertyName, PropertyUtils.getProperty(domain,
                primaryKeyPropertyName));

        // SQL 쐬
        StringBuffer sb = new StringBuffer(29);
        sb.append("UPDATE ").append(getTableName()).append(" SET ");
        int i = 0;
        for (String key : beanMap.keySet()) {
            if (key.equals(primaryKeyPropertyName)) {
                continue;
            }
            if (i != 0) {
                sb.append(", ");
            }
            sb.append(toColumnName(key)).append(" = :").append(key);
            i++;
        }
        sb.append(" WHERE ").append(getPrimaryKeyColumnName()).append(" = :")
            .append(primaryKeyPropertyName);
        String sql = sb.toString();

        getNamedParameterJdbcTemplate().update(sql, beanMap);
    }

    /**
     * @see #getTableName()
     * @see #getPrimaryKeyColumnName()
     */
    @Override
    public void delete(int id) {
        getSimpleJdbcTemplate().update("DELETE FROM " + getTableName()
            + " WHERE " + getPrimaryKeyColumnName() + " = ?", id);
    }

    /**
     * hCIuWFNg񖼂ƒl̃}bvɕϊ܂B\bh {@link #register(Object)},
     * {@link #registerNoReturnKey(Domain)}, {@link #update(Object)} ŎgĂ
     * B<p>
     *
     * ̎ł́Ae[uƃhCIuWFNg̗ɋʂf[^ڂݒ肵܂BhCIuWFNg
     * ̃vpeB SQL ^ŕ\łȂ^łꍇA̎͐ɓ삵܂BvpeB
     * {@link Enum} ^łꍇAIɕƂĈ܂B̃\bh͕KvɉăI[o[
     * ChĂB<p>
     *
     * @param domain hCIuWFNg
     * @return 񖼂ƒl̃}bv
     */
    protected Map<String, Object> toColumnValueMap(D domain) {
        Map<String, Object> args
            = new HashMap<String, Object>(columnNames.size());
        for (String columnName : columnNames) {
            Object value;
            try {
                value = PropertyUtils.getProperty(domain,
                        toPropertyName(columnName));
            } catch (UnsupportedBeanOperationException e) {
                if (logger.isInfoEnabled()) {
                    logger.info(e.getMessage());
                }
                continue;
            }
            if (value instanceof Enum) {
                value = value.toString();
            }
            args.put(columnName, value);
        }
        return args;
    }

    /**
     * RDBMS ̃e[uԂ܂B̎ł́Af[^ANZXhCNX̃pbP[W
     * NXԂ܂BKvɉăTuNXŃI[o[ChĂBł̓hCN
     * Xgăe[u쐬Ă܂BhCNXݒ肵ȂꍇÃ\bhI[o[Ch
     * 邱Ƃ͕K{ɂȂ܂B
     *
     * @return RDBMS ̃e[u
     */
    protected String getTableName() {
        if (domainFactory == null) {
            String message = "The domain factory is null. You have to override "
                + "'getTableName()' method.";
            logger.error(message);
            throw new IllegalStateException(message);
        }
        return getDomainClass().getSimpleName();
    }

    /**
     * L[ƂȂJ擾܂B̎
     * {@link #DEFAULT_PRIMARY_KEY_COLUMN_NAME} Ԃ܂BKvɉăI[o[ChĂ
     * B
     *
     * @return L[ƂȂJ
     */
    protected String getPrimaryKeyColumnName() {
        return DEFAULT_PRIMARY_KEY_COLUMN_NAME;
    }

    /**
     * {@link #getTableName()} găe[uA
     * {@link #getPrimaryKeyColumnName()} gĎL[ݒ肵܂B
     *
     * @param dataSource {@link DataSource}
     * @return {@link SimpleJdbcInsert}
     * @see DaoSupport#createSimpleJdbcInsert(DataSource)
     */
    @Override
    protected SimpleJdbcInsert createSimpleJdbcInsert(DataSource dataSource) {
        return new SimpleJdbcInsert(dataSource).withTableName(getTableName())
            .usingGeneratedKeyColumns(getPrimaryKeyColumnName());
    }

    /**
     * }bp[IuWFNg𐶐܂B<p>
     *
     * ̎ł́AJ̒lA{@link ResultSet#getObject(String)} găhCIuWF
     * Ng̑ΉvpeBɐݒ肵Ă܂BΉvpeBȂꍇ͉܂B
     * {@link Enum} ^̃vpeBɑ΂ẮA {@link Enum} ^ɕϊĐݒ肵܂Bh
     * CIuWFNg̐ɂ̓hCt@NggpĂ܂B̃\bh͕KvɉăTuNXŃI[
     * o[ChĂB
     *
     * @return }bp[IuWFNg
     */
    @Override
    public ParameterizedRowMapper<D> createRowMapper() {
        return new ParameterizedRowMapper<D>() {

            @Override
            public D mapRow(ResultSet rs, int rowNum) throws SQLException {
                D domainObject = domainFactory.prototype(getDomainClass());
                for (String columnName : columnNames) {
                    Object value = rs.getObject(columnName);
                    String propertyName = toPropertyName(columnName);
                    if (PropertyUtils.isEnum(domainObject, propertyName)) {
                        try {
                            PropertyUtils.setProperty(domainObject,
                                    propertyName,
                                Enum.valueOf(PropertyUtils.getType(domainObject,
                                    propertyName), value.toString()));
                        } catch (UnsupportedBeanOperationException e) {
                            if (logger.isInfoEnabled()) {
                                logger.info(e.getMessage());
                            }
                            continue;
                        }
                    } else {
                        PropertyUtils.setProperty(domainObject, propertyName,
                            value);
                    }
                }
                return domainObject;
            }
        };
    }

    protected abstract Class<D> getDomainClass();

    private String toColumnName(String propertyName) {
        StringBuffer sb = new StringBuffer(
                String.valueOf(propertyName.charAt(0)).toUpperCase(
                        Locale.ENGLISH));
        if (propertyName.length() > 1) {
            sb.append(propertyName.substring(1));
        }
        return sb.toString();
    }

    private String toPropertyName(String columnName) {
        StringBuffer sb = new StringBuffer(
                String.valueOf(columnName.charAt(0)).toLowerCase(
                        Locale.ENGLISH));
        if (columnName.length() > 1) {
            sb.append(columnName.substring(1));
        }
        return sb.toString();
    }

    protected DomainFactory getDomainFactory() {
        return domainFactory;
    }
}
