/*!
 * @file lsock.c
 * @bref listen socket control.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <glib.h>
#include <errno.h>
#include "logger_wrapper.h"
#include "l7vs.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_lsock.h"
#include "l7vs_module.h"
#include "l7vs_conn.h"
#include "l7vs_service.h"
#include "l7vs_sched.h"

/* static functions */
static int  l7vs_lsock_accept(struct l7vs_lsock *lsock);
static void l7vs_lsock_table_add(struct l7vs_lsock *lsock);
static void l7vs_lsock_table_remove(struct l7vs_lsock *lsock);
static gint l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b);
static int  l7vs_lsock_callback(struct l7vs_iomux *iom );

//! l7vs_lsock list 
static GList *l7vs_lsock_list;

/*!
 * inner function of fini.
 * all element free.
 * @param[in]   void
 * @return      void
 */
void freeAllList( gpointer data, gpointer userdata ) {
    struct l7vs_lsock* lsock = (struct l7vs_lsock*)data;

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,2,
            "lsock in_fuction freeAllList : lsock=%p", lsock);
    }
    free(lsock);
}

/*!
 * initialize functions
 * @param[in]   void
 * @return      everydays zero.
 */
int
l7vs_lsock_init(void)
{       
    if (l7vs_lsock_list == NULL) {       
        if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,3,
                "lsock in_fuction int l7vs_lsock_init(void) : l7vs_lsock_list == NULL return 0");
        }
        return 0;
    }
    g_list_foreach( l7vs_lsock_list, freeAllList, NULL );
    g_list_free( l7vs_lsock_list );
    l7vs_lsock_list = NULL;

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,4,
            "lsock in_fuction int l7vs_lsock_init(void) : l7vs_lsock_list == NULL return 0");
    }
    return 0;
}

/*!
 * finalize function.
 *  free list element memory.
 *  drop list memory.
 * @param[in]   void
 * @return      void
 */
void
l7vs_lsock_fini(void)
{
    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,5,
            "lsock in_fuction int l7vs_lsock_fini(void) : free l7vs_lsock_list = %p",
            l7vs_lsock_list);
    }

    g_list_foreach( l7vs_lsock_list, freeAllList, NULL );
    g_list_free( l7vs_lsock_list );
    l7vs_lsock_list = NULL;
}

/*!
 * look up socket
 * if target soket have link list then return having socket.
 * else create new socket to return.
 * @param[in]   sin     socket address struct( if ipv6 may be change this struct ) 
 * @param[in]   proto   TCS/UDP select.
 * @param[in]   backlog
 */
struct l7vs_lsock *
l7vs_lsock_get(struct sockaddr_in *sin, uint8_t proto, int backlog)
{
    struct l7vs_lsock *lsock;
    int stype;
    int ret;
    int on = 1;

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,20,
            "in_fuction:  struct l7vs_lsock * l7vs_lsock_get(struct sockaddr_in *sin, uint8_t proto, int backlog)"
            "*sin =%p :proto = %d :backlog = %d :",
            sin,proto,backlog);
    }

    if (sin == NULL) {
        return NULL;
    }

    lsock = l7vs_lsock_table_lookup(sin, proto);
    if (lsock != NULL) {
        lsock->refcnt++;
        if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
            char lsock_str[DEBUG_STR_LEN] = {0};
            l7vs_lsock_c_str(lsock_str, lsock);
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,21,
                "table look up: lsock=%s", lsock_str);
        }
        return lsock;
    }

    switch (proto) {
    case IPPROTO_TCP:
        stype = SOCK_STREAM;
        break;
    case IPPROTO_UDP:
        stype = SOCK_DGRAM;
        break;
    default:
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,21,
            "Protocol number should be TCP or UDP (%d)", proto);
        return NULL;
    }

    lsock = (struct l7vs_lsock *)calloc(1, sizeof(*lsock));
    if (lsock == NULL) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_MEMORY,14, "Could not allocate lsock");
        return lsock;
    }
    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,6,
            "in function l7vs_lsock_get: allocate memory"
            " : size=%zu ,pointer=%p", sizeof(*lsock),lsock);
    }

    lsock->iom = l7vs_iomux_get_from_avail_list();
    if (!lsock->iom) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,24, "can not get lsock_iomux");
        return NULL;
    }

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,22, "creating lsock %p", lsock);
    }
    lsock->proto = proto;
    lsock->iom->fd = socket(PF_INET, stype, proto);  //create a socket to get connection request from the client
    if (lsock->iom->fd < 0) {
        if (proto == IPPROTO_TCP) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,22,
                "socket: Protocol is IPPROTO_TCP,SockType is SOCK_STREAM (%s)", strerror(errno));
        }
        else {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,26,
                "socket: Protocol is IPPROTO_UDP,SockType is SOCK_DGRAM (%s)", strerror(errno));
        }
 
        if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,7,
                "in function l7vs_lsock_get: free memory"
                " : size=%zu ,pointer=%p", sizeof(*lsock),lsock);
        }
        free(lsock);
        return NULL;
    }

    setsockopt(lsock->iom->fd, SOL_SOCKET, SO_REUSEADDR, (char*) &on, sizeof(int));
    ret = bind(lsock->iom->fd, (struct sockaddr *)sin, sizeof(*sin)); //binding the socket for incoming client request
    if (ret < 0) {
        char addr_str[DEBUG_STR_LEN]={0};

        l7vs_lsock_sockaddr_in_c_str(addr_str, (struct sockaddr_in *)sin);
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,23, "Could not bind socket: %s (%s)",
            addr_str,strerror(errno));
        close(lsock->iom->fd);
        if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,8,
                "in function l7vs_lsock_get: free memory"
                " : size=%zu ,pointer=%p", sizeof(*lsock),lsock);
        }
        free(lsock);
        return NULL;
    }
    lsock->addr = *sin;

    ret = listen(lsock->iom->fd, backlog); //listening for client requests
    if (ret < 0) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,24, "Could not listen: %s (file descriptor :%d , backlog: %d)",
            strerror(errno),lsock->iom->fd, backlog);
        close(lsock->iom->fd);
        if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,9,
                "in function l7vs_lsock_get: free memory"
                " : size=%zu ,pointer=%p", sizeof(*lsock),lsock);
        }
        free(lsock);
        return NULL;
    }

    lsock->refcnt = 1;
    lsock->fast_schedule = 1;

    lsock->iom->callback = l7vs_lsock_callback;
    lsock->iom->data = lsock;
    lsock->iom->status = iomux_lsock_connect_waiting;
    l7vs_lsock_table_add(lsock);  //Add socket in the list. It may be used for maintaining session (Am not sure)
//    l7vs_iomux_add(&lsock->iom, L7VS_IOMUX_READ);

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,23,
            "creat: lsock=%s", lsock_str);
    }
    l7vs_iomux_add( lsock->iom, iom_read );
    return lsock;
}

/*!
 * lisning socket remove list and iomuxlist.
 * @param[in]   lsock   removing lisning socket
 * @return      void
 */

void
l7vs_lsock_put(struct l7vs_lsock *lsock)
{
    if (lsock == NULL) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,25, "error / lsock is null");
        return;
    }

    lsock->iom->status = iomux_lsock_released;

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,24,
            "lsock  removing : lsock=%s", lsock_str);
    }
    if (--lsock->refcnt > 0) {
        lsock->iom->status = iomux_lsock_connect_waiting;
        if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
            char lsock_str[DEBUG_STR_LEN] = {0};
            l7vs_lsock_c_str(lsock_str, lsock);
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,25,
                "not removing : lsock=%s", lsock_str);
        }
        return;
    }

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,26,
            "removing : lsock=%s", lsock_str);
    }

    lsock->iom->status = iomux_lsock_destroyed;

    l7vs_iomux_remove(lsock->iom);
    l7vs_lsock_table_remove(lsock);

    if (lsock->iom->fd > 0) {
        close(lsock->iom->fd);
    }
    l7vs_iomux_put_to_avail_list(lsock->iom);

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_MEMORY) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_MEMORY,10,
            "in function l7vs_lsock_put  : free lsock=%p", lsock);
    }
    free(lsock);
}

/*!
 * lsock_list append new lsock ponter
 * @param[in]   lsock   to insert table lsock
 * @return      void
 */
static void
l7vs_lsock_table_add(struct l7vs_lsock *lsock)
{
    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,27,
            "in_function l7vs_lsock_table_add : add table lsock=%s",
            lsock_str);
    }
    l7vs_lsock_list = g_list_append(l7vs_lsock_list, (gpointer)lsock);
    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,28,
            "out_function l7vs_lsock_table_add : add table lsock=%s",
            lsock_str);
    }
}

/*!
 * lsock_list remove lsock (don't free element)
 * @param[in]   lsock   to remove lsock
 * @return      void
 */
static void
l7vs_lsock_table_remove(struct l7vs_lsock *lsock)
{
    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,29,
            "in_function l7vs_lsock_table_remove : remove table lsock=%s",
            lsock_str);
    }
    l7vs_lsock_list = g_list_remove(l7vs_lsock_list, (gpointer)lsock);
    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,30,
            "out_function l7vs_lsock_table_remove : remove table lsock=%s",
            lsock_str);
    }
}


/*!
 * look up table for lsock.
 * @param[in]   *sin    socketaddr_in struct pointer (ex. to change then IPv6 target )
 * @param[in]   *proto  port no
 * @return      *l7vs_lsock
 */
struct l7vs_lsock *
l7vs_lsock_table_lookup(struct sockaddr_in *sin, uint8_t proto)
{
    struct l7vs_lsock tmpl;
    GList *l;

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,31,
            "in_fuction:  struct l7vs_lsock * l7vs_lsock_table_lookup(struct sockaddr_in *sin, uint8_t proto)"
            "*sin =%p :proto = %d ", sin,proto);
    }

    if (sin == NULL) {
        return NULL;
    }

    tmpl.addr = *sin;
    tmpl.proto = proto;

    l = g_list_find_custom(l7vs_lsock_list, (gpointer)&tmpl,
        l7vs_lsock_addr_cmp);
    if (l == NULL)
        return NULL;
    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, (struct l7vs_lsock *)l->data);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,32,
            "return : lsock=%s", lsock_str);
    }
    return (struct l7vs_lsock *)l->data;  //for checking up and finding the socket in the list for a particular client socket(not sure)
}

/*!
 * using foreach function from Compare
 * @param   a l7vs_lsock pointer
 * @param   b l7vs_lsock pointer
 * @return  compare pattern 
 */
static gint
l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b)
{
    struct l7vs_lsock *la = (struct l7vs_lsock *)a;
    struct l7vs_lsock *lb = (struct l7vs_lsock *)b;


    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,33,
            "in_fuction:  gint l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b)"
            "a =%p :b = %p ", a, b);
    }

    if (la->addr.sin_addr.s_addr != lb->addr.sin_addr.s_addr)
        return la->addr.sin_addr.s_addr - lb->addr.sin_addr.s_addr;
    if (la->addr.sin_port != lb->addr.sin_port)
        return la->addr.sin_port - lb->addr.sin_port;
    if (la->proto != lb->proto)
        return la->proto - lb->proto;

    if (logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,34,
            "out_fuction:  gint l7vs_lsock_addr_cmp(gconstpointer a, gconstpointer b)"
            "return = 0");
    }
    return 0;
}

/*!
 * select VertualService or SorryServer
 * @param   lsock       l7vs_lsock pointer
 * @param   conn        l7vs_conn pointer
 * @param   buf         char pointer
 * @param   len         size_t
 * @param   srv_ret     l7vs_service pointer pointer
 * @param   dest_ret    l7vs_dest pointer pointer
 * @return  int         ok:1 error:-1
 */
int
l7vs_lsock_select_service(struct l7vs_lsock *lsock, 
                          struct l7vs_conn *conn,
                          char *buf,
                          size_t len,
                          struct l7vs_service **srv_ret,
                          struct l7vs_dest **dest_ret)
{
    GList *l;
    struct l7vs_service *srv;
    struct l7vs_dest *dest;
    int ret, val;
    struct l7vs_service *srv_tmp = NULL;    //! temporary for srv
    struct l7vs_dest *dest_tmp   = NULL;    //! temporary for dest
    size_t len_tmp = 0;                     //! temporary for len
    int    sorry_save = 0;                  //! sorry dest save flag
    struct l7vs_service *srv_tmp2 = NULL;   //! temporary for srv when select_dest NG
    struct l7vs_dest *dest_tmp2   = NULL;   //! temporary for dest when select_dest NG
    size_t len_tmp2 = 0;                    //! temporary for len when select_dest NG
    int    sorry_save2 = 0;                 //! sorry dest save flag when select_dest NG

    if (logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,13,
            "in_fuction:  int l7vs_lsock_select_service(struct l7vs_lsock *lsock,"
            "struct l7vs_conn *conn,"
            "char *buf,"
            "size_t len,"
            "struct l7vs_service **srv_ret,"
            "struct l7vs_dest **dest_ret) "
            "lsock :%p, conn :%p buf : %p,len : %zu,",
            lsock,conn,buf,len);
    }
    val = -1;
    l = lsock->srv_list;
    while (l != NULL) {
        srv = (struct l7vs_service *)l->data;
        dest = NULL;
        ret = srv->pm->select_dest(srv, conn, buf, &len, &dest);
        if (ret == 0) {
            // check sorry status if ret == 0
            // sorry_check argument 1 means check connection count
            // Check the connection count(--upper, maxconn)
            if (l7vs_sched_sorry_check(srv, 1)) {
                if (!sorry_save) {
                    // save sorry dest to xxx_tmp only first time
                    srv_tmp = srv;
                    dest_tmp = dest;
                    len_tmp = len;
                    sorry_save = 1;
                }
            } else {
                // check cldata length if sorry_check != 1 (not sorry status)
                if (len > conn->cldata_len + L7VS_PROTOMOD_MAX_ADD_BUFSIZE) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SORRY_SERVER,1, "bufsize too long modified by protomod ");
                } else {
                    // service and destination is decided
                    *srv_ret = srv;
                    *dest_ret = dest;
                    conn->cldata_len = len;
                    sorry_save = 0;
                    sorry_save2 = 0;
                    val = 1;
                    break;
                }
            }
        }
        // when select_dest NG
        // No suitable realservers exist
        if (!sorry_save2) {
            // save sorry dest to xxx_tmp2 only first time
            srv_tmp2 = srv;
            dest_tmp2 = dest;
            len_tmp2 = len;
            sorry_save2 = 1;
        }
        // get next srv from srv_list
        l = g_list_next(l);
    }

    // Over maximum connections
    if (sorry_save) {
        // saved sorry dest is exist
        // set sorry-server destination
        *dest_ret = (struct l7vs_dest *)l7vs_sched_sorry_dest(srv_tmp, NULL, 0);
        if (!*dest_ret) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SORRY_SERVER,2, "sorry-server dest is NULL");
            if (logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,14,
                    "in_fuction:  int l7vs_lsock_select_service return -1");
            }
            return -1;
        }
        *srv_ret = srv_tmp;
        conn->old_dest = dest_tmp;
        conn->cldata_len = len_tmp;
        conn->sorry_conn_flag = 1;
        val = 1;

    // No suitable realservers exist
    } else if (sorry_save2) {
        // saved sorry dest2 is exist
        // set sorry-server destination
        *dest_ret = (struct l7vs_dest *)l7vs_sched_sorry_dest(srv_tmp2, NULL, 0);
        if (!*dest_ret) {
            LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SORRY_SERVER,3, "sorry-server dest is NULL");
            if (logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG) {
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,15,
                    "in_fuction:  int l7vs_lsock_select_service return -1");
            }
            return -1;
        }
        *srv_ret = srv_tmp2;
        conn->old_dest = dest_tmp2;
        conn->cldata_len = len_tmp2;
        conn->sorry_conn_flag = 1;
        val = 1;
    }
    // return val=1 service and destination is decided (to real-server or sorry-server)
    // return val=-1 service and destination not decided (select_dest result all NG)
    if (logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,16,
            "out_fuction:  int l7vs_lsock_select_service return %d",val);
    }
    return val;
}

/*!
 * table add service list
 *  append list memory.
 * @param[in]   void
 * @return      void
 */
void
l7vs_lsock_add_service(struct l7vs_lsock *lsock,
                       struct l7vs_service *srv)
{
    if (logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);

        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,94,
            "in_fuction:  void l7vs_lsock_add_service(struct l7vs_lsock *lsock,struct l7vs_service *srv)"
            "lsock: %s,srv : %p",
            lsock_str,srv);
    }
    lsock->srv_list = g_list_append(lsock->srv_list, srv);
    if (lsock->fast_schedule) {
        lsock->fast_schedule = srv->pm->fast_schedule;
    }
    if (logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);

        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,95,
            "out_fuction:  void l7vs_lsock_add_service(struct l7vs_lsock *lsock,struct l7vs_service *srv)"
            "lsock: %s,srv : %p", lsock_str,srv);
    }
}

/*!
 * table remove from service list
 *  remove list memory.
 * @param[in]   void
 * @return      void
 */
void
l7vs_lsock_remove_service(struct l7vs_lsock *lsock,
                          struct l7vs_service *srv)
{
    GList* slist = NULL;
    if (logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);

        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,96,
            "in_fuction:  void l7vs_lsock_remove_service (struct l7vs_lsock *lsock,struct l7vs_service *srv)"
            "lsock: %s,srv : %p",
            lsock_str,srv);
    }
    lsock->srv_list = g_list_remove(lsock->srv_list, srv);
    lsock->fast_schedule = 1;
    for (slist = g_list_first(lsock->srv_list); slist != NULL; slist = slist->next) {
        struct l7vs_service* s = (struct l7vs_service*) slist->data;
        if (s != NULL && s->pm != NULL && s->pm->fast_schedule == 0) {
            lsock->fast_schedule = 0;
            break;
        }
    }
    if (logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);

        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,97,
            "out_fuction:  void l7vs_lsock_remove_service (struct l7vs_lsock *lsock,struct l7vs_service *srv)"
            "lsock: %s,srv : %p",
            lsock_str,srv);
    }
}

/*!
 * socket acception function.
 * @param[in]   lsock   socket option
 * @return      success = 0 / false = -1
 */
static int
l7vs_lsock_accept(struct l7vs_lsock *lsock)
{
    if (lsock == NULL) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,20, "error / lsock is null");
        if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,50,
                "out_fuction:  int l7vs_lsock_accept return -1");
        }
        return -1;
    }
    if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
        char lsock_str[DEBUG_STR_LEN] = {0};
        l7vs_lsock_c_str(lsock_str, lsock);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,51,
            "in_fuction:  int l7vs_lsock_accept (struct l7vs_lsock *lsock)"
            "lsock: %s", lsock_str);
    }
    if (iomux_lsock_accepted != lsock->iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,21, "error / invalid status(%d)", lsock->iom->status);
        if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,52,
                "out_fuction:  int l7vs_lsock_accept return -1");
        }
        return -1;
    }

    if (!l7vs_conn_create(lsock->iom->fd, lsock)) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,22, "error / conn create failed");
        lsock->iom->status = iomux_lsock_conn_create_error;
        if (logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,8,
                "out_fuction:  int l7vs_lsock_accept return -1");
        }
        return -1;
    }

    lsock->iom->status = iomux_lsock_conn_created;
    if (logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,9,
            "out_fuction:  int l7vs_lsock_accept return 0");
    }
    return 0;
}

/*!
 * ListeningSocket call back function.
 * @param[in]   iom socket option
 * @return      success = 0 / false = -1
 */
static int
l7vs_lsock_callback(struct l7vs_iomux *iom )
{
    int ret;

    if (iom == NULL) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,23, "error / iom is null");
        if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,53,
                "out_fuction:  int l7vs_lsock_callback return -1");
        }
        return -1;
    }

    if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
        char iomux_str[DEBUG_STR_LEN] = {0};
        l7vs_iomux_c_str(iomux_str,iom);
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,54,
            "in_fuction:  int l7vs_lsock_callback (struct l7vs_iomux *iom)"
            "iom: %s", iomux_str);
    }

    if (iomux_lsock_connect_waiting != iom->status) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,24, "error / invalid status(%d)", iom->status);
        if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,55,
                "out_fuction:  int l7vs_lsock_callback return -1");
        }
        return -1;
    }

    iom->status = iomux_lsock_accepted;
    ret = l7vs_lsock_accept((struct l7vs_lsock *)iom->data); //for accepting data from clients
    if (-1 == ret) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,25, "error / lsock accept failed");
        iom->status = iomux_lsock_connect_waiting;
        l7vs_iomux_mod( iom, iom_read );
        if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
            LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,56,
                "out_fuction:  int l7vs_lsock_callback return -1");
        }
        return -1;
    }

    iom->status = iomux_lsock_connect_waiting;
    l7vs_iomux_mod( iom, iom_read );
    if (logger_get_log_level(LOG_CAT_L7VSD_EVENT) == LOG_LV_DEBUG) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,57,
            "out_fuction:  int l7vs_lsock_callback return 0");
    }

    return 0; 
}
