/*
 * @file        sched_rr.c
 * @brief       Round-Robin Scheduling Module for UltraMonkey-L7
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <string.h>
#include "l7vs_dest.h"
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_sched.h"

#define	IS_SCHEDRR_DEBUG if( LOG_LV_DEBUG == IS_DEBUG<struct l7vs_scheduler>( sched_rr_scheduler, LOG_CAT_L7VSD_SCHEDULE ))
#define	SCHED_DEBUG(X,Y...) PUT_LOG_DEBUG( sched_rr_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define SCHED_INFO(X,Y...)  PUT_LOG_INFO(  sched_rr_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define	SCHED_WARN(X,Y...)  PUT_LOG_WARN(  sched_rr_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define	SCHED_ERROR(X,Y...) PUT_LOG_ERROR( sched_rr_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )
#define SCHED_FATAL(X,Y...) PUT_LOG_FATAL( sched_rr_scheduler, LOG_CAT_L7VSD_SCHEDULE, X, ##Y )

/*!
 *	servicedest contenor struct
 */
struct servicedest_contenor{
	handle_t	handle; //! service handle
	struct in_addr	addr;	//! dest address
	u_short		port;	//! dest port
};

/*!	service dest list */
GList*	servicedest_list;

/*!
 *	schedule module finalize fnction
 *	@param void
 *	@return void
 */
static void fini(void);

/*!
 *	schedule module scheduler function
 *	@param[in] struct l7vs_service* srv vitual service struct pointer
 *	@param[in] struct l7vs_conn* conn connection struct pointer
 */
static struct l7vs_dest *l7vs_sched_rr_schedule(struct l7vs_service *srv, 
                                                struct l7vs_conn *conn);

/*!
 *	RoundRobin Scheduling struct
 */
static struct l7vs_scheduler sched_rr_scheduler = {
        NULL,				/*! schedule handle */
        "rr",				/*! RoundRobin schedule module name */
        0,				/*! refarence count */
        l7vs_sched_rr_schedule,         /*! schedule function */
        NULL,                           /*! bind function */
        NULL,                           /*! unbind function */
        fini,                           /*! fini function */
	NULL,				/*! loglevel get function */
	NULL,				/*! debug log put function */
	NULL,				/*! info log put function */
	NULL,				/*! warn log put function */
	NULL,				/*! error log put function */
	NULL				/*! fatal log put function */
};

/*!
 *	RoundRoin schedule module initialize function
 *	@param[in] shared object handle;
 *	@return	   l7vs_scheduler struct pointer
 */
extern "C" struct l7vs_scheduler *
init(void *handle)
{
	IS_SCHEDRR_DEBUG{
		SCHED_DEBUG(30, "in_function struct l7vs_scheduler* init( void* handle ) handle = %p", handle );
	}

	servicedest_list = NULL;
        sched_rr_scheduler.handle = handle;

	IS_SCHEDRR_DEBUG{
		SCHED_DEBUG(31, "out_function struct l7vs_scheduler* init( void* handle ) return = %p", &sched_rr_scheduler );
	}
        return &sched_rr_scheduler;
}

/*
 *	RoundRobin module finalize function
 *	remove servicedest_list.
 *	@param[in]	void
 *	@return 	void
 */

static void
fini(void)
{
	IS_SCHEDRR_DEBUG{
		SCHED_DEBUG(32, "in_function static void fini(void)" );
	}
	GList*	ptr = NULL;
	for( ptr = g_list_first( servicedest_list ); ptr ; ptr = g_list_next(ptr) ){
		IS_SCHEDRR_DEBUG{
			SCHED_DEBUG(33, "service_dest_list free element = %p" , ptr->data );
		}
		free( ptr->data );
		ptr->data = NULL;
	}
	g_list_free(servicedest_list);
	servicedest_list = NULL;
	IS_SCHEDRR_DEBUG{
		SCHED_DEBUG(34, "out_function static void fini(void)" );
	}
}

/*!
 *	RoundRobin module scheduling function
 *	@param[in]	struct l7vs_service* srv virtual service struct pointer
 *	@param[in]	struct l7vs_conn* conn connection struct pointer
 *	@return		l7vs_dest* scheduled dest pointer
 */
static struct l7vs_dest *
l7vs_sched_rr_schedule(struct l7vs_service *srv, struct l7vs_conn *conn)
{
	IS_SCHEDRR_DEBUG{
		char	buff[DEBUG_STR_LEN];
		char	srv_buff[DEBUG_STR_LEN];
		char	con_buff[DEBUG_STR_LEN];
		snprintf( buff, DEBUG_STR_LEN, "in_function static struct l7vs_dest* l7vs_sched_rr_schedule( struct l7vs_service*, struct l7vs_conn* ) /" );
		l7vs_service_c_str( srv_buff, srv );
		l7vs_conn_c_str( con_buff, conn );
		strncat( buff, "srv = ", DEBUG_STR_LEN - strlen(buff) );
		strncat( buff, srv_buff, DEBUG_STR_LEN - strlen(buff) );
		strncat( buff, " / conn = ", DEBUG_STR_LEN - strlen(buff) );
		strncat( buff, con_buff, DEBUG_STR_LEN - strlen(buff) );
		SCHED_DEBUG(35, buff );
	}
	struct l7vs_dest* ret = NULL;
	GList* active_dest_list = NULL;
	GList* ptr = NULL;
	struct servicedest_contenor* contenor = NULL;

	if( !srv ){
		SCHED_ERROR(16, "service struct pointer is NULL" );
		IS_SCHEDRR_DEBUG{
			SCHED_DEBUG(36, "out_function static struct l7vs_dest* l7vs_sched_rr_schedule( struct l7vs_service*, struct l7vs_conn* return NULL" );
		}
		return NULL;
	}

	//create active list
	for( ptr = g_list_first( srv->dest_list ); ptr ; ptr = g_list_next(ptr) ){
		if( ((struct l7vs_dest*) ( ptr->data ))->weight > 0 ) active_dest_list = g_list_append( active_dest_list, ptr->data );
	}

	//active dest don't have 
	if( !active_dest_list ){
		SCHED_INFO(2, "active dist none. don't schedule dest" );
		IS_SCHEDRR_DEBUG{
			SCHED_DEBUG(37, "out_function static struct l7vs_dest* l7vs_sched_rr_schedule( struct l7vs_service*, struct l7vs_conn ) return NULL" );
		}
		return NULL;
	}


	//first use on service
	if( !srv->sched_data ){
		ptr = g_list_first( active_dest_list );
		ret = (struct l7vs_dest*) ptr->data;
		contenor = (struct servicedest_contenor*) malloc( sizeof(struct servicedest_contenor) );
		servicedest_list = g_list_append( servicedest_list, contenor );
		srv->sched_data = (void*) contenor;
		goto L7VS_SCHED_RR_SCHEDULE_END;
	}
	contenor = (struct servicedest_contenor*) srv->sched_data;
	for( ptr = g_list_first( active_dest_list ); ptr ; ptr = g_list_next( ptr ) ){
		struct l7vs_dest* dest = (struct l7vs_dest*) ptr->data;
		if( !memcmp( &(dest->addr.sin_addr), &(contenor->addr), sizeof( struct in_addr ) )
		    &&
		    dest->addr.sin_port == contenor->port ){
		    ret = (struct l7vs_dest*)ptr->data;
		    break;
		}
	}
	if( ! ret ){//not have dest list
		ptr = g_list_first( active_dest_list );
		ret = ( struct l7vs_dest* ) ptr->data;
		goto L7VS_SCHED_RR_SCHEDULE_END;
	}
	ptr = g_list_next( ptr );
	if( !ptr ){
		ptr = g_list_first( active_dest_list );
		ret = ( struct l7vs_dest* ) ptr->data;
		goto L7VS_SCHED_RR_SCHEDULE_END;
	}
	ret = ( struct l7vs_dest* ) ptr->data; 

L7VS_SCHED_RR_SCHEDULE_END:
	contenor->handle = srv->handle;
	contenor->addr = ret->addr.sin_addr;
	contenor->port = ret->addr.sin_port;
	g_list_free( active_dest_list );
	IS_SCHEDRR_DEBUG{
		SCHED_DEBUG(38, "out_function static struct l7vs_dest* l7vs_sched_rr_schedule( struct l7vs_service*, struct l7vs_conn* ) return %p" , ret );
	}
	return ret;
}
