/*
 * Interrupt handling for RX-INTC IPR
 *
 * Copyright (C) 2010  Yoshinori Sato
 *
 */

#include <linux/linkage.h>
#include <linux/interrupt.h>
#include <asm/io.h>
#define IR (0x00087000)
#define IER (0x00087200)

static void disable_ipr_irq(unsigned int irq)
{
	void *ipr = get_irq_chip_data(irq);
	__raw_writeb(0, ipr);
}

static void enable_ipr_irq(unsigned int irq)
{
	unsigned int offset;
	unsigned int bit;
	u8 ier;
	void *ipr = get_irq_chip_data(irq);
	__raw_writeb(1, ipr);
	offset = irq / 8;
	bit = irq % 8;
	ier = __raw_readb((void __iomem *)(IER + offset));
	ier |= (1 << bit);		/* enable IRQ on ICU */
	__raw_writeb(ier, (void __iomem *)(IER + offset));
}

const static struct {
	int irq;
	int ipr;
} irq_info[] __initdata = {
#if defined(CONFIG_CPU_RX610)
	{ 20, 0x01}, { 22, 0x02}, /* FCU */
	{ 28, 0x04}, { 29, 0x05}, { 30, 0x06}, { 31, 0x07}, /* CMT */
	{ 64, 0x20}, { 65, 0x21}, { 66, 0x22}, { 67, 0x23}, /* EXT */
	{ 68, 0x24}, { 69, 0x25}, { 70, 0x26}, { 71, 0x27},
	{ 72, 0x28}, { 73, 0x29}, { 74, 0x2a}, { 75, 0x2b},
	{ 76, 0x2c}, { 77, 0x2d}, { 78, 0x2e}, { 79, 0x2f},
	{ 96, 0x40}, /* WDT */
	{ 98, 0x44}, { 99, 0x45}, {100, 0x46}, {101, 0x47}, /* ADC */
	{104, 0x4c}, {105, 0x4c}, {106, 0x4c}, {107, 0x4c}, /* TPU */
	{108, 0x4d}, 
	{111, 0x4e}, {112, 0x4e}, {115, 0x4f}, {116, 0x4f},
	{117, 0x50}, {118, 0x50}, {120, 0x51}, {121, 0x51},
	{122, 0x52}, {123, 0x52}, {124, 0x52}, {125, 0x52},
	{126, 0x53}, 
	{127, 0x54}, {128, 0x54}, {131, 0x55}, {132, 0x55},
	{133, 0x56}, {134, 0x56}, {136, 0x57}, {137, 0x57},
	{138, 0x58}, {139, 0x58}, {140, 0x58}, {141, 0x58},
	{142, 0x59}, 
	{145, 0x5a}, {146, 0x5a}, {149, 0x5b}, {150, 0x5b},
	{151, 0x5c}, {152, 0x5c}, {154, 0x5d}, {155, 0x5d},
	{156, 0x5e}, {157, 0x5e}, {158, 0x5e}, {159, 0x5e},
	{160, 0x5f}, 
	{161, 0x60}, {162, 0x60}, {165, 0x61}, {166, 0x61},
	{167, 0x62}, {168, 0x62}, {170, 0x63}, {171, 0x63},
	{174, 0x68}, {175, 0x68}, {176, 0x68}, /* TMR */
	{177, 0x69}, {178, 0x69}, {179, 0x69},
	{180, 0x6a}, {181, 0x6a}, {182, 0x6a},
	{183, 0x6b}, {184, 0x6b}, {185, 0x6b},
	{198, 0x70}, {199, 0x71}, {200, 0x72}, {201, 0x73}, /* DMA */
	{214, 0x80}, {215, 0x80}, {216, 0x80}, {217, 0x80}, /* SCI */
	{218, 0x81}, {219, 0x81}, {220, 0x81}, {221, 0x81},
	{222, 0x82}, {223, 0x82}, {224, 0x82}, {225, 0x82},
	{226, 0x83}, {227, 0x83}, {228, 0x83}, {229, 0x83},
	{230, 0x84}, {231, 0x84}, {232, 0x84}, {233, 0x84},
	{234, 0x85}, {235, 0x85}, {236, 0x85}, {237, 0x85},
	{238, 0x86}, {239, 0x86}, {240, 0x86}, {241, 0x86},
	{246, 0x88}, {247, 0x89}, {248, 0x8a}, {249, 0x8b}, /* RIIC */
	{250, 0x8c}, {251, 0x8d}, {252, 0x8e}, {253, 0x8f},
#endif
#if defined(CONFIG_CPU_RX62N)
	{ 20, 0x01}, { 22, 0x02}, /* FCU */
	{ 28, 0x04}, { 29, 0x05}, { 30, 0x06}, { 31, 0x07}, /* CMT */
	{ 32, 0x08}, /* EINT */
	{ 36, 0x0c}, { 37, 0x0d}, { 38, 0x0e}, /* USB */
	{ 40, 0x10}, { 41, 0x11}, { 42, 0x12},
	{ 44, 0x14}, { 45, 0x14}, { 46, 0x14}, { 47, 0x14}, /* RSPI0 */
	{ 48, 0x15}, { 49, 0x15}, { 50, 0x15}, { 51, 0x15},
	{ 56, 0x18}, { 57, 0x18}, { 58, 0x18}, { 59, 0x18}, { 60, 0x18}, /* CAN */
	{ 62, 0x1e}, { 63, 0x1f}, /* RTC */
	{ 64, 0x20}, { 65, 0x21}, { 66, 0x22}, { 67, 0x23}, /* EXT */
	{ 68, 0x24}, { 69, 0x25}, { 70, 0x26}, { 71, 0x27},
	{ 72, 0x28}, { 73, 0x29}, { 74, 0x2a}, { 75, 0x2b},
	{ 76, 0x2c}, { 77, 0x2d}, { 78, 0x2e}, { 79, 0x2f},
	{ 90, 0x3a}, { 91, 0x3b}, /* USB */
	{ 92, 0x3c}, /* RTC */
	{ 96, 0x40}, /* WDT */
	{ 98, 0x44}, { 99, 0x45}, {102, 0x48}, /* ADC */
	{114, 0x51}, {115, 0x51}, {116, 0x51}, {117, 0x51}, /* MTU */
	{118, 0x52}, {119, 0x52}, {120, 0x52},
	{121, 0x53}, {122, 0x53}, {123, 0x54}, {124, 0x54},
	{125, 0x55}, {126, 0x55}, {127, 0x56}, {128, 0x56},
	{129, 0x57}, {130, 0x57}, {131, 0x57}, {132, 0x57}, {133, 0x58},
	{134, 0x59}, {135, 0x59}, {136, 0x59}, {137, 0x59}, {138, 0x5a},
	{139, 0x5b}, {140, 0x5b}, {141, 0x5b},
	{142, 0x5c}, {143, 0x5c}, {144, 0x5c}, {145, 0x5c},
	{146, 0x5d}, {147, 0x5d}, {148, 0x5d},
	{149, 0x5e}, {150, 0x5e}, {151, 0x5f}, {152, 0x5f},
	{153, 0x60}, {154, 0x60}, {155, 0x61}, {156, 0x61},
	{157, 0x62}, {158, 0x62}, {159, 0x62}, {160, 0x62}, {161, 0x63},
	{162, 0x64}, {163, 0x64}, {164, 0x64}, {165, 0x64}, {166, 0x65},
	{167, 0x66}, {168, 0x66}, {169, 0x66},
	{170, 0x67}, {171, 0x67}, {172, 0x67}, {173, 0x67}, /* POE */
	{174, 0x68}, {175, 0x68}, {176, 0x68}, /* TMR */
	{177, 0x69}, {178, 0x69}, {179, 0x69},
	{180, 0x6a}, {181, 0x6a}, {182, 0x6a},
	{183, 0x6b}, {184, 0x6b}, {185, 0x6b},
	{198, 0x70}, {199, 0x71}, {200, 0x72}, {201, 0x73}, /* DMAC */
	{202, 0x74}, {203, 0x75}, /* EXDMA */
	{214, 0x80}, {215, 0x80}, {216, 0x80}, {217, 0x80}, /* SCI */
	{218, 0x81}, {219, 0x81}, {220, 0x81}, {221, 0x81},
	{222, 0x82}, {223, 0x82}, {224, 0x82}, {225, 0x82},
	{226, 0x83}, {227, 0x83}, {228, 0x83}, {229, 0x83},
	{234, 0x85}, {235, 0x85}, {236, 0x85}, {237, 0x85},
	{238, 0x86}, {239, 0x86}, {240, 0x86}, {241, 0x86},
	{246, 0x88}, {247, 0x89}, {248, 0x8a}, {249, 0x8b}, /* RIIC */
	{250, 0x8c}, {251, 0x8d}, {252, 0x8e}, {253, 0x8f},
#endif
};

void __init setup_rx_irq_desc(struct irq_chip *chip)
{
	int i;

	chip->mask = disable_ipr_irq;
	chip->unmask = enable_ipr_irq;
	chip->mask_ack = disable_ipr_irq;

	for (i = 0; i < ARRAY_SIZE(irq_info); i++) {
		struct irq_desc *irq_desc;

		irq_desc = irq_to_desc_alloc_node(irq_info[i].irq, numa_node_id());
		if (unlikely(!irq_desc)) {
			printk(KERN_INFO "can not get irq_desc for %d\n",
			       irq_info[i].irq);
			continue;
		}

		disable_irq_nosync(irq_info[i].irq);
		set_irq_chip_and_handler_name(irq_info[i].irq, chip, handle_level_irq,"");
		set_irq_chip_data(irq_info[i].irq, (void *)(0x00087300 +irq_info[i].ipr));
		disable_ipr_irq(irq_info[i].irq);
	}
}
