/*
 * Copyright (C) 2007 u6k.yu1@gmail.com, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.gr.java_conf.u6k.animation;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.awt.image.BufferStrategy;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFrame;

/**
 * <p>
 * アニメーションを行うウィンドウを表します。
 * </p>
 * 
 * @version $Id$
 */
public final class AnimationWindow {

    /**
     * <p>
     * ウィンドウが閉じたかどうか。
     * </p>
     */
    private boolean                 closed;

    /**
     * <p>
     * ウィンドウの表示領域の横幅。
     * </p>
     */
    private int                     width;

    /**
     * <p>
     * ウィンドウの表示領域の縦幅。
     * </p>
     */
    private int                     height;

    /**
     * <p>
     * ウィンドウへの表示などの処理を行うリスナー。
     * </p>
     */
    private List<AnimationListener> listeners = new ArrayList<AnimationListener>();

    /**
     * <p>
     * ウィンドウ。
     * </p>
     */
    private JFrame                  frame     = new JFrame();

    /**
     * <p>
     * 描画用のキャンバス。
     * </p>
     */
    private Graphics2D              g;

    /**
     * <p>
     * マウス情報。
     * </p>
     */
    private MouseInfo               mouseInfo = new MouseInfo();

    /**
     * <p>
     * キーボード情報。
     * </p>
     */
    private KeyInfo                 keyInfo   = new KeyInfo();

    /**
     * <p>
     * ウィンドウを初期化します。
     * </p>
     * 
     * @param width
     *            ウィンドウの表示領域の横幅。
     * @param height
     *            ウィンドウの表示領域の縦幅。
     * @param title
     *            ウィンドウのタイトル。
     */
    public AnimationWindow(int width, int height, String title) {
        this.width = width;
        this.height = height;

        this.frame.setTitle(title);

        this.setBackgroundColor(Color.BLACK);
        this.setForegroundColor(Color.WHITE);
    }

    /**
     * <p>
     * ウィンドウの横幅を返します。
     * </p>
     * 
     * @return ウィンドウの横幅。
     */
    public int getWidth() {
        return this.width;
    }

    /**
     * <p>
     * ウィンドウの縦幅を返します。
     * </p>
     * 
     * @return ウィンドウの縦幅。
     */
    public int getHeight() {
        return this.height;
    }

    /**
     * <p>
     * ウィンドウのタイトルを返します。
     * </p>
     * 
     * @return ウィンドウのタイトル。
     */
    public String getTitle() {
        return this.frame.getTitle();
    }

    /**
     * <p>
     * 背景色を返します。
     * </p>
     * 
     * @return 背景色。
     */
    public Color getBackgroundColor() {
        return this.frame.getBackground();
    }

    /**
     * <p>
     * 背景色を設定します。
     * </p>
     * 
     * @param backgroundColor
     *            背景色。
     */
    public void setBackgroundColor(Color backgroundColor) {
        this.frame.setBackground(backgroundColor);
    }

    /**
     * <p>
     * 前景色を返します。
     * </p>
     * 
     * @return 前景色。
     */
    public Color getForegroundColor() {
        return this.frame.getForeground();
    }

    /**
     * <p>
     * 前景色を設定します。
     * </p>
     * 
     * @param foregroundColor
     *            前景色。
     */
    public void setForegroundColor(Color foregroundColor) {
        this.frame.setForeground(foregroundColor);
    }

    /**
     * <p>
     * 描画用のキャンバスを返します。このメソッドは{@link AnimationListener}実装クラスにより呼び出されることを想定しています。
     * </p>
     * 
     * @return 描画用のキャンバス。
     */
    public Graphics2D getCanvas() {
        return this.g;
    }

    /**
     * <p>
     * マウス情報を返します。
     * </p>
     * 
     * @return マウス情報。
     */
    public MouseInfo getMouseInfo() {
        return this.mouseInfo;
    }

    /**
     * <p>
     * キーボード情報を返します。
     * </p>
     * 
     * @return キーボード情報。
     */
    public KeyInfo getKeyInfo() {
        return this.keyInfo;
    }

    /**
     * <p>
     * アニメーション・リスナーを追加します。
     * </p>
     * 
     * @param l
     *            アニメーション・リスナー。
     */
    public void addAnimationListener(AnimationListener l) {
        if (l != null) {
            this.listeners.add(l);
        }
    }

    /**
     * <p>
     * アニメーション・リスナーを削除します。
     * </p>
     * 
     * @param l
     *            アニメーション・リスナー。
     */
    public void removeAnimationListener(AnimationListener l) {
        this.listeners.remove(l);
    }

    /**
     * <p>
     * ウィンドウを閉じます。
     * </p>
     */
    public void close() {
        this.closed = true;
    }

    /**
     * <p>
     * ウィンドウを表示し、アニメーションを開始します。
     * </p>
     */
    public void show() {
        try {
            this.frame.addMouseListener(new MouseListener() {

                public void mouseClicked(MouseEvent e) {
                }

                public void mouseEntered(MouseEvent e) {
                }

                public void mouseExited(MouseEvent e) {
                }

                public void mousePressed(MouseEvent e) {
                    if (e.getButton() == MouseEvent.BUTTON1) {
                        AnimationWindow.this.mouseInfo.setLeftButton(true);
                    } else if (e.getButton() == MouseEvent.BUTTON3) {
                        AnimationWindow.this.mouseInfo.setRightButton(true);
                    }
                }

                public void mouseReleased(MouseEvent e) {
                    if (e.getButton() == MouseEvent.BUTTON1) {
                        AnimationWindow.this.mouseInfo.setLeftButton(false);
                    } else if (e.getButton() == MouseEvent.BUTTON3) {
                        AnimationWindow.this.mouseInfo.setRightButton(false);
                    }
                }

            });

            this.frame.addKeyListener(new KeyListener() {

                public void keyPressed(KeyEvent e) {
                    AnimationWindow.this.keyInfo.setKey(e.getKeyCode(), true);
                }

                public void keyReleased(KeyEvent e) {
                    AnimationWindow.this.keyInfo.setKey(e.getKeyCode(), false);
                }

                public void keyTyped(KeyEvent e) {
                }

            });

            this.frame.addMouseMotionListener(new MouseMotionListener() {

                public void mouseDragged(MouseEvent e) {
                    AnimationWindow.this.mouseInfo.setX(e.getX() - AnimationWindow.this.frame.getInsets().left);
                    AnimationWindow.this.mouseInfo.setY(e.getY() - AnimationWindow.this.frame.getInsets().top);
                }

                public void mouseMoved(MouseEvent e) {
                    AnimationWindow.this.mouseInfo.setX(e.getX() - AnimationWindow.this.frame.getInsets().left);
                    AnimationWindow.this.mouseInfo.setY(e.getY() - AnimationWindow.this.frame.getInsets().top);
                }

            });

            this.frame.addWindowListener(new WindowListener() {

                public void windowOpened(WindowEvent event) {
                }

                public void windowClosing(WindowEvent event) {
                    AnimationWindow.this.closed = true;
                }

                public void windowClosed(WindowEvent event) {
                }

                public void windowIconified(WindowEvent event) {
                }

                public void windowDeiconified(WindowEvent event) {
                }

                public void windowActivated(WindowEvent event) {
                }

                public void windowDeactivated(WindowEvent event) {
                }

            });

            this.frame.setVisible(true);

            int l = this.frame.getInsets().left;
            int t = this.frame.getInsets().top;
            int r = this.frame.getInsets().right;
            int b = this.frame.getInsets().bottom;

            this.frame.setSize(l + r + this.width, t + b + this.height);
            this.frame.setResizable(false);

            this.frame.createBufferStrategy(2);
            BufferStrategy bs = this.frame.getBufferStrategy();
            while (!this.closed) {
                this.g = (Graphics2D) bs.getDrawGraphics();
                try {
                    this.g.translate(l, t);
                    this.g.setClip(0, 0, this.width, this.height);
                    for (AnimationListener listener : this.listeners) {
                        listener.draw(this);
                    }
                    bs.show();
                } finally {
                    this.g.dispose();
                }
            }
        } finally {
            this.frame.dispose();
        }
    }

}
