/*
 * Copyright (C) 2007 u6k.yu1@gmail.com, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.gr.java_conf.u6k.animation;


/**
 * <p>
 * FPSを測定するリスナーです。1秒ごとに{@link #elapsed(double)}メソッドが呼び出され、計測したFPSが渡されます。
 * </p>
 * 
 * @version $Id$
 */
public abstract class AbstractFpsListener implements AnimationListener {

    /**
     * <p>
     * 1秒のナノ秒数。
     * </p>
     */
    private static final double NANO_SECONDS = 1000000000;

    /**
     * <p>
     * 基準時刻。この時刻と現在時刻の差を見て、1秒経過したかどうかを測定します。
     * </p>
     */
    private long                time;

    /**
     * <p>
     * 計測中の処理回数。
     * </p>
     */
    private int                 count;

    /**
     * <p>
     * 前回計測したFPS。
     * </p>
     */
    private double              fps;

    /**
     * <p>
     * 前回計測したFPSを返します。
     * </p>
     * 
     * @return 前回計測したFPS。
     */
    public final double getFps() {
        return this.fps;
    }

    /**
     * {@inheritDoc}
     */
    public final void draw(AnimationWindow window) {
        this.count++;

        double elapsedTime = (double) (System.nanoTime() - this.time) / AbstractFpsListener.NANO_SECONDS;
        if (elapsedTime > 1) {
            this.fps = (double) this.count / elapsedTime;
            this.elapsed(this.fps);

            this.time = System.nanoTime();
            this.count = 0;
        }
    }

    /**
     * <p>
     * 1秒間が経過したときに呼び出され、計測したFPSが渡されます。
     * </p>
     * 
     * @param fps
     *            計測したFPS。
     */
    public abstract void elapsed(double fps);

}
