#!/usr/bin/env python
import sys, time, locale, os, glob
from PyQt4 import QtCore, QtGui, uic
from twitturnConfigDialog import TwitturnConfigDialog
from twitturnTimeLineWidget import TwitturnTimeLineWidget
from twitturnCore import TwitturnCore
from iconCacheManager import IconChacheManager


TWITTER_MAX_STRING_LENGTH = 140
class ExtendedQTextEdit(QtGui.QTextEdit):
    def keyPressEvent(self, e):
        if e.key() in (QtCore.Qt.Key_Return, QtCore.Qt.Key_Enter):
            self.emit(QtCore.SIGNAL("ReteurnOrEnterPressed"))
        QtGui.QTextEdit.keyPressEvent(self, e)

class ExtendedQLineEdit(QtGui.QLineEdit):
    def setTreeWidget(self, treeWidget):
        self.treeWidget = treeWidget
    def setSearchTab(self, searchTab):
        self.searchTab = searchTab
    def keyPressEvent(self, keyEvent):
        if keyEvent.key() == QtCore.Qt.Key_Escape:
            self.setVisible(False)
        if keyEvent.key() in (
                QtCore.Qt.Key_Return,
                QtCore.Qt.Key_Enter):
            self.doSearching()
        QtGui.QLineEdit.keyPressEvent(self, keyEvent)
    def doSearching(self):
            self.searchTab.treeWidget.clear()
            rx = QtCore.QRegExp( 
                    self.text(), 
                    QtCore.Qt.CaseInsensitive,
                    QtCore.QRegExp.RegExp2
                    )
            for i in range(self.treeWidget.topLevelItemCount()):
                item = self.treeWidget.topLevelItem(i)
                for c in [1,2]:
                    if not (-1 == rx.indexIn( item.text(c))):
                        self.searchTab.treeWidget.addTopLevelItem(
                                item.clone())
                        break
            self.setVisible(False)
            self.searchTab.changeThisTab()


form_class, base_class = uic.loadUiType("mainwindow.ui")
class Twitturn(base_class, form_class):
    def __init__(self, *args):
        base_class.__init__(self, *args)

        self.setupUi(self)

        self.textEdit.__class__ = ExtendedQTextEdit
        self.tabWidget = QtGui.QTabWidget(self)
        self.verticalLayoutCenter.addWidget(self.tabWidget)
        self.mainTimeLine = TwitturnTimeLineWidget(self.tabWidget)
        self.tabWidget.addTab(self.mainTimeLine, self.tr("&TimeLine"))
        self.treeWidget = self.mainTimeLine.treeWidget
        self.treeWidget.setAttribute(QtCore.Qt.WA_InputMethodEnabled, False)
        self.lineEdit.__class__ = ExtendedQLineEdit
        self.lineEdit.setTreeWidget(self.treeWidget)
        self.lineEdit.setSearchTab(None)
        self.lineEdit.setVisible(False)

        self.loadingIcon = QtGui.QMovie("images/loader.mng")
        self.labelLoading.setMovie(self.loadingIcon)
        self.loadingIcon.start()

        self.relatedNickHighlighting = True
        self.iconCache = IconChacheManager()
        self.appendingStatusesQeue = None
        self.latestPostTimestamp = None
        self.userNick = None
        self.timer = QtCore.QTimer(self)
        self.rxReply = QtCore.QRegExp("@([a-zA-Z0-9_]+)")
        # If URL includes '@', this regexp will not accept.
        self.rxURL = QtCore.QRegExp("(https?:\/\/[-_.!~*'()a-zA-Z0-9;/?:&=+$,%#]+)")
        self.settings = QtCore.QSettings("Twitturn")

        self.core = TwitturnCore(self)
        self.restoreGUISettings()
        self.applySettings()
        self.connectActions()

        self.timer.start()
        self.fetchTimeline()
#        self.fillSamplePosts()
#        self.treeWidget.setCurrentItem( self.treeWidget.topLevelItem(0))


    def connectActions(self):
        self.connect(self.actionFocus_Imput_Erea, 
                QtCore.SIGNAL("triggered()"),
                self.textEdit, QtCore.SLOT("setFocus()"))
        self.connect(self.action_Quit, QtCore.SIGNAL("triggered()"),
                self.exit)
        self.connect(self.actionSelect_Top, QtCore.SIGNAL("triggered()"),
                self.selectTopPost)
        self.connect(self.action_Configure, QtCore.SIGNAL("triggered()"),
                self.showConfigDialog)
        self.connect(self.actionFocus_Time_Line, 
                QtCore.SIGNAL("triggered()"),
                self.focusTimeLine)
        self.connect(self.action_TimeLine, QtCore.SIGNAL("triggered()"),
                self.fetchTimeline)
        self.connect(self.action_Replies, QtCore.SIGNAL("triggered()"),
                self.fetchReplies)
        self.connect(self.action_DirectMessages, 
                QtCore.SIGNAL("triggered()"),
                self.fetchDirectMessages)
        self.connect(self.actionSearch,
                QtCore.SIGNAL("triggered()"),
                self.beginSearching)
        self.connect(self.actionSearch_Relative,
                QtCore.SIGNAL("triggered()"),
                self.searchRelative)
        self.connect(
                self.mainTimeLine.treeWidget, 
                QtCore.SIGNAL('currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)'),
                self.setCurrentPost)
        self.connect(
                self.mainTimeLine.treeWidget,
                QtCore.SIGNAL('itemDoubleClicked(QTreeWidgetItem *,int)'),
                self.putReplyHeader)
        self.connect(self.textEdit, QtCore.SIGNAL("textChanged()"),
                self.countText)
        self.connect(
                self.textEdit, 
                QtCore.SIGNAL("ReteurnOrEnterPressed"),
                self.update)
        self.connect(self.pushButtonWeb, QtCore.SIGNAL("clicked()"),
                self.openWeb)
        self.connect(self.pushButtonUpdate, QtCore.SIGNAL("clicked()"),
                self.update)
        self.connect(self.core, QtCore.SIGNAL("friendsTimelineUpdated"),
                self.updateTimeline)
        self.connect(self.core, QtCore.SIGNAL("done(bool)"), self.networkCheck)
        self.connect(
                self.core, 
                QtCore.SIGNAL("responseHeaderReceived (const QHttpResponseHeader&)"),
                self.printStatusCode)
        self.connect(self.core, QtCore.SIGNAL("statusUpdated"),
                self.doPostingPhase2)
        self.connect(self.timer, QtCore.SIGNAL("timeout()"),
                self.fetchTimeline)
        self.connect(
                self.iconCache, 
                QtCore.SIGNAL("iconReady(QIcon)"),
                self.appendStatusWithIcon)
        self.connect(
                self.tabWidget,
                QtCore.SIGNAL("currentChanged(int)"),
                self.changeCurrentTab)
        return

    def changeCurrentTab(self, tabIdx):
        self.treeWidget = self.tabWidget.currentWidget().treeWidget

        if self.treeWidget.indexOfTopLevelItem(
                self.treeWidget.currentItem()) == -1:
            self.treeWidget.setCurrentItem(
                self.treeWidget.topLevelItem(0))
        self.treeWidget.setFocus()

        current = self.treeWidget.currentItem()
        if not current == None:
            self.setCurrentPost(current, None)

    def applySettings(self):
        self.applyTwitturnCoreSettings()
        self.userNick = self.settings.value("twitter/user").toString()
        self.timer.setInterval(
                self.settings.value("twitter/interval").toInt()[0] * 1000)
        self.applyIconChacheSettings()
    def applyTwitturnCoreSettings(self):
        self.settings.beginGroup("twitter")
        self.core.setUser(
                self.settings.value("user").toString(),
                self.settings.value("password").toString())
        if self.settings.value("useGETMethod").toBool():
            self.core.setQueryMethod("GET")
        else:
            self.core.setQueryMethod("POST")
        self.settings.endGroup()
        self.settings.beginGroup("proxy")
        if self.settings.value("useProxy").toBool():
            self.core.setProxy(
                    self.settings.value("server").toString(),
                    self.settings.value("port").toInt())
        self.settings.endGroup()
    def applyIconChacheSettings(self):
        self.settings.beginGroup("proxy")
        if self.settings.value("useProxy").toBool():
            self.iconCache.setProxy(
                    self.settings.value("server").toString(),
                    self.settings.value("port").toInt())
        self.settings.endGroup()

    def fetchTimeline(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        self.core.queryFriendsTimeline(None, None, self.latestPostTimestamp)

    def doPostingPhase2(self):
        self.textEdit.clear()
        if self.settings.value("twitter/fetchTimeLineWhenPosting").toBool():
            print self.latestPostTimestamp
            self.core.queryFriendsTimeline(None, None, self.latestPostTimestamp)
        else:
            self.escapeFetchingMode()
        
    def networkCheck(self, error):
        if error:
            self.escapeFetchingMode()
            if self.core.AuthenticationRequiredError:
                QtGui.QMessageBox.warning(self,
                        self.tr("Twitturn"),
                        self.tr("""Twitter authentication error!
Incorrect user name or password.
Could you check your settings?"""))
                self.showConfigDialog()
            print self.core.errorString()
            return

    def printStatusCode(self, resp):
        if not resp.statusCode() == 200:
            self.escapeFetchingMode()
        print resp.statusCode()
        print resp.toString()
    def update(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        if self.textEdit.toPlainText() == "":
            print self.latestPostTimestamp
            self.core.queryFriendsTimeline(None, None, self.latestPostTimestamp)
        else:
            self.core.queryUpdate(str(QtCore.QUrl.toPercentEncoding(
                self.textEdit.toPlainText())))
    def fetchReplies(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        self.core.queryReplies()
    def fetchDirectMessages(self):
        if not self.pushButtonUpdate.isEnabled():
            return
        self.enterFetchingMode()
        self.core.queryDirectMessages()

    def updateTimeline(self, statuses):
        if len(statuses) == 0:
            self.escapeFetchingMode()
            return
        self.latestPostTimestamp = statuses[0][3]
        self.appendingStatusesQeue = statuses
        self.iconCache.queryIcon(
                self.appendingStatusesQeue[0][0],
                self.appendingStatusesQeue[0][1])
    def appendStatusWithIcon(self, icon):
        if len(self.appendingStatusesQeue) == 0:
            self.escapeFetchingMode()
            return
        self.addPost(icon, self.appendingStatusesQeue.pop(0))
        if len(self.appendingStatusesQeue) == 0:
            self.escapeFetchingMode()
            return
        self.iconCache.queryIcon(
                self.appendingStatusesQeue[0][0],
                self.appendingStatusesQeue[0][1])
        return
        
    def addPost(self, icon, post):
        if self.isDuplicatePost(post): 
            print "Duplicate post detected: statusID " + post[4]
            return

        treeEntry = QtGui.QTreeWidgetItem()
        treeEntry.setIcon(0, icon)
        for i, v in enumerate(post[1: ]):
            if i == 2:
                # Twitter API response sample: 
                # Wed Dec 17 02:30:10 +0000 2008
                # Sun Dec 07 22:25:26 +0000 2008 
                dt = QtCore.QLocale( 
                        QtCore.QLocale.English,
                        ).toDateTime(
                    v, 
                    "ddd MMM dd hh:mm:ss +0000 yyyy")
                dt.setTimeSpec(QtCore.Qt.UTC)
                v =  dt.toLocalTime().toString("yyyy/MM/dd hh:mm:ss")
            treeEntry.setText(i+1,v)


        self.emit(QtCore.SIGNAL("newPostCame"), treeEntry)
        self.mainTimeLine.addPost(treeEntry, False)

        item = self.treeWidget.currentItem()
        if not item == None:
            if item.text(1) == post[1]:
                treeEntry.setSelected(True)

        if self.isSelfPost(post[1], self.userNick):
            treeEntry.setForeground( 1, QtGui.QBrush(QtGui.QColor("darkcyan")))
            treeEntry.setForeground( 2, QtGui.QBrush(QtGui.QColor("darkcyan")))
            return

        if not self.isReply(post[2], self.userNick):
            return 
        treeEntry.setForeground(1, QtGui.QBrush(QtGui.QColor("red")))
        treeEntry.setForeground(2, QtGui.QBrush(QtGui.QColor("red")))

    def isDuplicatePost(self, post):
        statusIdColumn = 4
        items = self.mainTimeLine.treeWidget.findItems(
                post[statusIdColumn],
                QtCore.Qt.MatchExactly,
                statusIdColumn)
        return len(items) != 0

    def isReply(self, text, nick):
        return QtCore.QString(text).contains(nick)
    def isSelfPost(self, postNick, yourNick):
        return postNick == yourNick;

    def enterFetchingMode(self):
        self.pushButtonUpdate.setEnabled(False)
        self.loadingIcon.setPaused(False)
        for i in range(self.tabWidget.count()):
            self.tabWidget.setTabIcon(i, QtGui.QIcon())
    def escapeFetchingMode(self):
        self.pushButtonUpdate.setEnabled(True)
        self.loadingIcon.setPaused(True)

    def searchTimeline(self, str):
        if str == "":
            return
        if self.radioButtonNick.isChecked():
            self.searchNick(str)
        else:
            self.searchText(str)
    def searchNick(self, nick):
        items = self.treeWidget.findItems(
            nick, 
            QtCore.Qt.MatchStartsWith, 
            1)
        for item in items:
            self.treeWidget.setItemSelected(item, True)

        item = self.treeWidget.currentItem()
        if item == None:
            return
        while True:
            if item.text(1).startsWith(nick):
                self.treeWidget.setCurrentItem(item)
                break
            if self.radioButtonForward.isChecked():
                item = self.treeWidget.itemBelow(item)
            else:
                item = self.treeWidget.itemAbove(item)
            if item == None:
                break

    def searchText(self, nick):
        items = self.treeWidget.findItems(
            nick, 
            QtCore.Qt.MatchContains, 
            2)
        for item in items:
            self.treeWidget.setItemSelected(item, True)

        item = self.treeWidget.currentItem()
        if item == None:
            return
        while True:
            if item.text(2).contains(nick):
                self.treeWidget.setCurrentItem(item)
                break
            if self.radioButtonForward.isChecked():
                item = self.treeWidget.itemBelow(item)
            else:
                item = self.treeWidget.itemAbove(item)
            if item == None:
                break

    def searchPrevious(self):
        forward = self.radioButtonForward.isChecked()
        if forward:
            self.radioButtonBackward.setChecked(True)
        else:
            self.radioButtonForward.setChecked(True)
        self.searchNext()
        if not forward:
            self.radioButtonBackward.setChecked(True)
        else:
            self.radioButtonForward.setChecked(True)

    def searchNickNext(self, nick):
        items = self.treeWidget.findItems(
            nick, 
            QtCore.Qt.MatchStartsWith, 
            1)
        for item in items:
            self.treeWidget.setItemSelected(item, True)

        item = self.treeWidget.currentItem()
        if item == None:
            return
        while True:
            if self.radioButtonForward.isChecked():
                item = self.treeWidget.itemBelow(item)
            else:
                item = self.treeWidget.itemAbove(item)
            if item == None:
                break
            if item.text(1).startsWith(nick):
                self.treeWidget.setCurrentItem(item)
                break

    def searchTextNext(self, nick):
        items = self.treeWidget.findItems(
            nick, 
            QtCore.Qt.MatchContains, 
            2)
        for item in items:
            self.treeWidget.setItemSelected(item, True)

        item = self.treeWidget.currentItem()
        if item == None:
            return
        while True:
            if self.radioButtonForward.isChecked():
                item = self.treeWidget.itemBelow(item)
            else:
                item = self.treeWidget.itemAbove(item)
            if item == None:
                break
            if item.text(2).contains(nick):
                self.treeWidget.setCurrentItem(item)
                break

    def showConfigDialog(self):
        self.timer.stop()
        dialog = TwitturnConfigDialog(self)
        dialog.exec_()
        self.applySettings()
        self.timer.start()

    def keyPressEvent(self, keyEvent):
        if self.treeWidget.hasFocus():
            self.oneKeyEvent(keyEvent)

    def oneKeyEvent(self, keyEvent):
        if keyEvent.key() == QtCore.Qt.Key_F:
            self.toggleFavo(self.treeWidget.currentItem())
        if keyEvent.key() == QtCore.Qt.Key_S:
            self.toggleSpam(self.treeWidget.currentItem())
        if keyEvent.key() == QtCore.Qt.Key_J:
            self.focusBellowPost()
        if keyEvent.key() == QtCore.Qt.Key_K:
            self.focusAbovePost()
        if keyEvent.key() == QtCore.Qt.Key_N:
            self.searchNick()
        if keyEvent.key() == QtCore.Qt.Key_W:
            self.openWeb()
        if keyEvent.key() == QtCore.Qt.Key_H:
            self.openHome()
        if keyEvent.key() in (
                QtCore.Qt.Key_B,
                QtCore.Qt.Key_L):
            self.openLinks()
        if keyEvent.key() in (
                QtCore.Qt.Key_R,
                QtCore.Qt.Key_Return,
                QtCore.Qt.Key_Enter):
            self.putReplyHeader()
        if keyEvent.key() in (
                QtCore.Qt.Key_A,
                QtCore.Qt.Key_Right):
            self.appendMultiReplyHeader()
        if keyEvent.key() == QtCore.Qt.Key_D:
            self.putDirectMessageHeader()
        if keyEvent.key() in (
                QtCore.Qt.Key_Backspace, 
                QtCore.Qt.Key_Delete):
            self.textEdit.clear()
        if keyEvent.key() in (
                QtCore.Qt.Key_G, 
                QtCore.Qt.Key_O,
                QtCore.Qt.Key_1):
            #self.sortByTime()
            self.treeWidget.setCurrentItem(self.treeWidget.topLevelItem(0))
        if keyEvent.key() == QtCore.Qt.Key_Slash:
            self.beginSearching()
        if keyEvent.key() == QtCore.Qt.Key_Asterisk:
            self.searchRelative()
        if keyEvent.key() == QtCore.Qt.Key_I:
            self.textEdit.setFocus()

        # For Debugging
        if keyEvent.key() == QtCore.Qt.Key_T:
            self.sortByTime()
            print "Time Sorting Done"

    def beginSearching(self):
        self.lineEdit.setVisible(True)
        self.lineEdit.setFocus()
    def searchRelative(self):
        current = self.treeWidget.currentItem()
        nicks = self.pickupNicks(current.text(2))
        nicks.append(current.text(1))
        str = QtCore.QString()
        for i, nick in enumerate(nicks):
            if i == 0:
                str.append("("+nick+")")
            else:
                str.append("|("+nick+")")

        self.lineEdit.setText(str)
        self.lineEdit.doSearching()
    def searchNick(self):
        current = self.treeWidget.currentItem()
        self.lineEdit.setText("^"+current.text(1)+"$")
        self.lineEdit.doSearching()


#    def toggleNickSort(self):
#        if self.treeWidget.sortColumn() == 1:
#            self.sortByTime()
#        else:
#            self.sortByNick()
#
#    def sortByNick(self):
#        self.treeWidget.sortItems(1,QtCore.Qt.DescendingOrder)
#        self.treeWidget.sortItems(1,QtCore.Qt.DescendingOrder) #No sort toggling
#        self.treeWidget.scrollToItem(self.treeWidget.currentItem())
    def sortByTime(self):
        self.treeWidget.sortItems(3,QtCore.Qt.DescendingOrder)
        self.treeWidget.sortItems(3,QtCore.Qt.DescendingOrder) #No sort toggling
        self.treeWidget.scrollToItem(self.treeWidget.currentItem())

    def focusBellowPost(self):
        currentItem = self.treeWidget.currentItem()
        bellowItem = self.treeWidget.itemBelow(currentItem)
        if not bellowItem == None:
            self.treeWidget.setCurrentItem(bellowItem)
    def focusAbovePost(self):
        currentItem = self.treeWidget.currentItem()
        aboveItem = self.treeWidget.itemAbove(currentItem)
        if not aboveItem == None:
            self.treeWidget.setCurrentItem(aboveItem)

    def toggleFavo(self, post):
        if not self.pushButtonUpdate.isEnabled():
            return
        if post.text(8) == "":
            self.core.queryFavorite(post.text(4), True)
            post.setText(8, "F")
        else:
            self.core.queryFavorite(post.text(4), False)
            post.setText(8, "")
    def toggleSpam(self, post):
        if post.text(9) == "True":
            post.setText(9, "False")
        else:
            post.setText(9, "True")
    def openWeb(self):
        post = self.treeWidget.currentItem()
        if not post.text(6) == "":
            QtGui.QDesktopServices.openUrl(QtCore.QUrl(post.text(6)))
    def openHome(self):
        post = self.treeWidget.currentItem()
        if not post.text(1) == "":
            QtGui.QDesktopServices.openUrl(
                    QtCore.QUrl('http://twitter.com/' + post.text(1)))
        pass
    def openLinks(self):
        str = self.treeWidget.currentItem().text(2)
        pos = 0
        while True:
            pos = self.rxURL.indexIn(str, pos)
            if pos == -1:
                break;
            QtGui.QDesktopServices.openUrl( QtCore.QUrl( self.rxURL.cap()))
            pos += self.rxURL.matchedLength()
            time.sleep(1)
        return
    def putReplyHeader(self):
        self.textEdit.setText(
                '@' + self.treeWidget.currentItem().text(1) + ' ')
        self.textEdit.moveCursor(QtGui.QTextCursor.End)
        self.textEdit.setFocus()
    def putDirectMessageHeader(self):
        self.textEdit.setText(
                'D ' + self.treeWidget.currentItem().text(1) + ' ')
        self.textEdit.moveCursor(QtGui.QTextCursor.End)
        self.textEdit.setFocus()
    def appendMultiReplyHeader(self):
        str = self.textEdit.toPlainText()
        if str == "":
            str = '.'
        str += '@' + self.treeWidget.currentItem().text(1) + ' '
        self.textEdit.setText(str)
        self.textEdit.moveCursor(QtGui.QTextCursor.End)

    def countText(self):
        self.labelTextCount.setText(
                '%s' % (TWITTER_MAX_STRING_LENGTH - self.textEdit.toPlainText().size()))

    def closeEvent(self, event):
        self.saveGUISettings()

    def exit(self):
        self.saveGUISettings()
        QtGui.qApp.quit()

    def restoreGUISettings(self):
        self.settings.beginGroup("GUI")
        self.treeWidget.header().restoreState(
                self.settings.value("treeWidget/header").toByteArray())
        self.splitter.restoreState(
                self.settings.value("splitter").toByteArray())
        self.restoreGeometry(
                self.settings.value("mainwindow/geometory").toByteArray())
        self.settings.endGroup()

    def saveGUISettings(self):
        self.settings.beginGroup("GUI")
        self.settings.setValue("treeWidget/header",
                QtCore.QVariant(self.treeWidget.header().saveState()))
        self.settings.setValue("splitter",
                QtCore.QVariant(self.splitter.saveState()))
        self.settings.setValue("mainwindow/geometory",
                QtCore.QVariant(self.saveGeometry()))
        self.settings.endGroup()

    def appendTab(self, tabName, postAppendingHandler=None, tabIdx=-1):
        newTreeWidget = TwitturnTimeLineWidget(self.tabWidget)
        self.tabWidget.insertTab(
                tabIdx,
                newTreeWidget,
                QtCore.QString(tabName))
        if not postAppendingHandler==None:
            self.connect(self, QtCore.SIGNAL("newPostCame"),
                    lambda x:postAppendingHandler(newTreeWidget, x))
        self.connect(
                newTreeWidget.treeWidget, 
                QtCore.SIGNAL('currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)'),
                self.setCurrentPost)
        self.connect(
                newTreeWidget.treeWidget,
                QtCore.SIGNAL('itemDoubleClicked(QTreeWidgetItem *,int)'),
                self.putReplyHeader)
        return newTreeWidget

    def selectTopPost(self):
        self.treeWidget.setCurrentItem(
            self.treeWidget.topLevelItem(0))
        self.treeWidget.setFocus()
    def focusTimeLine(self):
        self.treeWidget.setFocus()

    def setCurrentPost(self, current, previous):
        if not current:
            current = previous
        self.textBrowser.setHtml( 
                self.convertNickTextToLink( 
                    self.convetUriToLink( 
                        current.text(2))))
        nicks = [] 
        nicks = self.pickupNicks(current.text(2))
        nicks.append(current.text(1))
        self.treeWidget.parent().highlightNics(nicks)
        self.pushButtonWeb.setEnabled(not current.text(6) == "")
        self.label.setPixmap(current.icon(0).pixmap(48,48))
        self.labelNick.setText(
                current.text(1) + '/' + current.text(5))
        self.labelTime.setText(current.text(3))

    def convetUriToLink(self, str):
        if not str.contains(self.rxURL):
            return str
        retStr = QtCore.QString()
        head = self.rxURL.indexIn(str, 0)
        retStr += str.mid(0, head)
        while True:
            if head == -1:
                break;
            url = self.rxURL.cap()
            retStr += '<a href=\"' + url + '\">'+url+'</a>'
            tail = head + self.rxURL.matchedLength()

            head = self.rxURL.indexIn(str, tail)
            len = head - tail
            if len < -1:
                retStr += str.mid(tail, -1)
            else:
                retStr += str.mid(tail, len)
        return retStr

    def convertNickTextToLink(self, str):
        if not str.contains(self.rxReply):
            return str
        retStr = QtCore.QString()
        list = self.splitString(str)
        for s in list:
            if s.startsWith('@'):
                tmp = s.mid(1)
                s = '<a href=\"http://twitter.com/'+tmp+'\">@'+tmp+'</a>'
            retStr += s
        return retStr

    def splitString(self, str):
        list = []
        head = self.rxReply.indexIn(str, 0)
        list.append(str.mid(0, head))
        while True:
            if head == -1:
                break;
            list.append(self.rxReply.cap())
            tail = head + self.rxReply.matchedLength()

            head = self.rxReply.indexIn(str, tail)
            len = head - tail
            if len < -1:
                list.append(str.mid(tail, -1))
            else:
                list.append(str.mid(tail, len))
        return list

    def pickupNicks(self, str):
        list = []
        pos = 0
        while True:
            pos = self.rxReply.indexIn(str, pos)
            if pos == -1:
                break;
            list.append(self.rxReply.cap(1))
            pos += self.rxReply.matchedLength()
        return list

    def fillSamplePosts(self):
        p1 = [
                ['loader.gif','j','ha@i@jzxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['loader.gif','j','hazxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['twitter_57.png','i','ihogehoge', '2008/03/11 13:33:01', "T", 'False'],
                ['loader.gif','j','hohohoge', '2008/03/11 13:30:02', 'F', 'True'],
                ['loader.gif','j','hohohohh','2008/03/11 13:30:00', 'F', 'False'],
                ['loader.gif','j','hazxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['twitter_57.png','i','i@j@khogehoge', '2008/03/11 13:33:01', "T", 'False'],
                ['loader.gif','j','hohohoge', '2008/03/11 13:30:02', 'F', 'True'],
                ['loader.gif','j','hohohohh','2008/03/11 13:30:00', 'F', 'False'],
                ['twitter_57.png','i','ihogehoge', '2008/03/11 13:33:01', "T", 'False'],
                ['loader.gif','j','hohohoge', '2008/03/11 13:30:02', 'F', 'True'],
                ['loader.gif','j','hohohohh','2008/03/11 13:30:00', 'F', 'False'],
                ['loader.gif','j','hazxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['twitter_57.png','i','ihogehoge', '2008/03/11 13:33:01', "T", 'False'],
                ['loader.gif','j','hohohoge', '2008/03/11 13:30:02', 'F', 'True'],
                ['loader.gif','j','hohohohh','2008/03/11 13:30:00', 'F', 'False'],
                ['loader.gif','k','hohohoge', '2008/03/11 13:30:02', 'F', 'True'],
                ['loader.gif','k','hohohohh','2008/03/11 13:30:00', 'F', 'False'],
                ['loader.gif','k','@j@i @khazxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['loader.gif','k','@j@i khazxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['loader.gif','k','@j@i is @khazxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['loader.gif','k','@j@i @khazxcbzxcbzxc', '2008/03/11 13:33:59', 'F', 'False'],
                ['loader.gif','k','@j@i is @k hahttp://www.ex@mple.com/zxcbzxcbzxc', '2008/03/11 13:34:00', 'F', 'False'],
                ['loader.gif', 'M', 'hoge', '2008/03/11 13:34:00', 'F', 'False'],
                ]
        for post in p1:
            treeEntry = QtGui.QTreeWidgetItem(self.treeWidget)
            for i, v in enumerate(post):
                if i==0:
                    if not self.iconCache.has_key(v):
                        icon = QtGui.QIcon("/home/yu/work/twitturn2/images/"+v)
                        if icon.isNull():
                            print "Icon cache Failed. Should network fetch"
                        else:
                            self.iconCache[v] = icon
                    treeEntry.setIcon(0, self.iconCache[v])
                else:
                    treeEntry.setText(i,v)



def replyTabFilter(widget, post, userNick):
    if post.text(2).contains(userNick) and post.text(7) == "stat":
        widget.addPost(post, True)
        widget.setNotify(True)

def directMessageTabFilter(widget, post):
    if post.text(7) == "DM":
        widget.addPost(post, True)
        widget.setNotify(True)

if __name__=="__main__":
    app = QtGui.QApplication(sys.argv)
    form = Twitturn()
    form.appendTab(form.tr("&Replies"), 
            lambda x,y: replyTabFilter(x,y, form.userNick))
    form.appendTab(form.tr("&DM box"),
            directMessageTabFilter)
    plugins = glob.glob(os.path.join("plugins",'*.py'))
    for plugin in plugins:
        print plugin
        execfile(plugin)
    form.lineEdit.setSearchTab(form.appendTab(form.tr("&Search")))
    form.show()
    app.exec_()
