# -*- coding: utf-8 -*-

require File.join(File.dirname(__FILE__), 'spec_helper')

ForceColumns::ForceDomain.set_domain_model_name('Domain')

class CreateSamples < ActiveRecord::Migration
  def self.up
    create_table :samples do |t|
      t.string  :name
    end
  end
  def self.down
    drop_table :samples
  end
end

class CreateNotes < ActiveRecord::Migration
  def self.up
    create_table :notes, {
      :domain_id     => false,
      :created_at    => false,
      :updated_at    => false,
      :created_by_id => false,
      :updated_by_id => false,
      :created_in_id => false,
      :updated_in_id => false,
      :lock_version  => false,
    } do |t|
      t.string :name
    end
  end
  def self.down
    drop_table :notes
  end
end

describe ForceColumns do
  describe 'force_columns' do
    before do
      swap_stdout{ CreateSamples.up }
      class Sample < ActiveRecord::Base
      end
    end
    after do
      swap_stdout{ CreateSamples.down }
    end
    [
     'domain_id',
     'created_at',
     'updated_at',
     'created_by_id',
     'updated_by_id',
     'created_in_id',
     'updated_in_id',
     'lock_version',
    ].each do |column_name|
      it "have a column named #{column_name}" do
        Sample.column_names.should be_include(column_name)
      end
    end
  end
  describe 'no force_columns' do
    before do
      swap_stdout{ CreateNotes.up }
      class Note < ActiveRecord::Base
      end
    end
    after do
      swap_stdout{ CreateNotes.down }
    end
    [
     'domain_id',
     'created_at',
     'updated_at',
     'created_by_id',
     'updated_by_id',
     'created_in_id',
     'updated_in_id',
     'lock_version',
    ].each do |column_name|
      it "do not have a column named #{column_name}" do
        Note.column_names.should_not be_include(column_name)
      end
    end
  end
end

describe ForceColumns::ForceDomain do
  describe 'Sample' do
    before :all do
      swap_stdout{ CreateSamples.up }
      class Domain < ActiveRecord::Base
        def self.current_id
          @current_id
        end
        def self.current_id=(id)
          @current_id = id
        end
      end
      class Sample < ActiveRecord::Base
      end
    end
    before do
      %w[a b c].each do |name|
        Domain.create(:name => name)
      end
      ('a'..'z').each_with_index do |c, index|
        Sample.create(:name => c, :domain_id => index % 3 + 1)
      end
    end
    after :all do
      swap_stdout{ CreateSamples.down }
    end
    after do
      Sample.delete_all
      Domain.delete_all
    end
    describe "Sample" do
      it "belongs to :domain" do
        Sample.instance_methods.should be_include('domain')
      end
    end
    describe "current domain is a" do
      before do
        Domain.current_id = 1
      end
      it "find 9 records" do
        Sample.filter_by_current_domain.should have(9).founds
      end
    end
    describe "create Sample" do
      before do
        Domain.current_id = 1
        @sample = Sample.create(:name => 'aa')
      end
      it "belongs_to domain 1" do
        @sample.reload
        @sample.domain_id.should == 1
      end
    end
    describe "update Sample" do
      before do
        Domain.current_id = 1
        @sample = Sample.create(:name => 'aa')
        @sample.reload
        @sample.name = 'bb'
        @sample.save
      end
      it "name == 'bb'" do
        @sample.reload
        @sample.name.should == 'bb'
      end
      it "belongs_to domain 1" do
        @sample.reload
        @sample.domain_id.should == 1
      end
    end
    describe "update Sample, change domain" do
      before do
        Domain.current_id = 1
        @sample = Sample.create(:name => 'aa')
        @sample.reload
        @sample.domain_id = 2
        @sample.name = 'bb'
        @sample.save
      end
      it "name == 'bb'" do
        @sample.reload
        @sample.name.should == 'bb'
      end
      it "belongs_to domain 2" do
        @sample.reload
        @sample.domain_id.should == 2
      end
    end
    describe "when call set_table_name" do
      it "do not raise Error" do
        class DummySample < ActiveRecord::Base
          set_table_name 'samples'
        end
      end
    end
    describe "when call set_table_name with force_domain" do
      before :all do
        class DummySample < ActiveRecord::Base
          set_table_name 'samples'
          force_domain
        end
      end
      before do
        Domain.current_id = 1
        @sample = DummySample.create(:name => 'aa')
      end
      after do
        DummySample.delete_all
      end
      it "belongs_to domain 1" do
        @sample.reload
        @sample.domain_id.should == 1
      end
      describe "current domain is a" do
        before do
          Domain.current_id = 2
        end
        it "find 9 records" do
          DummySample.filter_by_current_domain.should have(9).founds
        end
      end
    end
  end
end
