# -*- coding: utf-8 -*-

module AssertSQLIdentifier
  # この値で識別子の長さの制限を行う。
  LENGTH_MAX = 30

  ANSI_SQL_RESERVED = %w[ADD ALL ALTER AND ANY AS ASC BETWEEN BY CHAR CHECK CONNECT CREATE CURRENT DATE DECIMAL DEFAULT DELETE DESC DISTINCT DROP ELSE FLOAT FOR FROM GRANT GROUP HAVING IMMEDIATE IN INSERT INTEGER INTERSECT INTO IS LEVEL LIKE NOT NULL OF ON OPTION OR ORDER PRIOR PRIVILEGES PUBLIC REVOKE ROWS SELECT SESSION SET SIZE SMALLINT TABLE THEN TO UNION UNIQUE UPDATE USER VALUES VARCHAR VIEW WHENEVER WITH]
  ORACLE_SQL_RESERVED = %w[ACCESS AUDIT A ABORT ACCESSED ACCOUNT ACTIVATE ADMINISTER ADMINISTRATOR ADVISE ADVISOR AFTER ALGORITHM ALIAS ALLOCATE ALLOW ANALYZE CLUSTER COLUMN COMMENT COMPRESS EXCLUSIVE EXISTS FILE IDENTIFIED INCREMENT INDEX INITIAL LOCK LONG MAXEXTENTS MINUS MLSLABEL MODE MODIFY NOAUDIT NOCOMPRESS NOWAIT NUMBER OFFLINE ONLINE PCTFREE RAW RENAME RESOURCE ROW ROWID ROWNUM SHARE START SUCCESSFUL SYNONYM SYSDATE TRIGGER UID VALIDATE VARCHAR2 WHERE] # 'ADMIN' is also reserved in fact.
  LOCAL_RESERVED = %w[APPLICATION RECORD RECORD_ID RECORD_NAME RECORD_CODE] # for record picker

  def assert_length(*names)
    names.each do |name|
      raise ArgumentError, "exceed maximum length of name: #{name}" if name.to_s.length > LENGTH_MAX
    end
  end

  def assert_unreserved(*names)
    names.each do |name|
      raise ArgumentError, "reserved identifier: #{name}" if (ANSI_SQL_RESERVED | ORACLE_SQL_RESERVED).include? name.to_s.upcase
    end
  end
end

# フレームワークに導入するテーブルに付随する識別子を検証する。
module ForceColumns
  include AssertSQLIdentifier

  def create_table(name, options = {})
    # ActiveRecordStore
    # schema_migrations since Rails 2.1.0
    if %w|sessions schema_migrations|.include?(name.to_s)
      return super
    end

    # through only while loading db/schema.rb
    if options[:force] && !options[:force_columns]
      return super
    end

    assert_length name
    assert_unreserved name
    super do |t|
      class << t
        def column(name, type, options = {})
          self.extend(AssertSQLIdentifier)
          self.assert_length name
          self.assert_unreserved name
          super(name.to_s, type, options)
        end
      end
      [
        :domain_id,
      ].each do |column_name|
        unless options[column_name] == false
          t.column column_name, :integer, :null => false
        end
      end
      yield t
      [
        :created_at,
        :updated_at,
      ].each do |column_name|
        unless options[column_name] == false
          t.column column_name, :string, :limit => 14
        end
      end
      [
        :created_by_id,
        :updated_by_id,
        :created_in_id,
        :updated_in_id,
      ].each do |column_name|
        unless options[column_name] == false
          t.column column_name, :integer
        end
      end
      [
        :lock_version,
      ].each do |column_name|
        unless options[column_name] == false
          t.column column_name, :integer, :null => false, :default => 0
        end
      end
    end
    self.add_index name, :domain_id unless options[:domain_id] == false
  end

  def add_column(table_name, column_name, type, options = {})
    assert_length table_name, column_name
    assert_unreserved table_name, column_name
    super
  end

  def drop_table(name, options = {})
    self.remove_index name, :domain_id rescue nil
    super(name, options = {})
  end

  # 同じ構成の待機用テーブルも併せて作成する。
  def create_table_with_standby(name, options = {}, &block)
    create_table(name, options, &block)
    if /\Awf_(.+)\z/ =~ name.to_s
      create_table("wfb_#{$1}", options, &block)
    else
      create_table("b_#{name}", options, &block)
    end
  end

  # 待機用テーブルも併せて削除する。
  def drop_table_with_standby(name, options = {})
    if /\Awf_(.+)\z/ =~ name.to_s
      drop_table("wfb_#{$1}", options)
    else
      drop_table("b_#{name}", options)
    end
    drop_table(name, options)
  end

  module ForceDomain
    def self.set_domain_model_name(name)
      module_eval <<-"end_eval"
        def self.current_id
          #{name}.current_id
        end
      end_eval
    end

    def self.included(base) #:nodoc:
      super
      base.extend(ClassMethods)
    end

    module ClassMethods
      def force_domain
        return unless table_exists?
        return unless column_names.include?('domain_id')
        class_eval do
          belongs_to :domain
          named_scope :filter_by_current_domain, lambda{
            current_id = ::ForceColumns::ForceDomain.current_id
            if column_names.include?("domain_id") && current_id
              { :conditions => ["#{table_name}.domain_id = ?", current_id] }
            end
          }
          callbacks = Callbacks.new
          before_create callbacks
          before_update callbacks
        end
      end
    end

    class Callbacks # :nodoc: all
      def initialize
      end

      def before_create(record)
        set_current_domain(record)
      end

      def before_update(record)
        set_current_domain(record)
      end

      private
      def set_current_domain(record)
        if record.class.column_names.include?("domain_id") && record.read_attribute("domain_id").nil?
          record.write_attribute("domain_id", ::ForceColumns::ForceDomain.current_id)
        end
      end
    end
  end
end
