# -*- coding: utf-8 -*-

from genshi.builder import tag

from trac.core import TracError
from trac.db.api import get_column_names
from trac.resource import ResourceNotFound
from trac.ticket.report import ReportModule
from trac.util.datefmt import format_datetime, format_date, format_time, parse_date
from trac.util.presentation import to_json
from trac.util.translation import _


class ReportRenderer(object):

    def __init__(self, env):
        self.env = env

    def render(self, req, params):
        u"""TracReportsの結果をHTMLで返す。オプションによってはテーブル、グラフそれぞれを表示する。
        """
        self.index = req._reportinclude

        id, vars, opts = self._parse_params(req, params)
        report = self._get_report_info(id)

        if len(report) == 1:
            title, sql, desc = report[0]

            if opts['title'] == '':
                opts['title'] = title

            table, columns, table_data = self._render_table(req, id, vars, opts, title, sql, desc)
            table = self._render_graph(req, opts, table, columns, table_data, title, desc)
            return table
        else:
            raise ResourceNotFound(
                _('Report [%(num)] does not exist.', num=id),
                _('Invalid Report Number'))

    def _render_table(self, req, id, vars, opts, title, sql, desc):
        db = self.env.get_db_cnx()

        sql, vars , missing_vars = self._sql_sub_vars(sql, vars, db)

        cursor = db.cursor()
        cursor.execute(sql, vars)

        if opts['table'] == 'hide':
            table_display = 'none'
        else:
            table_display = 'inline'

        table = tag.table(class_="listing tickets",
                          id="reportgraphtable_%d" %(self.index),
                          style="display:%s" %(table_display))
        caption = tag.caption(tag.a('%s(report:%s)' %(title, id), href='%s' % req.href.report(id)),
                              align='center')
        table.append(caption)

        columns = get_column_names(cursor)
        table_data = dict((column, []) for column in columns)

        tr = tag.tr()
        for column in columns:
            if column.startswith('_'):
                continue
            tr.append(tag.th(column))

        table.append(tag.thead(tr))
        tbody = tag.tbody()
        table.append(tbody)

        for idx, row in enumerate(cursor):

            odd_or_even = (idx % 2 == 0) and 'odd' or 'even'
            tr = tag.tr()
            css_class = ''

            for col_idx, col in enumerate(row):
                column = columns[col_idx]

                if column == '__color__' and css_class is '':
                    css_class = 'color%s-' % col
                    continue
                if column.startswith('_'):
                    continue

                converted_col = self._convert_col(req, column, col)

                tr = tr(tag.td(converted_col))
                table_data[column].append(col)

            css_class = css_class + odd_or_even
            tr = tr(class_ = css_class)
            tbody.append(tr)

        #add_stylesheet(req, 'wikitable/css/wikitable.css')
        return table, columns, table_data

    def _render_graph(self, req, opts, table, columns, table_data, title, desc):
        u"""TracReportsの結果をグラフで表示する。
        """
        if opts['graph'] not in ('lines', 'bars', 'pie'):
            return table

        ticks = table_data[columns[0]]

        data = '['
        for column in columns[1:]:
            #print table_data[column]
            values = self._to_js_array(table_data[column])
            data = data + values + ','
        data = data[:-1] + ']'

        xaxisType = self._getXAxisType(req, ticks, opts)
        if isinstance(opts['barWidth'], basestring) and opts['barWidth'].isdigit():
            opts['barWidth'] = int(opts['barWidth'])

        script = """\
(function($) {
    var options = {
        graph: '%(graph)s', seriesLabel: %(seriesLabel)s, data: %(data)s,
        ticks: %(ticks)s, title: %(title)s, stack: %(stack)s,
        legendLoc: %(legendLoc)s,
        legendXOffset: %(legendXOffset)s, legendYOffset: %(legendYOffset)s,
        xaxisType: %(xaxisType)s, xaxisMin: %(xaxisMin)s, xaxisMax: %(xaxisMax)s,
        yaxisMin: %(yaxisMin)s, yaxisMax: %(yaxisMax)s,
        xaxisFormatString: %(xaxisFormatString)s,
        yaxisFormatString: %(yaxisFormatString)s, barWidth: %(barWidth)s
    };
    if ($.fn.reportGraph) {
        $('#reportgraph_%(index)d').reportGraph(options);
    }
    else {
        window.reportgraph_data_%(index)d = options;
    }
})(jQuery);
        """ % {
            'index': self.index,
            'graph': opts['graph'],
            'seriesLabel': self._to_js_array(columns[1:]),
            'data': data,
            'ticks': self._to_js_array(ticks),
            'title': to_json(opts['title']),
            'stack': opts['stack'].lower(),
            'legendLoc': to_json(opts['legendLoc']),
            'legendXOffset': opts['legendXOffset'],
            'legendYOffset': opts['legendYOffset'],
            'xaxisType': to_json(xaxisType),
            'xaxisMin': opts['xaxisMin'],
            'xaxisMax': opts['xaxisMax'],
            'yaxisMin': opts['yaxisMin'],
            'yaxisMax': opts['yaxisMax'],
            'xaxisFormatString': to_json(opts['xaxisFormatString']),
            'yaxisFormatString': to_json(opts['yaxisFormatString']),
            'barWidth': to_json(opts['barWidth']),
        }

        div = tag.div(
                      tag.div('',
                              id="reportgraph_%d" % self.index,
                              class_='reportgraph_chart',
                              style="width:%spx;height:%spx;" %
                              (opts["width"],opts["height"])),
                      tag.br(),
                      table,
                      tag.script(script, type="text/javascript"),
                      class_="reportgraph",
                      id="reporinclude_%d" % self.index)
        return div

    def _parse_params(self, req, params):
        opts = {
               'title':'',
               'renderer':'jqplot',
               'width':'536',
               'height':'300',
               'end':'',
               'start':'',
               'per':'day',
               'graph':'',
               'table':'inline',
               'dateFormat':'yyyy-MM-dd',
               'stack':'false',
               'legendLoc':'ne',
               'legendXOffset':'12',
               'legendYOffset':'12',
               'xaxisMin':'null',
               'xaxisMax':'null',
               'yaxisMin':'null',
               'yaxisMax':'null',
               'xaxisFormatString':'',
               'yaxisFormatString':'',
               'barWidth':'50'
               }

        vars = {}
        for (index, param) in enumerate(params.split(',')):
            if index == 0:
                id = param
                id, vars = self._parse_vars(req, id)
                continue

            colon_split = param.split(':')
            key = colon_split[0].strip()
            value = ''
            if len(colon_split) > 1:
                value = ':'.join(colon_split[1:])
            else:
                value = True
            opts[key] = value
        return id, vars, opts

    def _parse_vars(self, req, id):
        vars = {}

        if id.find('?') == -1:
            return id, vars

        id_and_params = id.split('?')
        params = id_and_params[1]
        id = id_and_params[0]

        if params.find('&') != -1:
            for (index, param) in enumerate(params.split('&')):
                if param.find('=') == -1:
                    continue
                entry = param.split('=')
                vars[entry[0]] = entry[1]
                if vars[entry[0]] == '$USER':
                    vars[entry[0]] = req.authname
        elif params.find('=') != -1:
            entry = params.split('=')
            vars[entry[0]] = entry[1]
            if vars[entry[0]] == '$USER':
                vars[entry[0]] = req.authname

        return id, vars

    def _get_report_info(self, id):
        db = self.env.get_db_cnx()
        cursor = db.cursor()
        cursor.execute("SELECT title,query,description from report "
                       "WHERE id=%s", (id,))

        report = [(title, sql, desc) for title, sql, desc in cursor]
        return report

    def _sql_sub_vars(self, sql, vars, db):
        rm = ReportModule(self.env)
        sql, vars, missing_vars = rm.sql_sub_vars(sql, vars, db)
        return sql, vars, missing_vars

    def _getXAxisType(self, req, ticks, opts):
        if ticks and len(ticks) > 0:
            try:
                int(ticks[0])
                return 'number'
            except ValueError:
                pass

            try:
                parse_date(ticks[0], req.tz)
                return 'date'
            except TracError:
                pass

            return 'string'

    def _to_js_array(self, array):
        return to_json(array)

    def _convert_col(self, req, column, col):
        if column == 'time' or column.endswith(u'時刻'):
            col = col != None and format_time(int(col)) or '--'
        if column in ('date', 'created', 'modified') or column.endswith(u'日付'):
            col = col != None and format_date(int(col)) or '--'
        if column == 'datetime' or column.endswith(u'日時'):
            col = col != None and format_datetime(int(col)) or '--'

        if column.lower() in ('ticket', 'id', u'チケット'):
            col = tag.a('#' + str(col), title='View Ticket', class_='ticket',
                        href=req.href.ticket(str(col)))
        return col
