package jp.osoite.tomu.xmpp.core;

import jp.osoite.tomu.xmpp.util.XMPPSetting;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.jivesoftware.smack.*;
import org.jivesoftware.smack.packet.Presence;

/**
 * XMPPネットワークに接続するためのマネージャクラスです．
 * 本クラスは1つのXMPPセッションのみを管理します．
 * @author shima
 */
public final class XMPPManager {

    private RosterImple roster;

    private XMPPSetting setting;
    /**
     * イベントハンドラを利用する場合は本コンストラクタを利用します．
     * @param setting XMPPサービスに接続するためのパラメータ
     * @param recv メッセージを受信した際のイベントハンドラ
     */
    public XMPPManager(XMPPSetting setting, MessageReceiver recv) {
        this.setting=setting;
        roster = new RosterImple(setting, recv);
    }

    public XMPPSetting getSetting(){
        return setting;
    }

    /**
     * メッセージを送信するのみの場合は，本コンストラクタを利用してください．
     * @param setting XMPPサービスに接続するためのパラメータ
     */
    public XMPPManager(XMPPSetting setting) {
        this(setting, new MessageReceiver());
    }

    /**
     * XMPPのネットワークにログインします．
     * @return ログインに成功した場合はtrue，失敗した場合はfalse
     * @throws org.jivesoftware.smack.XMPPException 接続時にエラーが発生した場合
     */
    public boolean login() {
        return roster.login();
    }

    /**
     * XMPPネットワークからログアウトします．
     */
    public void logoff() {
        roster.logoff();
    }

    /**
     * 指定したアカウントにメッセージを送信します．
     * @param account 送信先アカウント
     * @param message 送信メッセージ
     * @return メッセージ送信に成功した場合はtrue，失敗した場合はfalse
     */
    public boolean send(String account, String message) {
        return roster.send(account, message);
    }

    /**
     * ネットワークリストに登録されているアカウントすべてにメッセージを送信します．
     * @param str 送信メッセージ
     * @return メッセージ送信に成功した場合はtrue，失敗した場合はfalse
     */
    public boolean sendToAll(String str) {
        return roster.sendToAll(str);
    }

    /**
     * TomuCoreへ新しいXMPPアカウントを追加します．
     * @param account 追加するアカウント
     */
    public void addAccount(String account) {
        roster.addAccount(account);
    }

    /**
     * 引数のXMPPアカウントを削除します．
     * @param account 削除するアカウント
     */
    public void removeAccount(String account) {
        roster.removeAccount(account);
    }

    /**
     * 全アカウントを削除します．
     */
    public void removeAllAccount() {
        roster.removeAllAccount();
    }

    /**
     * 登録されている他のXMPPユーザのリストを取得します．
     */
    public Collection<RosterEntry> getUserList() {
        return roster.getUserList();
    }

    /**
     * 引数のアカウントが接続中かどうかを返します．
     * @return 引数のアカウントが接続中ならばtrue
     */
    public boolean isAvailable(String account) {
        return roster.roster.getPresence(account).isAvailable();
    }

    private static class RosterImple implements RosterListener {

        private ChatManager chatManager;
        private XMPPConnection connection;
        private Roster roster;
        private Map<String, Chat> chatMap;
        private ConnectionConfiguration config;
        private MessageReceiver recv;
        private XMPPSetting setting;
        private volatile boolean isLoggedin;

        public RosterImple(XMPPSetting setting, MessageReceiver recv) {
            Roster.setDefaultSubscriptionMode(Roster.SubscriptionMode.accept_all);
            config = new ConnectionConfiguration(setting.getHost(), setting.getPort(), setting.getService());
            chatMap = new ConcurrentHashMap<String, Chat>();
            isLoggedin = false;
            this.recv = recv;
            this.setting = setting;
        }

        public boolean login() {
            synchronized (this) {
                if (isLoggedin) {
                    return false;
                } else {
                    isLoggedin = true;
                }
            }
            connection = new XMPPConnection(config);
            try {
                connection.connect();
                connection.login(setting.getAccount(), setting.getPassword());
            } catch (XMPPException ex) {
                return (isLoggedin = false);
            }
            chatManager = connection.getChatManager();
            roster = connection.getRoster();
            roster.addRosterListener(this);
            for (RosterEntry entry : roster.getEntries()) {
                if (!chatMap.containsKey(entry.getUser())) {
                    Chat chat = chatManager.createChat(entry.getUser(), recv);
                    chatMap.put(entry.getUser(), chat);
                }
            }
            return isLoggedin;
        }

        public void logoff() {
            if (isLoggedin) {
                connection.disconnect();
                chatMap.clear();
                isLoggedin = false;
            }
        }

        public boolean send(String account, String message) {
            if (isLoggedin) {
                if (chatMap.containsKey(account) && roster.getPresence(account).isAvailable()) {
                    Chat chat = chatMap.get(account);
                    try {
                        chat.sendMessage(WordReplacer.replaceDotAndHttp(message));
                    } catch (XMPPException ex) {
                        return false;
                    }
                    waitMilliSecond(300);
                    return true;
                }
                return false;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        public boolean sendToAll(String str) {
            if (isLoggedin) {
                boolean result = true;
                for (RosterEntry entry : roster.getEntries()) {
                    try {
                        chatMap.get(entry.getUser()).sendMessage(WordReplacer.replaceDotAndHttp(str));
                        waitMilliSecond(300);
                    } catch (Exception e) {
                        result = false;
                    }
                }
                return result;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        public boolean addAccount(String account) {
            if (isLoggedin) {
                roster.reload();
                try {
                    roster.createEntry(account, account, null);
                } catch (XMPPException ex) {
                    return false;
                }
                return true;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        public boolean removeAccount(String account) {
            if (isLoggedin) {
                roster.reload();
                RosterEntry entry = roster.getEntry(account);
                if (entry != null) {
                    try {
                        roster.removeEntry(entry);
                    } catch (XMPPException ex) {
                        return false;
                    }
                }
                return true;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        public void removeAllAccount() {
            if (isLoggedin) {
                roster.reload();
                for (RosterEntry rentry : roster.getEntries()) {
                    try {
                        roster.removeEntry(rentry);
                    } catch (XMPPException e) {
                    }
                }
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        private void waitMilliSecond(long time) {
            try {
                TimeUnit.MILLISECONDS.sleep(time);
            } catch (InterruptedException ex) {
            }
        }

        public Collection<RosterEntry> getUserList() {
            if (isLoggedin) {
                roster.reload();
                return roster.getEntries();
            } else {
                throw new NullPointerException("Does not login");
            }
        }

        @Override
        public void entriesAdded(Collection<String> addresses) {
            if (isLoggedin) {
                for (String account : addresses) {
                    if (!chatMap.containsKey(account)) {
                        Chat chat = chatManager.createChat(account, recv);
                        chatMap.put(account, chat);
                    }
                }
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        @Override
        public void entriesDeleted(Collection<String> addresses) {
            if (isLoggedin) {
                for (String account : addresses) {
                    if (chatMap.containsKey(account)) {
                        chatMap.remove(account);
                    }
                }
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        @Override
        public void entriesUpdated(Collection<String> addresses) {
        }

        @Override
        public void presenceChanged(Presence presence) {
        }
    }
}
