package jp.osoite.tomu.xmpp.core;

import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.GregorianCalendar;

import jp.osoite.tomu.core.db.DBConnectionModule;
import jp.osoite.tomu.net.xmpp.XMPPDeamonRunnable;
import jp.osoite.tomu.util.TimeConverter;
import jp.osoite.tomu.xml.jaxb.object.TomuMessage;
import jp.osoite.tomu.xml.jaxb.util.JAXBConverter;
import jp.osoite.tomu.xml.jaxb.util.builder.AppsResponseBuilder;
import jp.osoite.tomu.xml.jaxb.wrapper.Axis;
import jp.osoite.tomu.xml.jaxb.wrapper.Axis.Direction;
import jp.osoite.tomu.xml.jaxb.wrapper.Location;
import jp.osoite.tomu.xml.jaxb.wrapper.RequestSfeerWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.TScanValue;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.RequestSfeerListener;

/**
 *
 * @author shima
 */
public class RequestSfeerMinimumSQL implements RequestSfeerListener {

    private XMPPManager manager;
    private PreparedStatement selectTScanData;
    private PreparedStatement selectPedestrianData;
    private PreparedStatement selectKito2Data;
    private GregorianCalendar cal;
    private int sfeerRequestSeq = 0;
    private Axis preAxis;
    private double preTemp = 0;
    private int preRegNum = 0;
    private boolean[] compassCheck;
    private static final boolean DEBUG = false;
    private volatile static boolean isEnabledEmergency = false;
    private volatile static int emergencyLevel = 0;
    private volatile static boolean isEnabledDebug = false;

    public RequestSfeerMinimumSQL() {
        try {
            Connection con = DBConnectionModule.getInstance().getConnection();
            //selectTScanData = con.prepareStatement("SELECT * FROM data_tscan WHERE tscan_id = ? && (sensed_time BETWEEN ? AND ?) ORDER BY sensed_time DESC");
            selectTScanData = con.prepareStatement("SELECT * FROM data_tscan WHERE tscan_id = ? ORDER BY sensed_time DESC LIMIT 10");
            //selectPedestrianData = con.prepareStatement("SELECT * FROM data_ped WHERE sensorid = ? && (timestamp BETWEEN ? AND ?) ORDER BY timestamp DESC");
            selectPedestrianData = con.prepareStatement("SELECT * FROM data_ped WHERE sensorid = ? ORDER BY timestamp DESC LIMIT 1");
            //selectKito2Data = con.prepareStatement("SELECT * FROM data_kito2 WHERE timestamp BETWEEN ? AND ? ORDER BY timestamp DESC LIMIT 50");
            selectKito2Data = con.prepareStatement("SELECT * FROM data_kito2 ORDER BY timestamp DESC LIMIT 100");
            cal = new GregorianCalendar();
            preAxis = new Axis();
            preAxis.setValue(0, 0, 0, Direction.PLUS);
            preAxis.setValue(0, 0, 0, Direction.MINUS);
        } catch (SQLException ex) {
            if (DEBUG) {
                ex.printStackTrace();
            }
        }
    }

    public void receiveRequestSfeer(RequestSfeerWrapper wrapper) {
        AppsResponseBuilder builder = new AppsResponseBuilder(wrapper.getSourceId(), "tomuxmpp2@gmail.com", wrapper.getMessageId(), sfeerRequestSeq++);
        long[] ids = wrapper.getTemperatureSensorIdList();
        for (long id : ids) {
            if (DEBUG) {
                System.out.println("Temperature ID : " + id);
            }
            searchTScanSensor(id, builder);
            if (DEBUG) {
                System.out.println("-----------");
            }
        }
        if (DEBUG) {
            System.out.println("===============");
        }

        ids = wrapper.getPedestrianSensorIdList();
        for (long id : ids) {
            if (DEBUG) {
                System.out.println("Pedestrian ID : " + id);
            }
            searchPedestrianSensor(id, builder);
            if (DEBUG) {
                System.out.println("-----------");
            }
        }
        if (DEBUG) {
            System.out.println("===============");
        }
        Location[] locs = wrapper.getLocationList();
        for (Location loc : locs) {
            if (DEBUG) {
                System.out.println("Kito2 Location ID : " + loc.getAreaId());
            }
            searchKito2Sensor(loc, builder);
            if (DEBUG) {
                System.out.println("-----------");
            }
        }

        TomuMessage msg = builder.build();
        sendMessage(msg);
    }

    private void searchTScanSensor(long id, AppsResponseBuilder builder) {
        //setTimespanCalendar();
        long currentTime = System.currentTimeMillis();
        try {
            selectTScanData.setString(1, InitRequestSfeerMinimumSQL.getSensorIdMap().get(id));
            //selectTScanData.setTimestamp(2, new Timestamp(cal.getTimeInMillis()));
            //selectTScanData.setTimestamp(3, new Timestamp(currentTime));
            ResultSet resultSet = selectTScanData.executeQuery();
            while (resultSet.next()) {
                TScanValue val = new TScanValue(
                        resultSet.getString("tscan_id"),
                        resultSet.getTimestamp("sensed_time").getTime(),
                        resultSet.getDouble("temperature"),
                        resultSet.getDouble("humidity"),
                        resultSet.getInt("lqi"),
                        resultSet.getInt("voltage"));
                if (DEBUG) {
                    System.out.println("* Add TScan = ID : " + id + " Temp : " + val.getTemperature() + " Time : " + TimeConverter.convertToString(val.getSensedTime()));
                }
                preTemp = val.getTemperature();
                if (preTemp > 60) {
                    continue;
                }
                currentTime = val.getSensedTime();
                break;
            }
            builder.addTemperatureSensor(id, preTemp, currentTime);
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
    }

    private void searchPedestrianSensor(long id, AppsResponseBuilder builder) {
        //setTimespanCalendar();
        long currentTime = System.currentTimeMillis();
        try {
            selectPedestrianData.setLong(1, id);
            //selectPedestrianData.setTimestamp(2, new Timestamp(cal.getTimeInMillis()));
            //selectPedestrianData.setTimestamp(3, new Timestamp(currentTime));
            ResultSet resultSet = selectPedestrianData.executeQuery();
            if (resultSet.next()) {
                if (DEBUG) {
                    System.out.println("Search Ped : " + id);
                }
                preAxis = new Axis();
                preAxis.setValue(resultSet.getInt("chief_plus_compass"),
                        resultSet.getInt("chief_plus_num"),
                        resultSet.getDouble("chief_plus_speed"),
                        Direction.PLUS);
                preAxis.setValue(resultSet.getInt("chief_minus_compass"),
                        resultSet.getInt("chief_minus_num"),
                        resultSet.getDouble("chief_minus_speed"),
                        Direction.MINUS);

                currentTime = resultSet.getTimestamp("timestamp").getTime();
                preRegNum = resultSet.getInt("residence_num");
                if (DEBUG) {
                    System.out.println("* Add Ped = ID : " + id + " Axis : " + preAxis + " Time : " + TimeConverter.convertToString(resultSet.getTimestamp("timestamp").getTime()));
                    System.out.println(" Resi : " + preRegNum);
                    System.out.println(" Axis Plus : " + preAxis.getCompass(Direction.PLUS) + " " + preAxis.getFlowValue(Direction.PLUS) + " " + preAxis.getAverageSpeed(Direction.PLUS));
                    System.out.println(" Axis Minus : " + preAxis.getCompass(Direction.MINUS) + " " + preAxis.getFlowValue(Direction.MINUS) + " " + preAxis.getAverageSpeed(Direction.MINUS));
                }
            }
            builder.addPedestrianSensor(id, currentTime, 60, preRegNum, preAxis);
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
    }

    public static void setSfeerEmergency(boolean flag, int level) {
        isEnabledEmergency = flag;
        emergencyLevel = level;
    }

    public static void setPhotoSearchDebugMode(boolean flag) {
        isEnabledDebug = flag;
    }

    private void searchKito2Sensor(Location loc, AppsResponseBuilder builder) {
        if (isEnabledEmergency && emergencyLevel == 3) {
            return;
        }
        //setTimespanCalendar();
        if (isEnabledEmergency && emergencyLevel == 1) {
            compassCheck = new boolean[]{false, true, false, true, false, true, false, true};
        } else {
            compassCheck = new boolean[]{false, false, false, false, false, false, false, false};
        }
        try {
            //selectKito2Data.setTimestamp(1, new Timestamp(cal.getTimeInMillis()));
            //selectKito2Data.setTimestamp(2, new Timestamp(currentTime));
            ResultSet resultSet = selectKito2Data.executeQuery();
            ArrayList<Kito2TempData> list = new ArrayList<Kito2TempData>();
            while (resultSet.next()) {
                Kito2TempData data = new Kito2TempData(String.valueOf(loc.getAreaId()),
                        resultSet.getTimestamp("timestamp").getTime(),
                        resultSet.getInt("compass"),
                        resultSet.getDouble("latitude"),
                        resultSet.getDouble("longitude"),
                        new URL(resultSet.getString("url")),
                        new URL(resultSet.getString("thumbnail_url")),
                        loc);
                list.add(data);
                if (DEBUG) {
                    System.out.println("* Add Kito2TempData = AreaID : " + data.id + " Time : " + TimeConverter.convertToString(data.time));
                }
            }
            if (!isEnabledDebug) {
                Collections.sort(list);
            }
            if (DEBUG) {
                System.out.println("List Size : " + list.size());
            }
            if (isEnabledEmergency && emergencyLevel == 2) {
                if (!list.isEmpty()) {
                    Kito2TempData data = list.get(0);
                    builder.addImageSensor(data.id, data.time, data.comp, data.lat, data.lon, data.url, data.thumb);
                    if (DEBUG) {
                        System.out.println(" >>>> Add Kito2 : comp " + calcCompass(data.comp) + " time : " + TimeConverter.convertToString(data.time));
                    }
                }
            } else {
                for (Kito2TempData data : list) {
                    if (isEnabledDebug || data.dist <= loc.getRadius()) {
                        int compNum = calcCompass(data.comp);
                        if (!compassCheck[compNum]) {
                            compassCheck[compNum] = true;
                            builder.addImageSensor(data.id, data.time, data.comp, data.lat, data.lon, data.url, data.thumb);
                            if (DEBUG) {
                                System.out.println(" >>>> Add Kito2 : comp " + calcCompass(data.comp) + " time : " + TimeConverter.convertToString(data.time));
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
            }
        }
    }

    private void setTimespanCalendar() {
        cal.setTimeInMillis(System.currentTimeMillis());
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
    }

    private int calcCompass(int compass) {
        if (DEBUG) {
            System.out.print("* Compass : ");
            for (boolean flag : compassCheck) {
                System.out.print(" " + flag);
            }
            System.out.println();
        }
        if (0 <= compass && compass < 15) {
            return 0;
        } else if (15 <= compass && compass < 75) {
            return 1;
        } else if (75 <= compass && compass < 105) {
            return 2;
        } else if (105 <= compass && compass < 165) {
            return 3;
        } else if (165 <= compass && compass < 195) {
            return 4;
        } else if (196 <= compass && compass < 255) {
            return 5;
        } else if (255 <= compass && compass < 285) {
            return 6;
        } else if (285 <= compass && compass < 345) {
            return 7;
        } else {
            return 0;
        }
    }

    private void sendMessage(TomuMessage msg) {
        try {
            if (manager == null) {
                manager = XMPPManagersFactory.getInstance().getXMPPManager(XMPPDeamonRunnable.DEF_TOMU_NUM);
            }
            String xml = JAXBConverter.convertToXml(msg);
            manager.send(msg.getHead().getMsgDest(), xml);
            if (DEBUG) {
                System.out.println("> Send Message : " + msg.getHead().getMsgDest());
            }
        } catch (Exception e) {
            if (DEBUG) {
                e.printStackTrace();
                System.out.println("SfeerRequest sending message: ERR " + msg.getHead().getMsgDest());
            }
        }
    }

    private class Kito2TempData implements Comparable<Kito2TempData> {

        String id;
        long time;
        int comp;
        double lat;
        double lon;
        URL url;
        URL thumb;
        Location loc;
        double dist;

        Kito2TempData(String id, long time, int comp, double lat, double lon, URL url, URL thumb, Location loc) {
            this.id = id;
            this.time = time;
            this.comp = comp;
            this.lat = lat;
            this.lon = lon;
            this.url = url;
            this.thumb = thumb;
            this.loc = loc;
            this.dist = CoordsCalculater.calcDistHubeny(lat, lon, loc.getLat(), loc.getLon());
        }

        public int compareTo(Kito2TempData o) {
            double compDist = CoordsCalculater.calcDistHubeny(o.lat, o.lon, loc.getLat(), loc.getLon());
            if (dist > compDist) {
                return 1;
            } else if (dist < compDist) {
                return -1;
            } else {
                return 0;
            }
        }
    }
}
