package jp.osoite.tomu.xml.jaxb.util.builder;

import jp.osoite.tomu.xml.jaxb.util.*;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.List;
import jp.osoite.tomu.xml.jaxb.object.*;

/**
 * JAXBオブジェクトを構築するクラスです．<br>
 * 各種セッターメソッドを利用してJAXBオブジェクトを構築します．
 * 本クラスですべてのTomuMessageを生成可能ですが，各メッセージに対応した，汎用Builderも用意されています．
 * @author shima
 */
public final class JAXBConstructor {

    private TomuMessage tomuMsg;
    private boolean hasHead = false;
    private Head head;
    private boolean hasResponse = false;
    private Response resp;
    private SensorDataSet sensorDataSet;
    private boolean hasRequest = false;
    private Request req;
    private SensorSourceDataRequest ssdReq;
    private boolean hasUpload = false;
    private Upload upload;
    private SensorSourceDataSet ssdSet;
    private boolean hasSeq = false;
    private InitResponse initResp;
    private boolean hasInitResp = false;
    private InitRequest initReq;
    private boolean hasInitReq = false;

    /**
     * コンストラクタ．
     * 作成するTomuMessageのメッセージタイプを引数にとります．
     * メッセージタイプごとに呼び出せるメソッドが異なります．
     * @param type TomuMessageType
     */
    public JAXBConstructor(TomuMessageType type) {
        tomuMsg = new TomuMessage();
        head = new Head();
        head.setMsgType(type.toString());
        tomuMsg.setHead(head);
        initMsgType(type);
    }

    private void initMsgType(TomuMessageType type) {
        switch (type) {
            case APPS_RESPONSE:
                resp = new Response();
                sensorDataSet = new SensorDataSet();
                resp.setSensorDataSet(sensorDataSet);
                tomuMsg.setResponse(resp);
                hasResponse = true;
                return;
            case APPS_REQUEST:
                req = new Request();
                tomuMsg.setRequest(req);
                hasRequest = true;
                return;
            case SOURCE_UPLOAD:
                upload = new Upload();
                ssdSet = new SensorSourceDataSet();
                upload.setSensorSourceDataSet(ssdSet);
                tomuMsg.setUpload(upload);
                hasUpload = true;
                return;
            case SOURCE_REQUEST:
                req = new Request();
                ssdReq = new SensorSourceDataRequest();
                req.setSensorSourceDataRequest(ssdReq);
                tomuMsg.setRequest(req);
                hasRequest = true;
                return;
            case APPS_INIT_REQUEST:
                initReq = new InitRequest();
                tomuMsg.setInitRequest(initReq);
                hasInitReq = true;
                return;
            case APPS_INIT_RESPONSE:
                initResp = new InitResponse();
                tomuMsg.setInitResponse(initResp);
                hasInitResp = true;
                return;
            case CORE_REQUEST:
                throw new IllegalArgumentException("CORE_REQUEST doesn't be implemented.");
            case CORE_RESPONSE:
                throw new IllegalArgumentException("CORE_RESPONSE doesn't be implemented.");
            case CORE_MANAGEMENT:
                throw new IllegalArgumentException("CORE_MANAGEMENT doesn't be implemented.");
        }
    }

    /**
     * 設定したTomuMessageオブジェクトを取得します．
     * @return TomuMessageオブジェクト
     * @throws jp.osoite.tomu.xml.jaxb.util.InitializeException 呼び出し必須のメソッドを呼び出していない場合
     */
    public TomuMessage build() throws InitializeException {
        if (!hasHead) {
            throw new InitializeException("Please do setHeader()");
        }
        if (!hasInitResp && !hasInitReq && !hasSeq) {
            throw new InitializeException("Please do setSequence()");
        }
        return tomuMsg;
    }

    /**
     * TomuMessageのヘッダーをセットします．
     * <br/><strong>Required</strong>
     * @param dest 送信先アドレス
     * @param src 発信元アドレス
     * @param msgId メッセージID
     */
    public void setHeader(String dest, String src, long msgId) {
        hasHead = true;
        head.setMsgId(BigInteger.valueOf(msgId));
        head.setMsgDest(dest);
        head.setMsgSrc(src);
    }

    /**
     * 各メッセージのシーケンス番号を設定します．
     * <br/><strong>Required</strong>
     * @param seq メッセージのシーケンス番号
     */
    public void setSequence(long seq) {
        if (hasResponse) {
            sensorDataSet.setSeq(BigInteger.valueOf(seq));
        } else if (hasUpload) {
            upload.setSeq(BigInteger.valueOf(seq));
        } else if (hasRequest) {
            req.setSeq(BigInteger.valueOf(seq));
        }
        hasSeq = true;
    }

    /**
     * センサのセンシング頻度をセットします．
     * <br/><strong>SOURCE_UPLOADのみ</strong>
     * @param frq
     */
    public void setFrequency(long frq) {
        if (hasUpload) {
            Frequency freq = new Frequency();
            freq.setSec(BigInteger.valueOf(frq));
            ssdSet.setFrequency(freq);
        }
    }

    //==========================================================================
    // StaticSensorData Constructor
    /**
     * 温度センサデータを追加します．
     * <br/><strong>SOURCE_UPLOAD / APPS_RESPONSE</strong>
     * @param val 温度
     * @param sensedTime センシング時刻
     * @param sensorId センサID
     */
    public void addTemperatureData(double val, long sensedTime, long sensorId) {
        StaticSensorData ssd = new StaticSensorData();
        ssd.setType(SensorType.TEMP.toString());
        ssd.setSensedTime(BigInteger.valueOf(sensedTime));
        ssd.setStaticSensorID(BigInteger.valueOf(sensorId));
        Value value = new Value();
        value.setTempValue(BigDecimal.valueOf(val));
        ssd.setValue(value);
        addStaticSensor(ssd);
    }

    /**
     * CO2センサデータを追加します．
     * <br/><strong>SOURCE_UPLOAD / APPS_RESPONSE</strong>
     * @param val CO2濃度
     * @param sensedTime センシング時刻
     * @param sensorId センサID
     */
    public void addCo2Data(double val, long sensedTime, long sensorId) {
        StaticSensorData ssd = new StaticSensorData();
        ssd.setType(SensorType.CO2.toString());
        ssd.setSensedTime(BigInteger.valueOf(sensedTime));
        ssd.setStaticSensorID(BigInteger.valueOf(sensorId));
        Value value = new Value();
        value.setCO2Value(BigDecimal.valueOf(val));
        ssd.setValue(value);
        addStaticSensor(ssd);
    }

    /**
     * 湿度センサデータを追加します．
     * <br/><strong>SOURCE_UPLOAD / APPS_RESPONSE</strong>
     * @param val 湿度
     * @param sensedTime センシング時刻
     * @param sensorId センサID
     */
    public void addHumidityData(double val, long sensedTime, long sensorId) {
        StaticSensorData ssd = new StaticSensorData();
        ssd.setType(SensorType.HUMIDITY.toString());
        ssd.setSensedTime(BigInteger.valueOf(sensedTime));
        ssd.setStaticSensorID(BigInteger.valueOf(sensorId));
        Value value = new Value();
        value.setHumValue(BigDecimal.valueOf(val));
        ssd.setValue(value);
        addStaticSensor(ssd);
    }

    /**
     * 人流センサデータを追加します．
     * <br/><strong>SOURCE_UPLOAD / APPS_RESPONSE</strong>
     * @param measurementTime センシング時間間隔
     * @param residence 滞留人数
     * @param axis 人流
     * @param sensedTime センシング時刻
     * @param sensorId センサID
     */
    public void addStreetPedestrianData(long measurementTime, int residence, ChiefAxis axis, long sensedTime, long sensorId) {
        StaticSensorData ssd = new StaticSensorData();
        ssd.setType(SensorType.PEDESTRIAN.toString());
        ssd.setSensedTime(BigInteger.valueOf(sensedTime));
        ssd.setStaticSensorID(BigInteger.valueOf(sensorId));
        Value value = new Value();
        StreetPedestrianFlowValue spfv = new StreetPedestrianFlowValue();
        spfv.setMeasurementTimeSec(BigInteger.valueOf(measurementTime));
        spfv.setResidence(BigInteger.valueOf(residence));
        spfv.setChiefAxis(axis);
        value.setStreetPedestrianFlowValue(spfv);
        ssd.setValue(value);
        addStaticSensor(ssd);
    }

    /**
     * 人流センサデータを追加します．
     * <br/><strong>SOURCE_UPLOAD / APPS_RESPONSE</strong>
     * @param measurementTime センシング時間間隔
     * @param residence 滞留人数
     * @param chiefAxis 人流
     * @param subAxis 人流
     * @param sensedTime センシング時刻
     * @param sensorId センサID
     */
    public void addInterPedestrianData(long measurementTime, int residence, ChiefAxis chiefAxis, SubAxis subAxis, long sensedTime, long sensorId) {
        StaticSensorData ssd = new StaticSensorData();
        ssd.setType(SensorType.INTER_PEDESTRIAN.toString());
        ssd.setSensedTime(BigInteger.valueOf(sensedTime));
        ssd.setStaticSensorID(BigInteger.valueOf(sensorId));
        Value value = new Value();
        StreetPedestrianFlowValue spfv = new StreetPedestrianFlowValue();
        spfv.setMeasurementTimeSec(BigInteger.valueOf(measurementTime));
        spfv.setResidence(BigInteger.valueOf(residence));
        spfv.setChiefAxis(chiefAxis);
        spfv.setSubAxis(subAxis);
        value.setStreetPedestrianFlowValue(spfv);
        ssd.setValue(value);
        addStaticSensor(ssd);
    }

    private void addStaticSensor(StaticSensorData ssd) {
        if (hasResponse) {
            List<StaticSensorData> list = sensorDataSet.getStaticSensorData();
            list.add(ssd);
        } else if (hasUpload) {
            List<StaticSensorData> list = ssdSet.getStaticSensorData();
            list.add(ssd);
        }
    }

    /**
     * 主方向の人流データオブジェクトを取得します．
     * @param plusCompass 方位(360度)
     * @param plusNum 人数
     * @param plusAvgSp 人流の平均速度
     * @param minCompass 方位(360度)
     * @param minNum 人数
     * @param minAvgSp 人流の平均速度
     * @return ChiefAxisオブジェクト
     */
    public static ChiefAxis getChief(
            int plusCompass, int plusNum, double plusAvgSp,
            int minCompass, int minNum, double minAvgSp) {
        ChiefAxis axis = new ChiefAxis();
        axis.setPlus(getPlus(plusCompass, plusNum, plusAvgSp));
        axis.setMinus(getMinus(minCompass, minNum, minAvgSp));
        return axis;
    }

    /**
     * 副方向の人流データオブジェクトを取得します．
     * @param plusCompass 方位(360度)
     * @param plusNum 人数
     * @param plusAvgSp 人流の平均速度
     * @param minCompass 方位(360度)
     * @param minNum 人数
     * @param minAvgSp 人流の平均速度
     * @return ChiefAxisオブジェクト
     */
    public static SubAxis getSub(
            int plusCompass, int plusNum, double plusAvgSp,
            int minCompass, int minNum, double minAvgSp) {
        SubAxis axis = new SubAxis();
        axis.setPlus(getPlus(plusCompass, plusNum, plusAvgSp));
        axis.setMinus(getMinus(minCompass, minNum, minAvgSp));
        return axis;
    }

    private static Plus getPlus(int plusCompass, int plusNum, double plusAvgSp) {
        Plus plus = new Plus();
        plus.setCompass(BigInteger.valueOf(plusCompass));
        plus.setNum(BigInteger.valueOf(plusNum));
        AverageSpeed pAvg = new AverageSpeed();
        pAvg.setMeterperSecond(BigDecimal.valueOf(plusAvgSp));
        plus.setAverageSpeed(pAvg);
        return plus;
    }

    private static Minus getMinus(int minCompass, int minNum, double minAvgSp) {
        Minus minus = new Minus();
        minus.setCompass(BigInteger.valueOf(minCompass));
        minus.setNum(BigInteger.valueOf(minNum));
        AverageSpeed mAvg = new AverageSpeed();
        mAvg.setMeterperSecond(BigDecimal.valueOf(minAvgSp));
        minus.setAverageSpeed(mAvg);
        return minus;
    }

    //==========================================================================
    // DynamicSensorData Constructor
    /**
     * 画像センサ(カメラ)データを追加します．
     * <br/><strong>SOURCE_UPLOAD / APPS_RESPONSE</strong>
     * @param sensorName センサ名
     * @param dir 方位(360度)
     * @param loc 緯度経度
     * @param imageUrl 画像のURL
     * @param thumbnail サムネイル画像のURL
     * @param sensedTime センシング時刻
     */
    public void addImageData(String sensorName, int dir, Location loc, String imageUrl, String thumbnail, long sensedTime) {
        DynamicSensorData dsd = new DynamicSensorData();
        dsd.setType(SensorType.IMAGE.toString());
        dsd.setSensedTime(BigInteger.valueOf(sensedTime));
        dsd.setSensorName(sensorName);
        dsd.setDirection(BigInteger.valueOf(dir));
        dsd.setLocation(loc);
        dsd.setUrlValue(imageUrl);
        dsd.setThumbnailUrlValue(thumbnail);
        if (hasResponse) {
            List<DynamicSensorData> list = sensorDataSet.getDynamicSensorData();
            list.add(dsd);
        } else if (hasUpload) {
            List<DynamicSensorData> list = ssdSet.getDynamicSensorData();
            list.add(dsd);
        }
    }

    /**
     * 緯度経度，半径の位置情報を持ったLocationオブジェクトを取得します．
     * @param lon 緯度
     * @param lat 経度
     * @param radius 半径
     * @return Locationオブジェクト
     */
    public static Location getLocation(double lon, double lat, int radius) {
        Location loc = new Location();
        loc.setLon(BigDecimal.valueOf(lon));
        loc.setLat(BigDecimal.valueOf(lat));
        loc.setRadius(BigInteger.valueOf(radius));
        return loc;
    }

    //==========================================================================
    // KitokitoSensorData Constructor
    /**
     * きときとセンサ(カメラ)データを追加します．
     * <br/><strong>SOURCE_UPLOAD / APPS_RESPONSE</strong>
     * @param sensorID センサID
     * @param dir 方位(360度)
     * @param loc 緯度経度
     * @param dpi 解像度
     * @param imageUrl 画像のURL
     * @param thumbnail サムネイル画像のURL
     * @param sensedTime センシング時刻
     */
    public void addKitokitoData(long sensorID, int dir, Location loc, int dpi, String imageUrl, String thumbnail, long sensedTime) {
        if (hasUpload) {
            KitokitoSensorData dsd = new KitokitoSensorData();
            dsd.setType(SensorType.IMAGE.toString());
            dsd.setSensedTime(BigInteger.valueOf(sensedTime));
            dsd.setSensorID(BigInteger.valueOf(sensorID));
            dsd.setDirection(BigInteger.valueOf(dir));
            dsd.setLocation(loc);
            dsd.setDpi(BigInteger.valueOf(dpi));
            dsd.setImageUrl(imageUrl);
            dsd.setThumbnailUrl(thumbnail);
            List<KitokitoSensorData> list = ssdSet.getKitokitoSensorData();
            list.add(dsd);
        }
    }
    //==========================================================================

    //==========================================================================
    // SfeerServerRequest Constructor
    /**
     * ＜SfeerServerRequest＞<br>
     * SfeerServerがTomuCoreへ問い合わせる際のメッセージを構築します．
     * 各々のパラメータはnullを取ることも可能です．
     * @param tSet 温度センサIDリスト
     * @param pSet 人流センサIDリスト
     * @param iSet 画像位置情報リスト
     */
    public void addSfeerServerRequestData(TemperatureSet tSet, PedestrianSet pSet, ImageSet iSet) {
        if (hasRequest) {
            SfeerRequestSet sfeerReq = new SfeerRequestSet();
            if (tSet != null) {
                sfeerReq.setTemperatureSet(tSet);
            }
            if (pSet != null) {
                sfeerReq.setPedestrianSet(pSet);
            }
            if (iSet != null) {
                sfeerReq.setImageSet(iSet);
            }
            req.setSfeerRequestSet(sfeerReq);
        }
    }

    /**
     * ＜SfeerServerRequest＞<br>温度センサのIDリストを取得します．
     * @param ids IDのlong配列
     * @return　TemperatureSet
     */
    public static TemperatureSet getTemperatureSet(List<Long> ids) {
        TemperatureSet result = new TemperatureSet();
        List<BigInteger> list = result.getStaticSensorID();
        for (long id : ids) {
            list.add(BigInteger.valueOf(id));
        }
        return result;
    }

    /**
     * ＜SfeerServerRequest＞<br>人流センサのIDリストを取得します．
     * @param ids IDのlong配列
     * @return　TemperatureSet
     */
    public static PedestrianSet getPedestrianSet(List<Long> ids) {
        PedestrianSet result = new PedestrianSet();
        List<BigInteger> list = result.getStaticSensorID();
        for (long id : ids) {
            list.add(BigInteger.valueOf(id));
        }
        return result;
    }

    /**
     * ＜SfeerServerRequest＞<br>画像要求用のリストを返します．
     * @param imageReqs 範囲リスト
     * @return ImageSet
     */
    public static ImageSet getImageSet(List<Location> locationList) {
        ImageSet result = new ImageSet();
        List<Location> list = result.getLocation();
        for (Location loc : locationList) {
            list.add(loc);
        }
        return result;
    }
    //==========================================================================

    //==========================================================================
    // SfeerServerInitMessage Constructor
    /**
     * ＜SfeerServerInitRequest＞<br>
     * Sfeerサーバが初期化を行う際に投げるメッセージをセットします．
     * @param areas エリア情報
     */
    public void addSfeerInitMessage(List<Area> areas) {
        if (hasInitReq || hasInitResp) {
            Sfeer sfeer = new Sfeer();
            AreaSet areaSet = new AreaSet();
            List<Area> list = areaSet.getArea();
            for (Area area : areas) {
                list.add(area);
            }
            sfeer.setAreaSet(areaSet);
            if (hasInitReq) {
                initReq.setSfeer(sfeer);
            } else if (hasInitResp) {
                initResp.setSfeer(sfeer);
            }
        }
    }

    /**
     * ＜SfeerServerInitRequest＞
     * @param id エリアID
     * @param loc 緯度経度，範囲
     * @param tSpan 時間間隔
     * @return
     */
    public static Area getRequestArea(int id, Location loc, TimeSpan tSpan) {
        Area result = new Area();
        result.setAreaId(BigInteger.valueOf(id));
        result.setLocation(loc);
        result.setTimeSpan(tSpan);
        return result;
    }

    /**
     * ＜SfeerServerInitResponse＞
     * @param id エリアID
     * @param initTemp 温度データ
     * @param initPed 人流データ
     * @return Areaオブジェクト
     */
    public static Area getResponseArea(int id, InitTemperature initTemp, InitPedestrian initPed) {
        Area result = new Area();
        result.setAreaId(BigInteger.valueOf(id));
        result.setInitTemperature(initTemp);
        result.setInitPedestrian(initPed);
        return result;
    }

    /**
     * ＜SfeerServerInitResponse＞<br>
     * 人流センサのIDと，現在のステータスを持ったInitPedestrianオブジェクトを生成します．
     * @param sensorId センサID
     * @param crrFlow 現在の人流
     * @param crrReg 現在の滞留人数
     * @param flowAvg 平均人流
     * @return InitPedestrianオブジェクト
     */
    public static InitPedestrian getInitPedestrian(long sensorId, int crrFlow, int crrReg, double flowAvg) {
        InitPedestrian result = new InitPedestrian();
        result.setStaticSensorID(BigInteger.valueOf(sensorId));
        result.setCurrentFlowValue(BigInteger.valueOf(crrFlow));
        result.setCurrentResidentValue(BigInteger.valueOf(crrReg));
        result.setAvgFlowValue(BigDecimal.valueOf(flowAvg));
        return result;
    }

    /**
     * ＜SfeerServerInitResponse＞<br>
     * 温度センサのIDと，現在のステータスを持ったInitTemperatureオブジェクトを生成します．
     * @param sensorId センサID
     * @param max 最高気温
     * @param min 最低気温
     * @param avg 平均気温
     * @param data 現在から5分間分の気温
     * @return InitTemperatureオブジェクト
     */
    public static InitTemperature getInitTemperature(long sensorId, double crr, double max, double min, double avg, FiveMinuteTempData data) {
        InitTemperature result = new InitTemperature();
        result.setStaticSensorID(BigInteger.valueOf(sensorId));
        result.setCurrentTempValue(BigDecimal.valueOf(crr));
        result.setMaxTempValue(BigDecimal.valueOf(max));
        result.setMinTempValue(BigDecimal.valueOf(min));
        result.setAvgTempValue(BigDecimal.valueOf(avg));
        result.setFiveMinuteTempData(data);
        return result;
    }

    public static FiveMinuteTempData getFiveMinuteTempData(double[] arry) {
        FiveMinuteTempData result = new FiveMinuteTempData();
        List<BigDecimal> list = result.getTempValue();
        for (double temp : arry) {
            list.add(BigDecimal.valueOf(temp));
        }
        return result;
    }

    /**
     * 時間範囲を表すTimeSpanオブジェクトを取得します．
     * @param begin 開始時刻
     * @param end 終了時刻
     * @return TimeSpanオブジェクト
     */
    public static TimeSpan getTimeSpan(long begin, long end) {
        TimeSpan result = new TimeSpan();
        result.setBegin(BigInteger.valueOf(begin));
        result.setEnd(BigInteger.valueOf(end));
        return result;
    }
    //==========================================================================

    //==========================================================================
    // TScanSensorData Constructor
    /**
     * TScanデータを追加します．
     * <br/><strong>SOURCE_UPLOAD</strong>
     * @param sensorId TScanセンサID
     * @param sensedTime センシング時刻
     * @param temp 温度（℃）
     * @param hum 湿度（％）
     * @param lqi リンククオリティ
     * @param vol 電力残量
     */
    public void addTScanData(String sensorId, long sensedTime, double temp, double hum, int lqi, int vol) {
        if (hasUpload) {
            TScanSensorData tscan = new TScanSensorData();
            tscan.setTScanID(sensorId);
            tscan.setSensedTime(BigInteger.valueOf(sensedTime));
            tscan.setTempValue(BigDecimal.valueOf(temp));
            tscan.setHumValue(BigDecimal.valueOf(hum));
            tscan.setLqi(BigInteger.valueOf(lqi));
            tscan.setVoltage(BigInteger.valueOf(vol));
            List<TScanSensorData> list = ssdSet.getTScanSensorData();
            list.add(tscan);
        }
    }
    //==========================================================================
}
