package jp.osoite.tomu.sensor.tscan;

import jp.osoite.tomu.xml.jaxb.util.SensorType;
import jp.osoite.tomu.util.TimeConverter;

/**
 *
 * @author shima
 */
public class SockTScanValue {

    private static final int ROUTER_ID_LOWER = 2;
    private static final int ROUTER_ID_UPPER = 4;
    private static final int LEAF_ID_LOWER = 6;
    private static final int LEAF_ID_UPPER = 8;
    private static final int RESERVE_AREA = 10;
    private static final int LQI = 12;
    private static final int VOLTAGE = 14;
    private static final int TEMPERATURE_LOWER = 16;
    private static final int TEMPERATURE_UPPER = 18;
    private static final int HUMIDITY_LOWER = 20;
    private static final int HUMIDITY_UPPER = 22;
    private String dataLine,  routerID,  leafID,  reserveArea;
    private int lqi,  voltage;
    private final int hashValue;
    private double temperature;
    private double humidity;
    private long time;
    private long arrivedTime;

    public SockTScanValue(String dataLine) throws IllegalArgumentException {
        this.dataLine = dataLine;
        int tIndex = dataLine.indexOf("T");
        String tmp = (tIndex > -1) ? dataLine.substring(0, tIndex) : dataLine;
        compile(dataLine);
        if (tIndex > -1) {
            time = TimeConverter.convertToLong(dataLine.substring(tIndex + 1));
        } else {
            time = System.currentTimeMillis();
        }
        hashValue = this.toString().hashCode();
        arrivedTime = System.currentTimeMillis();
    }

    public SensorType getType() {
        return SensorType.TEMP;
    }

    public long getMeasurementTime() {
        return arrivedTime;
    }

    public long getSensorID() {
        return System.currentTimeMillis();
    }

    private void compile(String dataLine) throws IllegalArgumentException {
        try {
            routerID = getData(dataLine, ROUTER_ID_UPPER);
            routerID = routerID + getData(dataLine, ROUTER_ID_LOWER);

            leafID = getData(dataLine, LEAF_ID_UPPER);
            leafID = leafID + getData(dataLine, LEAF_ID_LOWER);

            reserveArea = getData(dataLine, RESERVE_AREA);
            lqi = Integer.parseInt(getData(dataLine, LQI), 16);
            voltage = Integer.parseInt(getData(dataLine, VOLTAGE), 16);

            String strTemperature = getData(dataLine, TEMPERATURE_UPPER);
            strTemperature = strTemperature + getData(dataLine, TEMPERATURE_LOWER);
            temperature = (Integer.parseInt(strTemperature, 16) - 3960) / 100.0;

            String strHumidity = getData(dataLine, HUMIDITY_UPPER);
            strHumidity = strHumidity + getData(dataLine, HUMIDITY_LOWER);
            int humiTmp = Integer.parseInt(strHumidity, 16);
            humidity = (0.0367 * humiTmp) - (0.0000015955 * humiTmp * humiTmp) - 2.0468;
        } catch (Exception e) {
            throw new IllegalArgumentException();
        }
    }

    private String getData(String dataLine, int type) {
        return dataLine.substring(type, type + 2);
    }

    public String getRouterID() {
        return routerID;
    }

    public String getLeafID() {
        return leafID;
    }

    public String getReserveArea() {
        return reserveArea;
    }

    public int getLQI() {
        return lqi;
    }

    public int getVoltage() {
        return voltage;
    }

    public double getTemperature() {
        return temperature;
    }

    public double getHumidity() {
        return humidity;
    }

    public long getDate() {
        return time;
    }

    public String getPureData() {
        return dataLine;
    }

    public void printConsole() {

        System.out.println("---------------@----------------");
        System.out.println("GetTime  : " + time);
        System.out.println("RecvData : " + dataLine);
        System.out.println("RouterID : " + routerID);
        System.out.println("LeafID   : " + leafID);
        System.out.println("RsrvArea : " + reserveArea);
        System.out.println("LQI      : " + lqi);
        System.out.println("Voltage  : " + voltage);
        System.out.println("Temperature : " + temperature);
        System.out.println("Humidity : " + humidity);
        System.out.println("---------------@----------------");
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof SockTScanValue)) {
            return false;
        }
        SockTScanValue compareItem = (SockTScanValue) obj;
        if (!dataLine.equals(compareItem.dataLine)) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        return hashValue;
    }

    @Override
    public String toString() {
        return dataLine + "T" + TimeConverter.convertToString(time);
    }
}
