package jp.osoite.tomu.xmpp.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import jp.osoite.tomu.jaxb.object.TomuMessage;
import org.jivesoftware.smack.Chat;
import org.jivesoftware.smack.ChatManager;
import org.jivesoftware.smack.ConnectionConfiguration;
import org.jivesoftware.smack.Roster;
import org.jivesoftware.smack.RosterEntry;
import org.jivesoftware.smack.RosterListener;
import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.XMPPException;
import org.jivesoftware.smack.packet.Presence;

/**
 * XMPPネットワークに接続するためのマネージャクラスです．
 * 本クラスは1つのXMPPセッションのみを管理します．
 * @author shima
 */
public final class XMPPManager {

    private RosterImple rosterImple;
    private List<MessageReceiver> recvList;
    private XMPPSetting setting;

    /**
     * コンストラクタ
     * @param setting XMPPサービスに接続するためのパラメータ
     * @param recv メッセージを受信した際のイベントハンドラ
     */
    public XMPPManager(XMPPSetting setting, MessageReceiver recv) {
        this.setting = setting;
        recvList = new CopyOnWriteArrayList<MessageReceiver>();
        if (recv != null) {
            addMessageReceiver(recv);
        }
        rosterImple = new RosterImple(setting, new MessageReceiverImple());
    }

    /**
     * MessageReceiverを必要としない場合は本コンストラクタを利用できます．
     * @param setting XMPPサービスに接続するためのパラメータ
     */
    public XMPPManager(XMPPSetting setting) {
        this(setting, null);
    }

    /**
     * MessageReceiverを追加します．
     * @param recv 追加するMessageReceiver
     */
    public void addMessageReceiver(MessageReceiver recv) {
        recvList.add(recv);
    }

    /**
     * 引数のMessageReceiverを削除します．
     * @param recv 削除するMessageReceiver
     */
    public void removeMessageReceiver(MessageReceiver recv) {
        recvList.remove(recv);
    }

    private void notifyMessageReceiver(String mes) {
        for (MessageReceiver recv : recvList) {
            recv.receive(mes);
        }
    }

    private void notifyMessageReceiver(TomuMessage mes) {
        for (MessageReceiver recv : recvList) {
            recv.receive(mes);
        }
    }

    public XMPPSetting getSetting() {
        return setting;
    }

    /**
     * XMPPのネットワークにログインします．
     * @return ログインに成功した場合はtrue，失敗した場合はfalse
     * @throws org.jivesoftware.smack.XMPPException 接続時にエラーが発生した場合
     */
    public boolean login() {
        return rosterImple.login();
    }

    public boolean relogin(XMPPSetting newSetting) {
        logoff();
        this.setting = newSetting;
        rosterImple = new RosterImple(newSetting, new MessageReceiverImple());
        return rosterImple.login();
    }

    /**
     * XMPPネットワークからログアウトします．
     */
    public void logoff() {
        rosterImple.logoff();
    }

    /**
     * 指定したアカウントにメッセージを送信します．
     * @param account 送信先アカウント
     * @param message 送信メッセージ
     * @return メッセージ送信に成功した場合はtrue，失敗した場合はfalse
     */
    public boolean send(String account, String message) {
        return rosterImple.send(account, message);
    }

    /**
     * ネットワークリストに登録されているアカウントすべてにメッセージを送信します．
     * @param str 送信メッセージ
     * @return メッセージ送信に成功した場合はtrue，失敗した場合はfalse
     */
    public boolean sendToAll(String str) {
        return rosterImple.sendToAll(str);
    }

    /**
     * TomuCoreへ新しいXMPPアカウントを追加します．
     * @param account 追加するアカウント
     * @return アカウント追加に成功したときはtrue
     */
    public boolean addAccount(String account) {
        return rosterImple.addAccount(account);
    }

    /**
     * 引数のXMPPアカウントを削除します．
     * @param account 削除するアカウント
     * @return 引数のアカウントの削除に成功したときはtrue
     */
    public boolean removeAccount(String account) {
        return rosterImple.removeAccount(account);
    }

    /**
     * 全アカウントを削除します．
     */
    public void removeAllAccount() {
        rosterImple.removeAllAccount();
    }

    /**
     * 登録されている他のXMPPユーザのリストを取得します．
     * @return アカウントのStringリスト
     */
    public List<String> getUserList() {
        List<String> results = new ArrayList<String>();
        Collection<RosterEntry> entries = rosterImple.getUserList();
        for (RosterEntry entry : entries) {
            results.add(entry.getUser());
        }
        return results;
    }

    /**
     * 引数のアカウントが接続中かどうかを返します．
     * @return 引数のアカウントが接続中ならばtrue
     */
    public boolean checkOnline(String account) {
        return rosterImple.checkOnline(account);
    }

    private class MessageReceiverImple extends MessageReceiver {

        @Override
        public void receive(String mes) {
            notifyMessageReceiver(mes);
        }

        @Override
        public void receive(TomuMessage mes) {
            notifyMessageReceiver(mes);
        }
    }

    private class RosterImple implements RosterListener {

        private ChatManager chatManager;
        private XMPPConnection connection;
        private Roster roster;
        private Map<String, Chat> chatMap;
        private ConnectionConfiguration config;
        private MessageReceiver recv;
        private XMPPSetting setting;
        private volatile boolean isLoggedin;

        RosterImple(XMPPSetting setting, MessageReceiver recv) {
            Roster.setDefaultSubscriptionMode(Roster.SubscriptionMode.accept_all);
            config = new ConnectionConfiguration(setting.getHost(), setting.getPort(), setting.getService());
            chatMap = new ConcurrentHashMap<String, Chat>();
            isLoggedin = false;
            this.recv = recv;
            this.setting = setting;
        }

        boolean login() {
            synchronized (this) {
                if (isLoggedin) {
                    return false;
                } else {
                    isLoggedin = true;
                }
            }
            connection = new XMPPConnection(config);
            try {
                connection.connect();
                connection.login(setting.getAccount(), setting.getPassword());
            } catch (XMPPException ex) {
                return (isLoggedin = false);
            }
            chatManager = connection.getChatManager();
            roster = connection.getRoster();
            roster.addRosterListener(this);
            for (RosterEntry entry : roster.getEntries()) {
                if (!chatMap.containsKey(entry.getUser())) {
                    Chat chat = chatManager.createChat(entry.getUser(), recv);
                    chatMap.put(entry.getUser(), chat);
                }
            }
            return isLoggedin;
        }

        void logoff() {
            if (isLoggedin) {
                connection.disconnect();
                chatMap.clear();
                isLoggedin = false;
            }
        }

        boolean send(String account, String message) {
            if (isLoggedin) {
                if (chatMap.containsKey(account) && roster.getPresence(account).isAvailable()) {
                    Chat chat = chatMap.get(account);
                    try {
                        chat.sendMessage(WordReplacer.replaceDotAndHttp(message));
                    } catch (XMPPException ex) {
                        return false;
                    }
                    waitMilliSecond(300);
                    return true;
                }
                return false;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        boolean sendToAll(String str) {
            if (isLoggedin) {
                boolean result = true;
                for (RosterEntry entry : roster.getEntries()) {
                    try {
                        chatMap.get(entry.getUser()).sendMessage(WordReplacer.replaceDotAndHttp(str));
                        waitMilliSecond(300);
                    } catch (Exception e) {
                        result = false;
                    }
                }
                return result;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        boolean addAccount(String account) {
            if (isLoggedin) {
                roster.reload();
                if (hasAccount(account)) {
                    return true;
                }
                try {
                    roster.createEntry(account, account, null);
                    return true;
                } catch (XMPPException ex) {
                }
                return false;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        boolean removeAccount(String account) {
            if (isLoggedin) {
                roster.reload();
                RosterEntry entry = roster.getEntry(account);
                if (entry != null) {
                    try {
                        roster.removeEntry(entry);
                        return true;
                    } catch (XMPPException ex) {
                    }
                }
                return false;
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        void removeAllAccount() {
            if (isLoggedin) {
                roster.reload();
                for (RosterEntry rentry : roster.getEntries()) {
                    try {
                        roster.removeEntry(rentry);
                    } catch (XMPPException e) {
                    }
                }
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        void waitMilliSecond(long time) {
            try {
                TimeUnit.MILLISECONDS.sleep(time);
            } catch (InterruptedException ex) {
            }
        }

        Collection<RosterEntry> getUserList() {
            if (isLoggedin) {
                roster.reload();
                return roster.getEntries();
            } else {
                throw new NullPointerException("Does not login");
            }
        }

        boolean checkOnline(String account) {
            Collection<RosterEntry> entries = getUserList();
            for (RosterEntry entry : entries) {
                String address = entry.getUser();
                if (address.equals(account)) {
                    return roster.getPresence(account).isAvailable();
                }
            }
            return false;
        }

        boolean hasAccount(String account) {
            Collection<RosterEntry> entries = getUserList();
            for (RosterEntry entry : entries) {
                String address = entry.getUser();
                if (address.equals(account)) {
                    return true;
                }
            }
            return false;
        }

        @Override
        public void entriesAdded(Collection<String> addresses) {
            if (isLoggedin) {
                for (String account : addresses) {
                    if (!chatMap.containsKey(account)) {
                        Chat chat = chatManager.createChat(account, recv);
                        chatMap.put(account, chat);
                    }
                }
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        @Override
        public void entriesDeleted(Collection<String> addresses) {
            if (isLoggedin) {
                for (String account : addresses) {
                    if (chatMap.containsKey(account)) {
                        chatMap.remove(account);
                    }
                }
            } else {
                throw new IllegalStateException("Log-in flag is \"false\"");
            }
        }

        @Override
        public void entriesUpdated(Collection<String> addresses) {
        }

        @Override
        public void presenceChanged(Presence presence) {
        }
    }
}
