package jp.osoite.tomu.main;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import jp.osoite.tomu.main.jaxb.config.TableMap;
import jp.osoite.tomu.main.jaxb.config.TomuConfiguration;
import jp.osoite.tomu.main.jaxb.table.Element;
import jp.osoite.tomu.main.jaxb.table.Table;
import jp.osoite.tomu.main.jaxb.table.TomuSensorTable;
import jp.osoite.tomu.xmpp.core.XMPPSetting;

/**
 *
 * @author shima
 */
public final class TomuSetting {

    private static TomuSetting SETTING = null;
    private TomuConfiguration config;
    private TomuSensorTable tableConfig;
    private Map<String, String> tableMap;
    private Map<String, List<Element>> columnMap;
    private XMPPSetting xmppSetting;

    private TomuSetting() {
        try {
            readConfigurationFile();
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(1);
        }
        printSetting();
        initialTableMap();
        initialXmppSetting();
        System.out.println("!! Init Done !!");
    }

    private void readConfigurationFile() throws JAXBException {
        JAXBContext jaxbc = JAXBContext.newInstance("jp.osoite.tomu.main.jaxb.config");
        Unmarshaller unmConf = jaxbc.createUnmarshaller();
        config = (TomuConfiguration) unmConf.unmarshal(new File("./configs/TomuConfiguration.xml"));
        System.out.println("Read : ./configs/TomuConfiguration.xml");
        jaxbc = JAXBContext.newInstance("jp.osoite.tomu.main.jaxb.table");
        unmConf = jaxbc.createUnmarshaller();
        tableConfig = (TomuSensorTable) unmConf.unmarshal(new File("./configs/TomuSensorTable.xml"));
        System.out.println("Read : ./configs/TomuSensorTable.xml");
    }

    private void printSetting(){
        System.out.println("Use CommandPrompt : " + config.getTomuOption().isCommandPrompt());
        System.out.println("Use GUIPanel : " + config.getTomuOption().isGUIPanel());
        System.out.println("JDBC Driver : " + config.getDatabaseSetting().getDriver());
        System.out.println("DB URL : " + config.getDatabaseSetting().getUrl());
        System.out.println("DB User : " + config.getDatabaseSetting().getUser());
        System.out.println("DB User ITree : " + config.getDatabaseSetting().isUseItree());
        System.out.println("XMPP Account : " + config.getXmppSetting().getAccount());
        System.out.println("XMPP Host : " + config.getXmppSetting().getHost());
        System.out.println("XMPP Port : " + config.getXmppSetting().getPort());
    }

    private void initialTableMap() {
        tableMap = new HashMap<String, String>();
        List<TableMap> tableList = config.getDatabaseSetting().getTableMap();
        for (TableMap relation : tableList) {
            tableMap.put(relation.getSensorType(), relation.getTableName());
        }

        columnMap = new HashMap<String, List<Element>>();
        List<Table> columnList = tableConfig.getTable();
        for (Table tableElement : columnList) {
            columnMap.put(tableElement.getName(), tableElement.getElement());
        }
    }

    private void initialXmppSetting() {
        xmppSetting = new XMPPSetting(config.getXmppSetting().getAccount(),
                config.getXmppSetting().getPass(),
                config.getXmppSetting().getHost(),
                config.getXmppSetting().getPort().intValue(),
                config.getXmppSetting().getService());
    }

    public static TomuSetting getInstance() {
        if (SETTING == null) {
            SETTING = new TomuSetting();
        }
        return SETTING;
    }

    public String getDatabaseDriver() {
        return config.getDatabaseSetting().getDriver();
    }

    public String getDatabaseUrl() {
        return config.getDatabaseSetting().getUrl();
    }

    public String getDatabaseUser() {
        return config.getDatabaseSetting().getUser();
    }

    public String getDatabasePass() {
        return config.getDatabaseSetting().getPass();
    }

    public boolean isEnabledItree() {
        return config.getDatabaseSetting().isUseItree();
    }

    public int getItreeTimeWindow() {
        return config.getDatabaseSetting().getItreeWindowSize().intValue();
    }

    public String getTableName(String sensorType) {
        return tableMap.get(sensorType);
    }

    public XMPPSetting getXMPPSetting() {
        return xmppSetting;
    }

    public List<String> getXMPPAccountList() {
        return config.getXmppSetting().getConnection();
    }

    public List<Table> getTableConfiguration() {
        return tableConfig.getTable();
    }

    public List<Element> getColumnList(String tableName){
        return columnMap.get(tableName);
    }

    public boolean requireCommandPrompt(){
        return config.getTomuOption().isCommandPrompt();
    }

    public boolean requireGuiPanel(){
        return config.getTomuOption().isGUIPanel();
    }
}
