package jp.osoite.tomu.main;

import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.xml.bind.JAXBException;
import jp.osoite.tomu.database.DatabaseConnectionManager;
import jp.osoite.tomu.database.InsertBuilder;
import jp.osoite.tomu.database.SelectBuilder;
import jp.osoite.tomu.database.TableBuilder;
import jp.osoite.tomu.itree.ITree;
import jp.osoite.tomu.itree.ITreeCore;
import jp.osoite.tomu.itree.NodeDeleteListener;
import jp.osoite.tomu.itree.util.Constants;
import jp.osoite.tomu.jaxb.object.Head;
import jp.osoite.tomu.jaxb.object.Query;
import jp.osoite.tomu.jaxb.object.SensorData;
import jp.osoite.tomu.jaxb.object.TomuMessage;
import jp.osoite.tomu.jaxb.util.HeadBuilder;
import jp.osoite.tomu.jaxb.util.JAXBConverter;
import jp.osoite.tomu.jaxb.util.ResponseBuilder;
import jp.osoite.tomu.jaxb.util.TomuMessageType;
import jp.osoite.tomu.main.jaxb.table.Table;
import jp.osoite.tomu.xmpp.core.MessageReceiver;
import jp.osoite.tomu.xmpp.core.XMPPManager;

public final class TomuRunner {

    private DatabaseConnectionManager dbManager;
    private InsertBuilder insert;
    private SelectBuilder selectBuilder;
    private ITree itree;
    private Map<Long, String> itreeIdMap;
    private XMPPManager xmppManager;
    private TomuCommandPrompt prompt;
    private final Object lock = new Object();

    public TomuRunner() throws ClassNotFoundException, SQLException {
        initializeDatabase();
        initializeXmpp();
        if (TomuSetting.getInstance().requireCommandPrompt()) {
            prompt = new TomuCommandPrompt(xmppManager);
            prompt.start();
        }
    }

    private void initializeDatabase() throws ClassNotFoundException, SQLException {
        dbManager = new DatabaseConnectionManager(TomuSetting.getInstance());
        List<Table> list = TomuSetting.getInstance().getTableConfiguration();
        for (Table t : list) {
            TableBuilder.createSensorTable(t, dbManager.getDefaultStatement());
        }
        if (TomuSetting.getInstance().isEnabledItree()) {
            itree = new ITreeCore(
                    "tomu.rtr",
                    128,
                    Constants.CACHESIZE,
                    2,
                    TomuSetting.getInstance().getItreeTimeWindow(),
                    new ItreeIdMapEraser());
            itreeIdMap = new ConcurrentHashMap<Long, String>();
        }
        insert = new InsertBuilder(TomuSetting.getInstance(), dbManager.getDefaultConnection());
        selectBuilder = new SelectBuilder(TomuSetting.getInstance(), dbManager.getDefaultConnection());
    }

    private void initializeXmpp() {
        xmppManager = new XMPPManager(TomuSetting.getInstance().getXMPPSetting());
        xmppManager.addMessageReceiver(new TomuMessageReceiver());
        xmppManager.login();
        List<String> list = TomuSetting.getInstance().getXMPPAccountList();
        for (String acc : list) {
            xmppManager.addAccount(acc);
        }
    }

    public static void main(String[] args) {
        try {
            TomuRunner runner = new TomuRunner();
        } catch (ClassNotFoundException ex) {
            ex.printStackTrace();
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
    }

    // =================================================================
    private class TomuMessageReceiver extends MessageReceiver {

        private volatile long sensorDataId;

        @Override
        public void receive(TomuMessage tomuMsg) {
            switchMessage(tomuMsg);
        }

        private void switchMessage(TomuMessage mes) {
            String type = mes.getType();
            if (TomuMessageType.REQUEST.equals(type)) {
                switchRequest(mes.getRequest().getQuery(), mes.getHead());
            } else if (TomuMessageType.RESPONSE.equals(type)) {
            } else if (TomuMessageType.SOURCE_UPLOAD.equals(type)) {
                switchSourceUpload(mes.getUpload().getSensorData());
            }
        }

        private void switchSourceUpload(List<SensorData> list) {
            for (SensorData data : list) {
                try {
                    synchronized (lock) {
                        insert.insert(data, ++sensorDataId);
                        if (TomuSetting.getInstance().isEnabledItree()) {
                            itree.insert(
                                    sensorDataId,
                                    data.getLatitude().doubleValue(),
                                    data.getLongitude().doubleValue(),
                                    data.getLatitude().doubleValue(),
                                    data.getLongitude().doubleValue(),
                                    data.getSensedTime().longValue());
                            itreeIdMap.put(sensorDataId, data.getSensorType());
                        }
                    }
                } catch (SQLException ex) {
                    ex.printStackTrace();
                }

            }
        }

        private void switchRequest(List<Query> list, Head head) {
            ResponseBuilder builder = new ResponseBuilder(HeadBuilder.buildReturnHead(head));
            for (Query q : list) {
                try {
                    synchronized (lock) {
                        List<SensorData> sensorDataList = null;
                        if (TomuSetting.getInstance().isEnabledItree()) {
                            sensorDataList = selectBuilder.select(q, itree);
                        } else {
                            sensorDataList = selectBuilder.select(q);
                        }
                        if (sensorDataList != null) {
                            builder.addResult(q.getQueryId().intValue(), sensorDataList);
                        }
                    }
                } catch (SQLException ex) {
                    ex.printStackTrace();
                }
            }
            TomuMessage mes = builder.getTomuMessage();
            try {
                xmppManager.send(mes.getHead().getDestID(), JAXBConverter.convertToXml(mes));
            } catch (JAXBException ex) {
                ex.printStackTrace();
            }
        }
    }

    private class ItreeIdMapEraser implements NodeDeleteListener {

        public void delete(List<Long> list) {
            for (Long id : list) {
                synchronized (lock) {
                    itreeIdMap.remove(id);
                }
            }
        }
    }
}
