package jp.osoite.tomu.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import jp.osoite.tomu.itree.ITree;
import jp.osoite.tomu.jaxb.object.Query;
import jp.osoite.tomu.jaxb.object.SensorData;
import jp.osoite.tomu.jaxb.object.TimeSpan;
import jp.osoite.tomu.jaxb.object.Value;
import jp.osoite.tomu.jaxb.util.SensorDataBuilder;
import jp.osoite.tomu.main.TomuSetting;
import jp.osoite.tomu.main.jaxb.table.Element;
import jp.osoite.tomu.main.jaxb.table.Table;
import jp.osoite.tomu.util.NullChecker;

/**
 *
 * @author shima
 */
public final class SelectBuilder {

    private static final double LAT_10_M = 0.000090437;
    private static final double LON_10_M = 0.000111256;
    private static final String SELECT_SQL = "SELECT * FROM ";
    private List<Table> tableList;
    private Map<String, PreparedStatement> preMap;
    private TomuSetting setting;

    public SelectBuilder(TomuSetting setting, Connection conn) {
        this.setting = setting;
        tableList = setting.getTableConfiguration();
        buildPreparedStatement(conn);
    }

    private void buildPreparedStatement(Connection conn) {
        preMap = new ConcurrentHashMap<String, PreparedStatement>();
        for (Table tableElement : tableList) {
            StringBuilder builder = new StringBuilder(SELECT_SQL);
            builder.append(tableElement.getName());
            if (setting.isEnabledItree()) {
                builder.append(" WHERE ").append(DefaultColumn.DATA_ID).append(" = ?");
            } else {
                builder.append(" WHERE ").append(DefaultColumn.SENSED_TIME).append(" BETWEEN ? AND ?");
            }
            builder.append(" ORDER BY ").append(DefaultColumn.SENSED_TIME).append(" DESC LIMIT ?");
            try {
                preMap.put(tableElement.getName(), conn.prepareStatement(builder.toString()));
            } catch (SQLException ex) {
                ex.printStackTrace();
            }
        }
    }

    public List<SensorData> select(Query query) throws SQLException {
        TimestampSpan span;
        if(query.getTimeSpan() != null){
            TimeSpan ts = query.getTimeSpan();
            span = new TimestampSpan(ts.getEnd().longValue(), ts.getBegin().longValue());
        } else {
            span = new TimestampSpan();
        }
        List<SensorData> result = new ArrayList<SensorData>();
        if (TomuAnnotation.TYPE_ANY.equals(query.getSensorType())) {
            Set<String> keySet = preMap.keySet();
            for (String tableName : keySet) {
                PreparedStatement stat = preMap.get(tableName);
                stat.setTimestamp(1, span.getEnd());
                stat.setTimestamp(2, span.getBegin());
                stat.setInt(3, query.getResultNumberRange().intValue());
                result.addAll(calcResultSet(stat, query, tableName));
            }
        } else {
            String tableName = setting.getTableName(query.getSensorType());
            if (!NullChecker.isNull(tableName)) {
                PreparedStatement stat = preMap.get(tableName);
                stat.setTimestamp(1, span.getEnd());
                stat.setTimestamp(2, span.getBegin());
                stat.setInt(3, query.getResultNumberRange().intValue());
                result = calcResultSet(stat, query, tableName);
            }
        }
        return result;
    }

    public List<SensorData> select(Query query, ITree itree) throws SQLException {
        // ============================================
        // Lack of Accuracy !!!
        int meterRadius = query.getRadius().getValue().intValue();
        if (query.getRadius().getUnit().equals(TomuAnnotation.UNIT_KM)) {
            meterRadius *= 1000;
        }
        Location distLocPlus = getRadiusPoint(
                query.getLatitude().doubleValue(),
                query.getLongitude().doubleValue(),
                meterRadius, 1);
        Location distLocMinus = getRadiusPoint(
                query.getLatitude().doubleValue(),
                query.getLongitude().doubleValue(),
                meterRadius, -1);
        long[] ids = itree.search(
                distLocMinus.getLatitude(),
                distLocMinus.getLongitude(),
                distLocPlus.getLatitude(),
                distLocPlus.getLongitude());
        // ============================================
        List<SensorData> result = new ArrayList<SensorData>();
        if (TomuAnnotation.TYPE_ANY.equals(query.getSensorType())) {
            Set<String> keySet = preMap.keySet();
            for (String tableName : keySet) {
                for (long id : ids) {
                    PreparedStatement stat = preMap.get(tableName);
                    stat.setLong(1, id);
                    stat.setInt(2, query.getResultNumberRange().intValue());
                    result.addAll(calcResultSet(stat, query, tableName));
                }
            }
        } else {
            String tableName = setting.getTableName(query.getSensorType());
            if (!NullChecker.isNull(tableName)) {
                for (long id : ids) {
                    PreparedStatement stat = preMap.get(tableName);
                    stat.setLong(1, id);
                    stat.setInt(2, query.getResultNumberRange().intValue());
                    result = calcResultSet(stat, query, tableName);
                }
            }
        }
        return result;
    }

    private Location getRadiusPoint(double lat, double lon, int radius, int bias) {
        double resultLat = lat + LAT_10_M * (radius / 10) * bias;
        double resultLon = lon + LON_10_M * (radius / 10) * bias;
        Location loc = new Location(resultLat, resultLon);
        return loc;
    }

    private List<SensorData> calcResultSet(PreparedStatement stat, Query query, String tableName) throws SQLException {
        List<Element> columnList = setting.getColumnList(tableName);
        ResultSet results = stat.executeQuery();
        List<SensorData> result = new ArrayList<SensorData>();
        while (results.next()) {
            List<Value> valueList = new ArrayList<Value>();
            double resultLat = results.getDouble(DefaultColumn.LATITUDE);
            double resultLon = results.getDouble(DefaultColumn.LONGITUDE);
            for (Element elem : columnList) {
                Value val = getValue(results, elem.getType(), elem.getValue());
                valueList.add(val);
            }
            double dist = CoordsCalculater.calcDistHubeny(
                    query.getLatitude().doubleValue(),
                    query.getLongitude().doubleValue(),
                    resultLat, resultLon);
            if (dist <= query.getRadius().getValue().doubleValue()) {
                SensorDataBuilder builder = new SensorDataBuilder(query.getSensorType(),
                        results.getLong(DefaultColumn.SENSOR_ID),
                        results.getTimestamp(DefaultColumn.SENSED_TIME).getTime(),
                        resultLat,
                        resultLon);
                builder.setDistance(dist, query.getRadius().getUnit());
                result.add(builder.getSensorData());
            }
        }
        return result;
    }

    private Value getValue(ResultSet results, String type, String name) throws SQLException {
        Value value = new Value();
        value.setId(name);
        if (TomuAnnotation.TYPE_INT.equals(type)) {
            value.setValue(Integer.toString(results.getInt(name)));
        } else if (TomuAnnotation.TYPE_BIGINT.equals(type)) {
            value.setValue(Long.toString(results.getLong(name)));
        } else if (TomuAnnotation.TYPE_DOUBLE.equals(type)) {
            value.setValue(Double.toString(results.getDouble(name)));
        } else if (TomuAnnotation.TYPE_TEXT.equals(type)) {
            value.setValue(results.getString(name));
        } else if (TomuAnnotation.TYPE_TIMESTAMP.equals(type)) {
            value.setValue(Long.toString(results.getTimestamp(name).getTime()));
        } else {
            return null;
        }
        return value;
    }

    public void close() throws SQLException {
        Collection<PreparedStatement> list = preMap.values();
        for (PreparedStatement pre : list) {
            pre.close();
        }
    }

    @Override
    @SuppressWarnings("FinalizeDeclaration")
    protected void finalize() throws Throwable {
        try {
            close();
        } finally {
            super.finalize();
        }
    }
}
