package jp.osoite.tomu.database;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import jp.osoite.tomu.jaxb.object.SensorData;
import jp.osoite.tomu.jaxb.object.Value;
import jp.osoite.tomu.main.TomuSetting;
import jp.osoite.tomu.main.jaxb.table.Element;
import jp.osoite.tomu.main.jaxb.table.Table;
import jp.osoite.tomu.util.NullChecker;

/**
 *
 * @author shima
 */
public final class InsertBuilder {

    private static final String INSERT_SQL = "INSERT INTO ";
    private List<Table> tableList;
    private Map<String, PreparedStatement> preMap;
    private TomuSetting setting;

    public InsertBuilder(TomuSetting setting, Connection conn) {
        this.setting = setting;
        tableList = setting.getTableConfiguration();
        buildPreparedStatement(conn);
    }

    private void buildPreparedStatement(Connection conn) {
        preMap = new ConcurrentHashMap<String, PreparedStatement>();
        for (Table tableElement : tableList) {
            StringBuilder builder = new StringBuilder(INSERT_SQL);
            builder.append(tableElement.getName());
            builder.append(" VALUES(?, ?, ?, ?, ?");
            List<Element> elementList = tableElement.getElement();
            for (int i = 0, size = elementList.size(); i < size; i++) {
                builder.append(", ?");
            }
            builder.append(")");
            try {
                preMap.put(tableElement.getName(), conn.prepareStatement(builder.toString()));
            } catch (SQLException ex) {
                ex.printStackTrace();
            }
        }
    }

    public void insert(SensorData data, long dataId) throws SQLException {
        String insertedTable = setting.getTableName(data.getSensorType());
        if (!NullChecker.isNull(insertedTable)) {
            int numOfColumn = 1;
            PreparedStatement preStat = preMap.get(insertedTable);
            preStat.setLong(numOfColumn++, dataId);
            preStat.setLong(numOfColumn++, data.getSensorID().longValue());
            preStat.setTimestamp(numOfColumn++, new Timestamp(data.getSensedTime().longValue()));
            preStat.setDouble(numOfColumn++, data.getLatitude().doubleValue());
            preStat.setDouble(numOfColumn++, data.getLongitude().doubleValue());
            List<Element> elementList = setting.getColumnList(insertedTable);
            List<Value> valueList = data.getValues().getValue();
            for (Element elem : elementList) {
                String name = elem.getValue();
                for (Value val : valueList) {
                    if (name.equals(val.getId())) {
                        switchSetType(elem.getType(), val.getValue(), preStat, numOfColumn++);
                    }
                }
            }
            preStat.execute();
        }
    }

    private void switchSetType(String type, String value, PreparedStatement preStat, int count) throws SQLException {
        if (TomuAnnotation.TYPE_INT.equals(type)) {
            preStat.setInt(count, Integer.parseInt(value));
        } else if (TomuAnnotation.TYPE_BIGINT.equals(type)) {
            preStat.setLong(count, Long.parseLong(value));
        } else if (TomuAnnotation.TYPE_DOUBLE.equals(type)) {
            preStat.setDouble(count, Double.parseDouble(value));
        } else if (TomuAnnotation.TYPE_TEXT.equals(type)) {
            preStat.setString(count, value);
        } else if (TomuAnnotation.TYPE_TIMESTAMP.equals(type)) {
            preStat.setTimestamp(count, new Timestamp(Long.parseLong(value)));
        }
    }

    public void close() throws SQLException {
        Collection<PreparedStatement> list = preMap.values();
        for (PreparedStatement pre : list) {
            pre.close();
        }
    }

    @Override
    @SuppressWarnings("FinalizeDeclaration")
    protected void finalize() throws Throwable {
        try {
            close();
        } finally {
            super.finalize();
        }
    }
}
