package jp.osoite.tomu.xml.jaxb.util;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import jp.osoite.tomu.core.db.DBConnectionModule;
import jp.osoite.tomu.core.db.sqlbuilder.ItreeSQLBuilder;
import jp.osoite.tomu.core.db.sqlbuilder.ItreeSQLBuilder.ITreeTableItem;
import jp.osoite.tomu.itree.core.ITree;
import jp.osoite.tomu.itree.core.ITreeCore;
import jp.osoite.tomu.itree.util.Constants;
import jp.osoite.tomu.xml.jaxb.object.Head;
import jp.osoite.tomu.xml.jaxb.object.SensorSourceDataSet;
import jp.osoite.tomu.xml.jaxb.object.Sfeer;
import jp.osoite.tomu.xml.jaxb.object.SfeerRequestSet;
import jp.osoite.tomu.xml.jaxb.object.StaticSensorData;
import jp.osoite.tomu.xml.jaxb.object.TomuMessage;
import jp.osoite.tomu.xml.jaxb.wrapper.InitRequestSfeerWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.InitResponseSfeerWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.Location;
import jp.osoite.tomu.xml.jaxb.wrapper.RequestSfeerWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.ResponseAppsWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.TScanValue;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadCo2Wrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadHumidityWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadKito2Wrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadPedestrianWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadTScanWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadTempWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.InitRequestSfeerListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.InitResponseSfeerListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.RequestSfeerListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.ResponseAppsListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadCo2Listener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadHumidityListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadKito2Listener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadPedestrianListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadTScanListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadTempListener;
import jp.osoite.tomu.xmpp.core.RequestITreeImpl;

/**
 * TomuMessageを各種ラッパーへスイッチするためのクラスです．
 * @author shima
 */
public class TomuMessageSwitchManager {

    private static volatile long sensorId;
    private CopyOnWriteArrayList<UploadPedestrianListener> uploadPedestrianList;
    private CopyOnWriteArrayList<UploadKito2Listener> uploadKito2List;
    private CopyOnWriteArrayList<UploadTempListener> uploadTempList;
    private CopyOnWriteArrayList<UploadCo2Listener> uploadCo2List;
    private CopyOnWriteArrayList<UploadTScanListener> uploadTScanList;
    private CopyOnWriteArrayList<UploadHumidityListener> uploadHumidityList;
    private CopyOnWriteArrayList<InitRequestSfeerListener> initReqSfeerList;
    private CopyOnWriteArrayList<InitResponseSfeerListener> initRespSfeerList;
    private CopyOnWriteArrayList<RequestSfeerListener> reqSfeerList;
    private CopyOnWriteArrayList<ResponseAppsListener> respAppsList;
    private ItreeSQLBuilder itreeSql;
    private ITree itree;
    private RequestITreeImpl requestItreeImpl;

    public TomuMessageSwitchManager() {
        Connection con = DBConnectionModule.getInstance().getConnection();
        try {
            sensorId = 0;
            itreeSql = new ItreeSQLBuilder(con, "itree_main");
            itreeSql.dropTable();
            itreeSql.createTable();
            itree = new ITreeCore("test.rtr", 128, Constants.CACHESIZE, 2, 60000);
            requestItreeImpl = new RequestITreeImpl();
        } catch (Exception e) {
            e.printStackTrace();
            itreeSql = null;
        }
    }

    public void switchMessage(TomuMessage tomuMsg) {
        Head head = tomuMsg.getHead();
        String messageType = head.getMsgType();
        if (messageType.equals(TomuMessageType.APPS_REQUEST.toString())) {
            switchAppsRequest(tomuMsg);
        } else if (messageType.equals(TomuMessageType.APPS_RESPONSE.toString())) {
            switchAppsResponse(tomuMsg);
        } else if (messageType.equals(TomuMessageType.APPS_INIT_REQUEST.toString())) {
            switchAppsInitRequest(tomuMsg);
        } else if (messageType.equals(TomuMessageType.APPS_INIT_RESPONSE.toString())) {
            switchAppsInitResponse(tomuMsg);
        } else if (messageType.equals(TomuMessageType.SOURCE_UPLOAD.toString())) {
            switchSourceUpload(tomuMsg);
        }
    }

    private void switchAppsRequest(TomuMessage tomuMsg) {
        SfeerRequestSet sfeerReq = tomuMsg.getRequest().getSfeerRequestSet();
        if (sfeerReq != null) {
            try {
                notifyRequestSfeerListener(new RequestSfeerWrapper(tomuMsg));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        jp.osoite.tomu.xml.jaxb.object.ITree itreeObj = tomuMsg.getRequest().getITree();
        if (itreeObj != null) {
            searchITree(itreeObj, tomuMsg.getHead().getMsgSrc(), tomuMsg.getHead().getMsgDest(), tomuMsg.getHead().getMsgId().longValue());
        }
    }

    private void searchITree(jp.osoite.tomu.xml.jaxb.object.ITree itreeObj, String srcId, String destId, long mesId) {
        double lat1 = itreeObj.getLatitude1().doubleValue();
        double lon1 = itreeObj.getLongitude1().doubleValue();
        double lat2 = itreeObj.getLatitude2().doubleValue();
        double lon2 = itreeObj.getLongitude2().doubleValue();
        long[] resultId = itree.search(lat1, lon1, lat2, lon2);
        if (resultId.length != 0) {
            requestItreeImpl.initBuilder(srcId, destId, mesId);
            for (long id : resultId) {
                try {
                    List<ITreeTableItem> itemList = itreeSql.select(id);
                    requestItreeImpl.receive(itemList);
                } catch (SQLException ex) {
                    ex.printStackTrace();
                }
            }
            requestItreeImpl.send();
        }
    }

    private void switchAppsResponse(TomuMessage tomuMsg) {
        notifyResponseAppsListener(new ResponseAppsWrapper(tomuMsg));
    }

    private void switchAppsInitRequest(TomuMessage tomuMsg) {
        Sfeer sfeer = tomuMsg.getInitRequest().getSfeer();
        if (sfeer != null) {
            notifyInitRequestSfeerListener(new InitRequestSfeerWrapper(tomuMsg));
        }
    }

    private void switchAppsInitResponse(TomuMessage tomuMsg) {
        Sfeer sfeer = tomuMsg.getInitResponse().getSfeer();
        if (sfeer != null) {
            notifyInitResponseSfeerListener(new InitResponseSfeerWrapper(tomuMsg));
        }
    }

    private void switchSourceUpload(TomuMessage tomuMsg) {
        SensorSourceDataSet ssds = tomuMsg.getUpload().getSensorSourceDataSet();
        if (ssds.getStaticSensorData() != null) {
            List<StaticSensorData> ssdList = ssds.getStaticSensorData();
            if (!ssdList.isEmpty()) {
                StaticSensorData ssd = ssdList.get(0);
                String sensorType = ssd.getType();
                if (sensorType.equals("TEMP")) {
                    notifyUploadTempListener(new UploadTempWrapper(tomuMsg));
                } else if (sensorType.equals("CO2")) {
                    notifyUploadCo2Listener(new UploadCo2Wrapper(tomuMsg));
                } else if (sensorType.equals("HUMIDITY")) {
                    notifyUploadHumidityListener(new UploadHumidityWrapper(tomuMsg));
                } else if (sensorType.contains("PEDESTRIAN")) {
                    notifyUploadPedestrianListener(new UploadPedestrianWrapper(tomuMsg));
                }
            } else if (!ssds.getKitokitoSensorData().isEmpty()) {
                notifyUploadKito2Listener(new UploadKito2Wrapper(tomuMsg));
            } else if (!ssds.getTScanSensorData().isEmpty()) {
                notifyUploadTScanListener(new UploadTScanWrapper(tomuMsg));
            }
        } else if (ssds.getDynamicSensorData() != null) {
        }
    }

    // -------------------------------------------------------------------------
    // Listeners
    // -------------------------------------------------------------------------
    public void addUploadPedestrianListener(UploadPedestrianListener listener) {
        if (uploadPedestrianList == null) {
            uploadPedestrianList = new CopyOnWriteArrayList<UploadPedestrianListener>();
        }
        uploadPedestrianList.add(listener);
    }

    public void addUploadKito2Listener(UploadKito2Listener listener) {
        if (uploadKito2List == null) {
            uploadKito2List = new CopyOnWriteArrayList<UploadKito2Listener>();
        }
        uploadKito2List.add(listener);
    }

    public void addInitRequestSfeerListener(InitRequestSfeerListener listener) {
        if (initReqSfeerList == null) {
            initReqSfeerList = new CopyOnWriteArrayList<InitRequestSfeerListener>();
        }
        initReqSfeerList.add(listener);
    }

    public void addInitResponseSfeerListener(InitResponseSfeerListener listener) {
        if (initRespSfeerList == null) {
            initRespSfeerList = new CopyOnWriteArrayList<InitResponseSfeerListener>();
        }
        initRespSfeerList.add(listener);
    }

    public void addRequestSfeerListener(RequestSfeerListener listener) {
        if (reqSfeerList == null) {
            reqSfeerList = new CopyOnWriteArrayList<RequestSfeerListener>();
        }
        reqSfeerList.add(listener);
    }

    public void addUploadTempListener(UploadTempListener listener) {
        if (uploadTempList == null) {
            uploadTempList = new CopyOnWriteArrayList<UploadTempListener>();
        }
        uploadTempList.add(listener);
    }

    public void addUploadCo2Listener(UploadCo2Listener listener) {
        if (uploadCo2List == null) {
            uploadCo2List = new CopyOnWriteArrayList<UploadCo2Listener>();
        }
        uploadCo2List.add(listener);
    }

    public void addUploadHumidityListener(UploadHumidityListener listener) {
        if (uploadHumidityList == null) {
            uploadHumidityList = new CopyOnWriteArrayList<UploadHumidityListener>();
        }
        uploadHumidityList.add(listener);
    }

    public void addResponseAppsListener(ResponseAppsListener listener) {
        if (respAppsList == null) {
            respAppsList = new CopyOnWriteArrayList<ResponseAppsListener>();
        }
        respAppsList.add(listener);
    }

    public void addUploadTScanListener(UploadTScanListener listener) {
        if (uploadTScanList == null) {
            uploadTScanList = new CopyOnWriteArrayList<UploadTScanListener>();
        }
        uploadTScanList.add(listener);
    }

    // -------------------------------------------------------------------------
    // Notifyer
    // -------------------------------------------------------------------------
    private void notifyInitRequestSfeerListener(InitRequestSfeerWrapper wrapper) {
        if (initReqSfeerList == null) {
            return;
        }
        for (InitRequestSfeerListener listener : initReqSfeerList) {
            listener.receiveInitRequestSfeer(wrapper);
        }
    }

    private void notifyInitResponseSfeerListener(InitResponseSfeerWrapper wrapper) {
        if (initRespSfeerList == null) {
            return;
        }
        for (InitResponseSfeerListener listener : initRespSfeerList) {
            listener.receiveInitResponseSfeer(wrapper);
        }
    }

    private void notifyRequestSfeerListener(RequestSfeerWrapper wrapper) {
        if (reqSfeerList == null) {
            return;
        }
        for (RequestSfeerListener listener : reqSfeerList) {
            listener.receiveRequestSfeer(wrapper);
        }
    }

    private void notifyResponseAppsListener(ResponseAppsWrapper wrapper) {
        if (respAppsList == null) {
            return;
        }
        for (ResponseAppsListener listener : respAppsList) {
            listener.receiveResponseApps(wrapper);
        }
    }

    private void notifyUploadTempListener(UploadTempWrapper wrapper) {
        if (uploadTempList == null) {
            return;
        }
        for (UploadTempListener listener : uploadTempList) {
            listener.receiveUploadTemp(wrapper);
        }
    }

    private void notifyUploadHumidityListener(UploadHumidityWrapper wrapper) {
        if (uploadHumidityList == null) {
            return;
        }
        for (UploadHumidityListener listener : uploadHumidityList) {
            listener.receiveUploadHumidity(wrapper);
        }
    }

    private void notifyUploadCo2Listener(UploadCo2Wrapper wrapper) {
        if (uploadCo2List == null) {
            return;
        }
        for (UploadCo2Listener listener : uploadCo2List) {
            listener.receiveUploadCo2(wrapper);
        }
    }

    private void notifyUploadTScanListener(UploadTScanWrapper wrapper) {
        if (uploadTScanList == null) {
            return;
        }
        for (UploadTScanListener listener : uploadTScanList) {
            listener.receiveUploadTScan(wrapper);
        }
        List<TScanValue> valueList = wrapper.getTScanValueList();
        for (TScanValue val : valueList) {
            Location loc = TScanLocation.get(val.getTscanId());
            if (loc != null) {
                try {
                    long id = addAndGet(1);
                    itreeSql.insert(id, val.getSensedTime(), "TSCAN", val.getTscanId(), loc.getLat(), loc.getLon());
                    itree.insert(id, loc.getLat(), loc.getLon(), loc.getLat(), loc.getLon(), val.getSensedTime());
                } catch (SQLException ex) {
                    ex.printStackTrace();
                }
            }
        }
        System.out.print("Itree Elements : ");
        itree.printAllTreeElements();
        System.out.println();
    }

    private void notifyUploadKito2Listener(UploadKito2Wrapper wrapper) {
        if (uploadKito2List == null) {
            return;
        }
        try {
            long id = addAndGet(1);
            itreeSql.insert(id, wrapper.getSensedTime(), "KITO2", Long.toString(wrapper.getSensorId()), wrapper.getLatitude(), wrapper.getLongitude());
            itree.insert(id, wrapper.getLatitude(), wrapper.getLongitude(), wrapper.getLatitude(), wrapper.getLongitude(), wrapper.getSensedTime());
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        for (UploadKito2Listener listener : uploadKito2List) {
            listener.receiveUploadKito2(wrapper);
        }
    }

    private void notifyUploadPedestrianListener(UploadPedestrianWrapper wrapper) {
        if (uploadPedestrianList == null) {
            return;
        }
        try {
            Location loc = PedestrianLocation.get(wrapper.getSensorId());
            long id = addAndGet(1);
            itreeSql.insert(id, wrapper.getSensedTime(), "PEDESTRIAN", Long.toString(wrapper.getSensorId()), loc.getLat(), loc.getLon());
            itree.insert(id, loc.getLat(), loc.getLon(), loc.getLat(), loc.getLon(), wrapper.getSensedTime());
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        for (UploadPedestrianListener listener : uploadPedestrianList) {
            listener.receiveUploadPedestrian(wrapper);
        }
    }

    private synchronized long addAndGet(long i) {
        sensorId += i;
        return sensorId;
    }
}
