/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2012 Yuichi Watanabe
 */

#include <core/printf.h>
#include <core/string.h>
#include "constants.h"
#include "convert.h"
#include "cpu_emul.h"
#include "cpuid.h"
#include "current.h"
#include "panic.h"
#include "pcpu.h"
#include "vmctl.h"
#include "vt.h"
#include "vt_internal.h"

static void vt_extint_pending (bool pending);
static bool vt_extint_is_blocked ();
static void vt_tsc_offset_changed (void);
static void vt_generate_pagefault (ulong err, ulong cr2);
static void vt_generate_external_int (vector_t vector);

static struct vmctl_func func = {
	.vminit			= vt_vminit,
	.vmexit			= vt_vmexit,
	.run_vm			= vt_run_vm,
	.generate_pagefault	= vt_generate_pagefault,
	.generate_external_int	= vt_generate_external_int,
	.read_general_reg	= vt_read_general_reg,
	.write_general_reg	= vt_write_general_reg,
	.read_control_reg	= vt_read_control_reg,
	.write_control_reg	= vt_write_control_reg,
	.read_sreg_sel		= vt_read_sreg_sel,
	.read_sreg_acr		= vt_read_sreg_acr,
	.read_sreg_base		= vt_read_sreg_base,
	.read_sreg_limit	= vt_read_sreg_limit,
	.spt_setcr3		= vt_spt_setcr3,
	.read_ip		= vt_read_ip,
	.write_ip		= vt_write_ip,
	.read_flags		= vt_read_flags,
	.write_flags		= vt_write_flags,
	.read_gdtr		= vt_read_gdtr,
	.write_gdtr		= vt_write_gdtr,
	.read_idtr		= vt_read_idtr,
	.write_idtr		= vt_write_idtr,
	.write_realmode_seg	= vt_write_realmode_seg,
	.writing_sreg		= vt_writing_sreg,
	.read_msr		= vt_read_msr,
	.write_msr		= vt_write_msr,
	.cpuid			= call_cpuid,
	.extint_pending		= vt_extint_pending,
	.extint_is_blocked	= vt_extint_is_blocked,
	.extern_iopass		= vt_extern_iopass,
	.tsc_offset_changed	= vt_tsc_offset_changed,
	.panic_dump		= vt_panic_dump,
	.invlpg			= vt_invlpg,
	.reset			= vt_reset,
};

void
vt_vmctl_init (void)
{
	memcpy ((void *)&current->vmctl, (void *)&func, sizeof func);
}

static void
vt_generate_pagefault (ulong err, ulong cr2)
{
	struct vt_intr_data *vid = &current->u.vt.intr;

	if (vid->intr_info.s.valid == INTR_INFO_VALID_VALID) {
		panic ("Can't inject page fault because the other event is already pending. "
		       "vector 0x%x type 0x%x, err_valid 0x%x",
		       vid->intr_info.s.vector,
		       vid->intr_info.s.type,
		       vid->intr_info.s.err);
	}

	vid->intr_info.v = 0;
	vid->intr_info.s.vector = EXCEPTION_PF;
	vid->intr_info.s.type = INTR_INFO_TYPE_HARD_EXCEPTION;
	vid->intr_info.s.err = INTR_INFO_ERR_VALID;
	vid->intr_info.s.valid = INTR_INFO_VALID_VALID;
	vid->exception_errcode = err;
	vid->instruction_len = 0;
	current->u.vt.vr.cr2 = cr2;
}

static void
vt_generate_external_int (vector_t vector)
{
	struct vt_intr_data *vid = &current->u.vt.intr;
	if (vid->intr_info.s.valid == INTR_INFO_VALID_VALID) {
		panic ("Can't inject external interrupt because the other event is already pending. "
		       "vector 0x%x type 0x%x, err_valid 0x%x",
		       vid->intr_info.s.vector,
		       vid->intr_info.s.type,
		       vid->intr_info.s.err);
	}
	vid->intr_info.v = 0;
	vid->intr_info.s.vector = vector;
	vid->intr_info.s.type = INTR_INFO_TYPE_EXTERNAL;
	vid->intr_info.s.err = INTR_INFO_ERR_INVALID;
	vid->intr_info.s.valid = INTR_INFO_VALID_VALID;
	vid->instruction_len = 0;
}

static bool
vt_extint_is_blocked (void)
{
	ulong int_state;
	ulong rflags;
	struct vt_intr_data *vid = &current->u.vt.intr;

	vt_read_flags (&rflags);
	asm_vmread (VMCS_GUEST_INTERRUPTIBILITY_STATE, &int_state);

	return (!(rflags & RFLAGS_IF_BIT)
		|| ((int_state
		     & (VMCS_GUEST_INTERRUPTIBILITY_STATE_BLOCKING_BY_STI_BIT |
			VMCS_GUEST_INTERRUPTIBILITY_STATE_BLOCKING_BY_MOV_SS_BIT)))
		|| vid->intr_info.s.valid == INTR_INFO_VALID_VALID);
}

static void
vt_extint_pending (bool pending)
{
	ulong ctl;

	asm_vmread (VMCS_PROC_BASED_VMEXEC_CTL, &ctl);
	if (pending) {
		/*
		 * Make VM-exit occur at the beginning of next
		 * interrupt window.
		 */
		ctl |= VMCS_PROC_BASED_VMEXEC_CTL_INTRWINEXIT_BIT;
	} else {
		ctl &= ~(ulong)VMCS_PROC_BASED_VMEXEC_CTL_INTRWINEXIT_BIT;
	}
	asm_vmwrite (VMCS_PROC_BASED_VMEXEC_CTL, ctl);
}

static void
vt_tsc_offset_changed (void)
{
	u32 l, h;

	conv64to32 (current->tsc_offset, &l, &h);
	asm_vmwrite (VMCS_TSC_OFFSET, l);
	asm_vmwrite (VMCS_TSC_OFFSET_HIGH, h);
}
