/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2012 Yuichi Watanabe
 */

#include <core/printf.h>
#include <core/string.h>
#include "asm.h"
#include "cpu_emul.h"
#include "cpuid.h"
#include "current.h"
#include "panic.h"
#include "svm.h"
#include "svm_init.h"
#include "svm_io.h"
#include "svm_main.h"
#include "svm_msr.h"
#include "svm_panic.h"
#include "svm_regs.h"
#include "vmctl.h"

#define UNIMPLEMENTED() do { \
	panic ("DEBUG : file %s line %d function %s NOT IMPLEMENTED", \
	       __FILE__, __LINE__, __FUNCTION__); \
} while (0)

static void svm_generate_pagefault (ulong err, ulong cr2);
static void svm_extint_pending (bool pending);
static bool svm_extint_is_blocked ();
static void svm_generate_external_int (vector_t num);
static void svm_tsc_offset_changed (void);

static struct vmctl_func func = {
	.vminit			= svm_vminit,
	.vmexit			= svm_vmexit,
	.run_vm			= svm_run_vm,
	.generate_pagefault	= svm_generate_pagefault,
	.generate_external_int	= svm_generate_external_int,
	.read_general_reg	= svm_read_general_reg,
	.write_general_reg	= svm_write_general_reg,
	.read_control_reg	= svm_read_control_reg,
	.write_control_reg	= svm_write_control_reg,
	.read_sreg_sel		= svm_read_sreg_sel,
	.read_sreg_acr		= svm_read_sreg_acr,
	.read_sreg_base		= svm_read_sreg_base,
	.read_sreg_limit	= svm_read_sreg_limit,
	.spt_setcr3		= svm_spt_setcr3,
	.read_ip		= svm_read_ip,
	.write_ip		= svm_write_ip,
	.read_flags		= svm_read_flags,
	.write_flags		= svm_write_flags,
	.read_gdtr		= svm_read_gdtr,
	.write_gdtr		= svm_write_gdtr,
	.read_idtr		= svm_read_idtr,
	.write_idtr		= svm_write_idtr,
	.write_realmode_seg	= svm_write_realmode_seg,
	.writing_sreg		= svm_writing_sreg,
	.read_msr		= svm_read_msr,
	.write_msr		= svm_write_msr,
	.cpuid			= call_cpuid,
	.extint_pending		= svm_extint_pending,
	.extint_is_blocked	= svm_extint_is_blocked,
	.extern_iopass		= svm_extern_iopass,
	.tsc_offset_changed	= svm_tsc_offset_changed,
	.panic_dump		= svm_panic_dump,
	.invlpg			= svm_invlpg,
	.reset			= svm_reset,
};

static void
svm_generate_pagefault (ulong err, ulong cr2)
{
	struct svm_intr_data *sid = &current->u.svm.intr;

	sid->intr_info.v = 0;
	sid->intr_info.s.vector = EXCEPTION_PF;
	sid->intr_info.s.type = VMCB_EVENTINJ_TYPE_EXCEPTION;
	sid->intr_info.s.ev = 1;
	sid->intr_info.s.v = 1;
	sid->intr_info.s.errorcode = (u32)err;
	current->u.svm.vi.vmcb->cr2 = cr2;
}

static void
svm_extint_pending (bool pending)
{
	if (pending) {
		current->u.svm.vi.vmcb->v_irq = 1;
		current->u.svm.vi.vmcb->v_ign_tpr = 1;
		current->u.svm.vi.vmcb->intercept_vintr = 1;
	} else {
		current->u.svm.vi.vmcb->v_irq = 0;
		current->u.svm.vi.vmcb->intercept_vintr = 0;
	}
}

static bool
svm_extint_is_blocked (void)
{
	ulong rflags;

	svm_read_flags (&rflags);
	return (!(rflags & RFLAGS_IF_BIT)
		|| current->u.svm.vi.vmcb->interrupt_shadow
		|| current->u.svm.intr.intr_info.s.v == 1);
}

static void
svm_generate_external_int (vector_t vector)
{
	struct svm_intr_data *sid = &current->u.svm.intr;

	if (sid->intr_info.s.v == 1) {
		panic ("Can't inject external interrupt because the other event is already pending. "
		       "vector 0x%x type 0x%x, ev 0x%x",
		       sid->intr_info.s.vector,
		       sid->intr_info.s.type,
		       sid->intr_info.s.ev);
	}
	sid->intr_info.v = 0;
	sid->intr_info.s.vector = vector;
	sid->intr_info.s.type = VMCB_EVENTINJ_TYPE_INTR;
	sid->intr_info.s.ev = 0;
	sid->intr_info.s.v = 1;
}

static void
svm_tsc_offset_changed (void)
{
	current->u.svm.vi.vmcb->tsc_offset = current->tsc_offset;
}

void
svm_vmctl_init (void)
{
	memcpy ((void *)&current->vmctl, (void *)&func, sizeof func);
}
