﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace TabTextFinder.Finder
{
    // This class should be instantiated while the file is locked by this process.
    // Otherwise, the consistency of data & tag may not be assured.
    sealed class FileInfoComparer : IEqualityComparer<FileInfo>
    {
        public bool Equals( FileInfo a, FileInfo b )
        {
            if (a == null || b == null) { return a == null && b == null; }
            return a.FullName == b.FullName && a.Length == b.Length && a.LastWriteTime == b.LastWriteTime;
        }

        public int GetHashCode( FileInfo info )
        {
            return info.GetHashCode();
        }
    }

    interface IFileCache
    {
        FileInfo Info { get; }
        long Size { get; }              // the size in bytes
        uint Generation { get; set; }   // the last generation when this cache was used
    }

    sealed class FileEncodingCache : IFileCache
    {
        public FileInfo Info { get; private set; }
        public long Size { get { return 0; } }         // to cache this always
        public uint Generation { get; set; }
        public string Encoding { get; private set; }

        public FileEncodingCache( FileInfo info, string encoding )
        {
            Info = info;
            Encoding = encoding;
        }
    }

    sealed class FileContentCache : IFileCache
    {
        public FileInfo Info { get; private set; }
        public long Size { get { return (string.IsNullOrEmpty( Content ) ? 0 : (long) Content.Length * 2/*char*/); } }
        public uint Generation { get; set; }
        public string Encoding { get; private set; }
        public string Content { get; private set; }

        public FileContentCache( FileInfo info, string encoding, string content )
        {
            Info = info;
            Encoding = encoding;
            Content = content;
        }
    }

    sealed class FileWeakContentCache : IFileCache
    {
        public FileInfo Info { get; private set; }
        public long Size { get { return 0; } }
        public uint Generation { get; set; }

        private string encoding;
        private WeakReference weak_content;

        public FileWeakContentCache( FileContentCache cache )
        {
            Info = cache.Info;
            encoding = cache.Encoding;
            weak_content = new WeakReference( cache.Content );
        }

        public FileContentCache Lock()
        {
            string content = (string) weak_content.Target;
            if (content == null) { return null; }
            return new FileContentCache( Info, encoding, content );
        }
    }

    class FileCacheMap<T> : Dictionary<string, T> where T : IFileCache
    {
        public long Size { get; private set; }

        public new void Add( string key, T cache )
        {
            base.Add( key, cache );
            Size += cache.Size;
        }

        public new bool Remove( string key )
        {
            T cache = this[key];
            if (cache == null) { return false; }

            Size -= cache.Size;
            return base.Remove( key );
        }

        // Throw away old caches whose generation is older than current generation by more than `threshold'
        public void Purge( uint generation, int threshold )
        {
            // List up unreferenced cache keys anticipating that all caches are referenced everytime hopefully
            List<string> paths = new List<string>();
            foreach (string path in Keys) {
                T cache = this[path];
                if (generation - cache.Generation >= threshold) {
                    paths.Add( path );
                }
            }
            foreach (string path in paths) {
                Remove( path );
            }

            CheckSize();
        }

        private bool CheckSize()
        {
            long size = 0;
            foreach (T cache in Values) {
                size += cache.Size;
            }
            Debug.Assert( Size == size );
            return (Size == size);
        }
    }

    class FileLock : IDisposable
    {
        public bool Locked { get { return fs != null; } }

        private FileStream fs;

        public FileLock( string path )
        {
            try {
                fs = new FileStream( path, FileMode.Open, FileAccess.Read, FileShare.Read );
            }
            catch (Exception) {
            }
        }

        public void Dispose()
        {
            if (fs != null) {
                fs.Close();
                fs.Dispose();
                fs = null;
            }
        }
    }

    class FileCacheManager
    {
        public long MaxCacheSize { get; private set; }

        private FileInfoComparer comparer = new FileInfoComparer();
        private FileCacheMap<FileEncodingCache> mapEncoding;
        private FileCacheMap<FileContentCache> mapContent;
        private FileCacheMap<FileWeakContentCache> mapWeakContent;

        private const int threshold = 3;
        private uint generation;

        public void SetPolicy( bool cache_content, long cache_size )
        {
            enableCache( true, cache_content );
            MaxCacheSize = cache_size * 1024 * 1024;
        }

        public void IncrementGeneration()
        {
            generation++;
        }

        public void PurgeElderCache()
        {
            if (generation < threshold) { return; }
            if (mapEncoding != null) {
                mapEncoding.Purge( generation, threshold );
            }
            if (mapContent != null) {
                mapContent.Purge( generation, threshold );
            }
            if (mapWeakContent != null) {
                mapWeakContent.Purge( generation, threshold );
            }
        }

        private void enableCache( bool bCacheEncodings, bool bCacheContent )
        {
            recreateCache( ref mapEncoding, bCacheEncodings );
            recreateCache( ref mapContent, bCacheContent );
            recreateCache( ref mapWeakContent, !bCacheContent );
        }

        private void recreateCache<T>( ref FileCacheMap<T> mapCache, bool bEnable ) where T : IFileCache
        {
            if (bEnable) {
                mapCache = new FileCacheMap<T>();
            } else {
                mapCache = null;
            }
        }

        private T getCache<T>( FileCacheMap<T> mapCache, FileInfo info ) where T : IFileCache
        {
            if (mapCache == null || mapCache.Count == 0) { return default( T ); }

            lock (mapCache) {
                T cache;
                if (mapCache.TryGetValue( info.FullName, out cache )) {
                    if (comparer.Equals( cache.Info, info )) {
                        // mark as referenced
                        cache.Generation = generation;
                        return cache;
                    }
                    // stale cache
                    mapCache.Remove( info.FullName );
                }
            }
            return default( T );
        }

        private T addCache<T>( FileCacheMap<T> mapCache, T cache ) where T : IFileCache
        {
            if (mapCache == null) { return cache; }

            lock (mapCache) {
                string key = cache.Info.FullName;
                // In this application, the same path would hardly be added in a single find operation.
                // Thus, no racing would not occur in the following Add operation.
                // But just in case, use the last one
                if (mapCache.ContainsKey( key )) {
                    mapCache.Remove( key );
                }
                cache.Generation = generation;
                if (mapCache.Size + cache.Size <= MaxCacheSize) {
                    mapCache.Add( key, cache );
                }
            }
            return cache;
        }

        // need FileLock before calling
        private FileEncodingCache getEncoding( FileInfo info )
        {
            FileEncodingCache cache = getCache( mapEncoding, info );
            if (cache != null) { return cache; }

            try {
                using (FileStream fs = new FileStream( info.FullName, FileMode.Open, FileAccess.Read, FileShare.Read )) {
                    Dobon.Jcode jcode = new Dobon.Jcode();
                    Encoding enc = jcode.GetCode( fs );
                    if (enc == null) { return null; }
                    return addCache( mapEncoding, new FileEncodingCache( info, enc.WebName ) );
                }
            }
            catch (Exception) {
                return null;
            }
        }

        // no need of FileLock, just look at cache
        private FileContentCache getCachedContent( FileInfo info )
        {
            FileContentCache cache = getCache( mapContent, info );
            if (cache != null) { return cache; }

            FileWeakContentCache wcache = getCache( mapWeakContent, info );
            if (wcache != null) {
                cache = wcache.Lock();
                if (cache != null) { return cache; }
            }

            return null;
        }

        // need FileLock before calling
        private FileContentCache getContent( FileInfo info )
        {
            FileContentCache cache = getCachedContent( info );
            if (cache != null) { return cache; }

            FileEncodingCache encoding = getEncoding( info );
            if (encoding == null) { return null; }

            try {
                // with Azuki, looks following is just suffice
                Encoding enc = Encoding.GetEncoding( encoding.Encoding, EncoderFallback.ReplacementFallback, DecoderFallback.ReplacementFallback );

                // this method consumes unexpectedly big memory?
                //using (StreamReader reader = new StreamReader( info.FullName, enc, false, 1 << 14 )) {
                //    string content = reader.ReadToEnd();
                //}

                byte[] bytes = File.ReadAllBytes( info.FullName );
                string content = enc.GetString( bytes );

                cache = new FileContentCache( encoding.Info, encoding.Encoding, content );
                if (mapWeakContent != null) {
                    addCache( mapWeakContent, new FileWeakContentCache( cache ) );
                }
                return addCache( mapContent, cache );
            }
            catch (Exception) {
                return null;
            }
        }

        // read the whole file contenet
        public FileContentCache GetContent( string file )
        {
            using (FileLock flock = new FileLock( file )) {
                if (!flock.Locked) { return null; }
                try {
                    FileInfo info = new FileInfo( file );
                    return getContent( info );
                }
                catch (Exception) {
                    return null;
                }
            }
        }

        // return the reader from string when cached, or from file
        public TextReader GetContentReader( string path, ref FoundFile file )
        {
            // FileLock takes longer time, so check the cache first
            try {
                FileInfo info = new FileInfo( path );
                FileContentCache cache = getCachedContent( info );
                if (cache != null) {
                    file = new FoundFile( cache.Info, cache.Encoding );
                    return new StringReader( cache.Content );
                }
            }
            catch (Exception) {
                return null;
            }

            using (FileLock flock = new FileLock( path )) {
                if (!flock.Locked) { return null; }
                try {
                    FileInfo info = new FileInfo( path );
                    if (mapContent != null && mapContent.Size + info.Length * 2 <= MaxCacheSize) {
                        FileContentCache cache = getContent( info );
                        if (cache == null) { return null; }

                        file = new FoundFile( cache.Info, cache.Encoding );
                        return new StringReader( cache.Content );
                    }

                    FileEncodingCache encoding = getEncoding( info );
                    if (encoding == null) { return null; }

                    // with Azuki, looks following is just suffice
                    Encoding enc = Encoding.GetEncoding( encoding.Encoding, EncoderFallback.ReplacementFallback, DecoderFallback.ReplacementFallback );
                    file = new FoundFile( info, enc.WebName );
                    return new StreamReader( path, enc );
                }
                catch (Exception) {
                    return null;
                }
            }
        }
    }

    static class FileCache
    {
        public static FileCacheManager Instance { get { return cache; } }
        private static readonly FileCacheManager cache = new FileCacheManager();
    }
}
