#include <stdio.h>
#include <math.h>
#include <AR/matrix.h>

//#define     VZERO           1e-16
#define     EPS             DoubleToARVAL( 1e-6 )
#define     MAX_ITER        100
//#define     xmalloc(V,T,S)  { if( ((V) = (T *)malloc( sizeof(T) * (S) ))== NULL ) {printf("malloc error\n"); exit(1);} }

static int EX( ARMat *input, ARVec *mean );
static int CENTER( ARMat *inout, ARVec *mean );
static int PCA( ARMat *input, ARMat *output, ARVec *ev );
static int x_by_xt( ARMat *input, ARMat *output );
static int xt_by_x( ARMat *input, ARMat *output );
static int EV_create( ARMat *input, ARMat *u, ARMat *output, ARVec *ev );
static int QRM( ARMat *u, ARVec *ev );


/* === matrix definition ===

Input:
<---- clm (Data dimention)--->
[ 10  20  30 ] ^
[ 20  10  15 ] |
[ 12  23  13 ] row
[ 20  10  15 ] |(Sample number)
[ 13  14  15 ] v

Evec:
<---- clm (Eigen vector)--->
[ 10  20  30 ] ^
[ 20  10  15 ] |
[ 12  23  13 ] row
[ 20  10  15 ] |(Number of egen vec)
[ 13  14  15 ] v

Ev:
<---- clm (Number of eigen vector)--->
[ 10  20  30 ] eigen value

Mean:
<---- clm (Data dimention)--->
[ 10  20  30 ] mean value

=========================== */


int arMatrixPCA( ARMat *input, ARMat *evec, ARVec *ev, ARVec *mean )
{
	ARMat     *work;
	ARVAL  srow, sum;
	int     row, clm;
	int     check, rval;
	int     i;

	row = input->row;
	clm = input->clm;
	check = (row < clm)? row: clm;
	if( row < 2 || clm < 2 ) return(-1);
	if( evec->clm != input->clm || evec->row != check ) return(-1);
	if( ev->clm   != check )      return(-1);
	if( mean->clm != input->clm ) return(-1);

	work = arMatrixAllocDup( input );
	if( work == NULL ) return -1;

	srow = SqrtARVAL( IntToARVAL( row ) );
	if( EX( work, mean ) < 0 ) {
		arMatrixFree( work );
		return(-1);
	}
	if( CENTER( work, mean ) < 0 ) {
		arMatrixFree( work );
		return(-1);
	}
	for(i=0; i<row*clm; i++) 
		work->ml[i] = DivARVAL( work->ml[i], srow );

	rval = PCA( work, evec, ev );
	arMatrixFree( work );

	sum = 0;
	for( i = 0; i < ev->clm; i++ ) 
		sum += ev->vl[i];
	for( i = 0; i < ev->clm; i++ ) 
		ev->vl[i] = DivARVAL( ev->vl[i], sum );

	return( rval );
}

int arMatrixPCA2( ARMat *input, ARMat *evec, ARVec *ev )
{
	ARMat   *work;
	// double  srow; // unreferenced
	ARVAL  sum;
	int     row, clm;
	int     check, rval;
	int     i;

	row = input->row;
	clm = input->clm;
	check = (row < clm)? row: clm;
	if( row < 2 || clm < 2 ) return(-1);
	if( evec->clm != input->clm || evec->row != check ) return(-1);
	if( ev->clm   != check )      return(-1);

	work = arMatrixAllocDup( input );
	if( work == NULL ) return -1;

	/*
	srow = sqrt((double)row);
	for(i=0; i<row*clm; i++) work->m[i] /= srow;
	*/

	rval = PCA( work, evec, ev );
	arMatrixFree( work );

	sum = 0;
	for( i = 0; i < ev->clm; i++ ) sum += ev->vl[i];
	for( i = 0; i < ev->clm; i++ ) ev->vl[i] = DivARVAL( ev->vl[i], sum );

	return( rval );
}

static int PCA( ARMat *input, ARMat *output, ARVec *ev )
{
	ARMat     *u;
	ARVAL  *m1, *m2;
	int     row, clm, min;
	int     i, j;

	row = input->row;
	clm = input->clm;
	min = (clm < row)? clm: row;
	if( row < 2 || clm < 2 )        return(-1);
	if( output->clm != input->clm ) return(-1);
	if( output->row != min )        return(-1);
	if( ev->clm != min )            return(-1);

	u = arMatrixAlloc( min, min );
	if( u->row != min || u->clm != min ) return(-1);
	if( row < clm ) {
		if( x_by_xt( input, u ) < 0 ) { arMatrixFree(u); return(-1); }
	}
	else {
		if( xt_by_x( input, u ) < 0 ) { arMatrixFree(u); return(-1); }
	}

	if( QRM( u, ev ) < 0 ) { arMatrixFree(u); return(-1); }

	if( row < clm ) {
		if( EV_create( input, u, output, ev ) < 0 ) {
			arMatrixFree(u);
			return(-1);
		}
	}
	else{
		m1 = u->ml;
		m2 = output->ml;
		for( i = 0; i < min; i++){
			if( ev->vl[i] < 1 ) break;
			for( j = 0; j < min; j++ ) *(m2++) = *(m1++);
		}
		for( ; i < min; i++){
			ev->vl[i] = 0;
			for( j = 0; j < min; j++ ) *(m2++) = 0;
		}
	}

	arMatrixFree(u);

	return( 0 );
}

static int EX( ARMat *input, ARVec *mean )
{
	ARVAL  *m, *v;
	int     row, clm;
	int     i, j;

	row = input->row;
	clm = input->clm;
	if( row <= 0 || clm <= 0 )  return(-1);
	if( mean->clm != clm )      return(-1);

	for( i = 0; i < clm; i++ ) mean->vl[i] = 0;

	m = input->ml;
	for( i = 0; i < row; i++ ) {
		v = mean->vl;
		for( j = 0; j < clm; j++ )
			*(v++) += *(m++);
	}

	for( i = 0; i < clm; i++ )
		mean->vl[i] /= row;

	return(0);
}

static int CENTER( ARMat *inout, ARVec *mean )
{
	ARVAL  *m, *v;
	int     row, clm;
	int     i, j;

	row = inout->row;
	clm = inout->clm;
	if( mean->clm != clm ) return(-1);

	m = inout->ml;
	for( i = 0; i < row; i++ ) {
		v = mean->vl;
		for( j = 0; j < clm; j++ ) 
			*(m++) -= *(v++);
	}

	return(0);
}

static int x_by_xt( ARMat *input, ARMat *output )
{
	ARVAL  *in1, *in2, *out;
	int     row, clm;
	int     i, j, k;

	row = input->row;
	clm = input->clm;
	if( output->row != row || output->clm != row ) return(-1);

	out = output->ml;
	for( i = 0; i < row; i++ ) {
		for( j = 0; j < row; j++ ) {
			if( j < i ) {
				*out = output->ml[j*row+i];
			}
			else {
				in1 = &(input->ml[clm*i]);
				in2 = &(input->ml[clm*j]);
				*out = 0;
				for( k = 0; k < clm; k++ ) {
					*out += MulARVAL( *(in1++), *(in2++) );
				}
			}
			out++;
		}
	}

	return(0);
}

static int xt_by_x( ARMat *input, ARMat *output )
{
	ARVAL  *in1, *in2, *out;
	int     row, clm;
	int     i, j, k;

	row = input->row;
	clm = input->clm;
	if( output->row != clm || output->clm != clm ) return(-1);

	out = output->ml;
	for( i = 0; i < clm; i++ ) {
		for( j = 0; j < clm; j++ ) {
			if( j < i ) {
				*out = output->ml[j*clm+i];
			}
			else {
				in1 = &(input->ml[i]);
				in2 = &(input->ml[j]);
				*out = 0;
				for( k = 0; k < row; k++ ) {
					*out += MulARVAL( *in1, *in2 );
					in1 += clm;
					in2 += clm;
				}
			}
			out++;
		}
	}

	return(0);
}

static int EV_create( ARMat *input, ARMat *u, ARMat *output, ARVec *ev )
{
	ARVAL  *m, *m1, *m2;
	ARVAL  sum, work;
	int     row, clm;
	int     i, j, k;

	row = input->row;
	clm = input->clm;
	if( row <= 0 || clm <= 0 ) return(-1);
	if( u->row != row || u->clm != row ) return(-1);
	if( output->row != row || output->clm != clm ) return(-1);
	if( ev->clm != row ) return(-1);

	m = output->ml;
	for( i = 0; i < row; i++ ) {
		if( ev->vl[i] < 1 ) break;
		work = InvSqrtARVAL( AbsARVAL(ev->vl[i]) );
		for( j = 0; j < clm; j++ ) {
			sum = 0;
			m1 = &(u->ml[i*row]);
			m2 = &(input->ml[j]);
			for( k = 0; k < row; k++ ) {
				sum += MulARVAL( *m1, *m2 );
				m1++;
				m2 += clm;
			}
			*(m++) = MulARVAL( sum, work );
		}
	}
	for( ; i < row; i++ ) {
		ev->vl[i] = 0;
		for( j = 0; j < clm; j++ ) *(m++) = 0;
	}

	return(0);
}

static int QRM( ARMat *a, ARVec *dv )
{
	ARVec     *ev, ev1;
	ARVAL  w, t, s, x, y, c;
	ARVAL  *v1, *v2;
	int     dim, iter;
	int     i, j, k, h;

	dim = a->row;
	if( dim != a->clm || dim < 2 ) return(-1);
	if( dv->clm != dim ) return(-1);

	ev = arVecAlloc( dim );
	if( ev == NULL ) return(-1);

	ev1.clm = dim-1;
	ev1.vl = &(ev->vl[1]);
	if( arVecTridiagonalize( a, dv, &ev1 ) < 0 ) {
		arVecFree( ev );
		return(-1);
	}

	ev->vl[0] = 0;
	for( h = dim-1; h > 0; h-- ) {
		for ( j=h; j>0; j-- )
		{
			if ( AbsARVAL(ev->vl[j]) <= MulARVAL( EPS, AbsARVAL(dv->vl[j-1])+AbsARVAL(dv->vl[j]) ) )
				break;
		}
		if( j == h ) continue;

		iter = 0;
		do{
			iter++;
			if( iter > MAX_ITER ) break;

			w = (dv->vl[h-1] - dv->vl[h]) >> 1;
			t = MulARVAL( ev->vl[h], ev->vl[h] );
			s = SqrtARVAL( MulARVAL( w, w ) + t ); 
			if( w < 0 ) s = -s;
			x = dv->vl[j] - dv->vl[h] + DivARVAL( t, w+s );
			y = ev->vl[j+1];
			for( k = j; k < h; k++ ) {
				if( AbsARVAL(x) >= AbsARVAL(y) ) {
					if( AbsARVAL(x) > 1 ) {
						t = -DivARVAL( y, x );
						c = InvSqrtARVAL( MulARVAL(t,t) + IntToARVAL( 1 ) );
						s = MulARVAL( t, c );
					}
					else{
						c = IntToARVAL( 1 );
						s = 0;
					}
				}
				else{
					t = -DivARVAL( x, y );
					s = InvSqrtARVAL( MulARVAL(t,t) + IntToARVAL( 1 ) );
					c = MulARVAL( t, s );
				}
				w = dv->vl[k] - dv->vl[k+1];
				t = MulARVAL( (MulARVAL( w, s ) + 2 * MulARVAL( c, ev->vl[k+1] )), s );
				dv->vl[k]   -= t;
				dv->vl[k+1] += t;
				if( k > j) ev->vl[k] = MulARVAL( c, ev->vl[k] ) - MulARVAL( s, y );
				ev->vl[k+1] += MulARVAL( s, (MulARVAL( c, w ) - 2 * MulARVAL( s, ev->vl[k+1])) );

				// ƂŌ덷}
				ev->vl[ k+1 ] &= ~((1LL<<(REALBITS_ARVAL/2)) - 1);

				for( i = 0; i < dim; i++ ) {
					x = a->ml[k*dim+i];
					y = a->ml[(k+1)*dim+i];
					a->ml[k*dim+i]     = MulARVAL( c, x ) - MulARVAL( s, y );
					a->ml[(k+1)*dim+i] = MulARVAL( s, x ) + MulARVAL( c, y );
				}
				if( k < h-1 ) {
					x = ev->vl[k+1];
					y = -MulARVAL( s, ev->vl[k+2] );
					ev->vl[k+2] = MulARVAL( ev->vl[k+2], c );
				}
			}
		} while(AbsARVAL(ev->vl[h]) > MulARVAL( EPS, AbsARVAL(dv->vl[h-1])+AbsARVAL(dv->vl[h])) );
	}

	for( k = 0; k < dim-1; k++ ) {
		h = k;
		t = dv->vl[h];
		for( i = k+1; i < dim; i++ ) {
			if( dv->vl[i] > t ) {
				h = i;
				t = dv->vl[h];
			}
		}
		dv->vl[h] = dv->vl[k];
		dv->vl[k] = t;
		v1 = &(a->ml[h*dim]);
		v2 = &(a->ml[k*dim]);
		for( i = 0; i < dim; i++ ) {
			w = *v1;
			*v1 = *v2;
			*v2 = w;
			v1++;
			v2++;
		}
	}

	arVecFree( ev );
	return(0);
}
