﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.ServiceModel;
using System.Windows.Forms;
using FDK;

namespace SST
{
	[ServiceBehavior(InstanceContextMode = InstanceContextMode.Single)]	// サービスインターフェースをシングルスレッドで呼び出す。
	class StrokeStyleT : SST.IStrokeStyleTService
	{
		// グローバルリソース (static) 

		public static SST.フォルダ フォルダ
		{
			get { return StrokeStyleT._フォルダ; }
		}

		public static FDK.メディア.サウンド.WASAPI.Device サウンドデバイス
		{
			get { return StrokeStyleT._サウンドデバイス; }
		}

		public static System.Random 乱数
		{
			get { return StrokeStyleT._乱数; }
		}

		public static SST.ユーザ.ユーザ管理 ユーザ管理
		{
			get { return StrokeStyleT._ユーザ管理; }
		}

		public static SST.曲.曲ツリー管理 曲ツリー管理
		{
			get { return StrokeStyleT._曲ツリー管理; }
		}

		public static SSTFormat.スコア 演奏スコア
		{
			get;
			set;
		} = null;

		public static SST.設定.Config Config
		{
			get { return StrokeStyleT._Config; }
		}

		public static bool ビュアーモードである
		{
			get;
			set;
		} = false;

		public static bool ビュアーモードではない
		{
			get { return !StrokeStyleT.ビュアーモードである; }
			set { StrokeStyleT.ビュアーモードである = !value; }
		}

		public static SST.ViewerMessage 最後に取得したビュアーメッセージ
		{
			get;
			protected set;
		} = null;

		public static FDK.入力.Keyboard キーボード入力
		{
			get { return StrokeStyleT._キーボード入力; }
		}

		public static FDK.入力.MidiIn MIDI入力
		{
			get { return StrokeStyleT._MIDI入力; }
		}

		public static void すべての入力デバイスをポーリングする()
		{
			// hack: 追加の入力デバイスができたら、ここにポーリングコードを追加すること。
			StrokeStyleT.キーボード入力?.ポーリングする();
			StrokeStyleT.MIDI入力?.ポーリングする();
		}

		static StrokeStyleT()
		{
			// フォルダ変数を真っ先に登録する。（ほかのメンバのコンストラクタでフォルダ変数を利用できるようにするため。）
			StrokeStyleT._フォルダ = new SST.フォルダ();
			SST.フォルダ.フォルダ変数を追加する( "Static", StrokeStyleT.フォルダ.StaticFolder );
			SST.フォルダ.フォルダ変数を追加する( "AppData", StrokeStyleT.フォルダ.AppDataFolder );
			SST.フォルダ.フォルダ変数を追加する( "User", null );
		}

		public StrokeStyleT( string[] args )
		{
			#region " ビュアーモードかどうかを確認する。"
			//----------------
			foreach( var arg in args )
			{
				if( ( "-v" == arg.ToLower() ) || ( "-viewer" == arg.ToLower() ) )
				{
					StrokeStyleT.ビュアーモードである = true;
					break;
				}
			}
			//----------------
			#endregion

			this._State = ApplicationState.起動;

			this._MainForm = new RenderForm();
			this._MainForm.BackColor = System.Drawing.Color.Black;
			this._MainForm.AllowUserResizing = false;	// ユーザはフォームサイズを変更できない。
			this._MainForm.UserResized += _フォームサイズが変更された;
			this._MainForm.FormClosing += ( sender, e ) => {
				this._終了する();
			};
		}

		/// <summary>
		///		アプリのメインエントリ。
		/// </summary>
		public void Run()
		{
			Debug.Assert( null != this._MainForm );

			SharpDX.Windows.RenderLoop.Run( this._MainForm, () => {

				// アプリケーションの状態に応じて処理分岐。
				switch( this._State )
				{
					case ApplicationState.起動:
						this._起動する();
						this._State = ApplicationState.初期化;
						break;  // 初期化に移る前に、一度ウィンドウメッセージ処理を行わせて、画面を再描画させる。

					case ApplicationState.初期化:
						this._初期化する();
						this._State = ApplicationState.進行描画;
						break;

					case ApplicationState.進行描画:
						this._進行描画する();
						if( this._State == ApplicationState.終了 )
							this._終了する();
						break;

					case ApplicationState.終了:
						// 何もしない
						break;
				}

			} );
		}

		#region " WCF サービスインターフェースの実装 "
		//----------------
		// ・このサービスインターフェースは、シングルスレッド（GUIスレッド）で同期実行される。（StrokeStyleT クラスの ServiceBehavior属性を参照。）
		// ・このサービスホストはシングルトンであり、すべてのクライアントセッションは同一（単一）のサービスインスタンスへ接続される。（Program.Main() を参照。）

		/// <summary>
		///		曲を読み込み、演奏を開始する。
		/// </summary>
		/// <remarks>
		///		ビュアーモードのときのみ有効。
		/// </remarks>
		/// <param name="path">
		///		曲ファイルパス
		///	</param>
		/// <param name="startPart">
		///		演奏開始小節番号(0～)
		///	</param>
		/// <param name="drumsSound">
		///		ドラムチップ音を発声させるなら true。
		/// </param>
		public void ViewerPlay( string path, int startPart = 0, bool drumsSound = true )
		{
			if( StrokeStyleT.ビュアーモードではない )
				return;

			this._ビュアーメッセージキュー.Enqueue( new ViewerMessage() {
				種別 = ViewerMessageType.演奏開始,
				曲ファイルパス = path,
				演奏開始小節番号 = startPart,
				ドラムチップ発声 = drumsSound,
			} );
		}

		/// <summary>
		///		現在の演奏を停止する。
		/// </summary>
		/// <remarks>
		///		ビュアーモードのときのみ有効。
		/// </remarks>
		public void ViewerStop()
		{
			if( StrokeStyleT.ビュアーモードではない )
				return;

			this._ビュアーメッセージキュー.Enqueue( new ViewerMessage() {
				種別 = ViewerMessageType.演奏停止,
			} );
		}

		/// <summary>
		///		サウンドデバイスの発声遅延[ms]を返す。
		/// </summary>
		/// <returns>
		///		遅延量[ms]
		///	</returns>
		public float GetSoundDelay()
		{
			if( StrokeStyleT.ビュアーモードではない )
				return 0f;

			return ( null != StrokeStyleT.サウンドデバイス ) ? (float) ( StrokeStyleT.サウンドデバイス.遅延sec * 1000.0 ) : 0f;
		}
		//----------------
		#endregion

		private enum ApplicationState { 起動, 初期化, 進行描画, 終了 }

		private ApplicationState _State;

		private SST.RenderForm _MainForm = null;

		private SharpDX.Size2F _設計画面サイズdpx = SharpDX.Size2F.Empty;

		private ConcurrentQueue<ViewerMessage> _ビュアーメッセージキュー = new ConcurrentQueue<ViewerMessage>();

		private FDK.メディア.デバイスリソース _デバイスリソース = null;

		private SST.ステージ.ステージ _最初のダミーステージ = null;

		private SST.ステージ.起動.起動ステージ _起動ステージ = null;

		private SST.ステージ.タイトル.タイトルステージ _タイトルステージ = null;

		private SST.ステージ.ログイン.ログインステージ _ログインステージ = null;

		private SST.ステージ.選曲.選曲ステージ _選曲ステージ = null;

		private SST.ステージ.曲読込.曲読込ステージ _曲読込ステージ = null;

		private SST.ステージ.演奏.演奏ステージ _演奏ステージ = null;

		private SST.ステージ.結果.結果ステージ _結果ステージ = null;

		private SST.ステージ.ステージ _現在のステージ = null;

		private static SST.フォルダ _フォルダ = null;

		private static FDK.入力.Keyboard _キーボード入力 = null;

		private static FDK.入力.MidiIn _MIDI入力 = null;

		private static FDK.メディア.サウンド.WASAPI.Device _サウンドデバイス = null;

		private static readonly System.Random _乱数 = new Random( DateTime.Now.Millisecond );

		private static SST.ユーザ.ユーザ管理 _ユーザ管理 = null;

		private static SST.曲.曲ツリー管理 _曲ツリー管理 = null;

		private static SST.設定.Config _Config = null;

		private void _起動する()
		{
			FDK.Log.現在のスレッドに名前をつける( "Render" );
			FDK.Log.BeginInfo( $"{FDK.Utilities.現在のメソッド名}" );

			#region " コンフィグ を初期化する。"
			//----------------
			StrokeStyleT._Config = new 設定.Config();
			StrokeStyleT._Config.ConfigXmlを読み込む();
			//----------------
			#endregion

			#region " コンフィグで指定されたウィンドウサイズに変更する。"
			//----------------
			this._MainForm.ClientSize = new System.Drawing.Size( StrokeStyleT.Config.物理画面サイズpx.Width, StrokeStyleT.Config.物理画面サイズpx.Height );
			//----------------
			#endregion
			#region " フォームタイトルと設計画面サイズを設定する。"
			//----------------
			this._MainForm.Text = $"StrokeStyle<T> {System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString()}";
			if( StrokeStyleT.ビュアーモードである )
				this._MainForm.Text += " (Viewer)";

			// 設計画面サイズは、フォームサイズ（物理画面サイズ）とは独立して固定。
			this._設計画面サイズdpx = new SharpDX.Size2F( 1920f, 1080f );
			//----------------
			#endregion

			FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		private void _初期化する()
		{
			FDK.Log.BeginInfo( $"{FDK.Utilities.現在のメソッド名}" );

			// 開始条件チェック。
			Debug.Assert( null == this._デバイスリソース, "デバイスリソースの作成前であること。" );

			StrokeStyleT._ユーザ管理 = new ユーザ.ユーザ管理();
			StrokeStyleT._曲ツリー管理 = new 曲.曲ツリー管理();

			#region " 高解像度タイマを使えないならエラー。"
			//-----------------
			if( false == System.Diagnostics.Stopwatch.IsHighResolution )
				throw new SSTException( "このシステムは、高解像度タイマをサポートしていません。" );
			//-----------------
			#endregion
			#region " MediaFoundation を起動する。"
			//-----------------
			SharpDX.MediaFoundation.MediaManager.Startup();
			//-----------------
			#endregion
			#region " Sleep 精度を上げる。"
			//-----------------
			Win32.timeBeginPeriod( 1 );
			//-----------------
			#endregion

			#region " デバイスリソースを作成する。"
			//----------------
			FDK.Log.Info( $"設計画面サイズ: {this._設計画面サイズdpx}" );
			FDK.Log.Info( $"物理画面サイズ: {this._MainForm.ClientSize}" );
			this._デバイスリソース = new FDK.メディア.デバイスリソース( this._設計画面サイズdpx );
			this._デバイスリソース.すべてのリソースを作成する( this._MainForm.ClientSize, this._MainForm.Handle );
			//----------------
			#endregion

			#region " ステージを生成する。"
			//----------------
			this._最初のダミーステージ = new ステージ.ステージ();
			this._起動ステージ = new ステージ.起動.起動ステージ();
			this._タイトルステージ = new ステージ.タイトル.タイトルステージ();
			this._ログインステージ = new ステージ.ログイン.ログインステージ();
			this._選曲ステージ = new ステージ.選曲.選曲ステージ();
			this._曲読込ステージ = new ステージ.曲読込.曲読込ステージ();
			this._演奏ステージ = new ステージ.演奏.演奏ステージ();
			this._結果ステージ = new ステージ.結果.結果ステージ();

			// 外部依存アクションを接続する。
			this._曲読込ステージ.読込曲のファイルパスを取得する = () => ( ( StrokeStyleT.曲ツリー管理.現在選択されているノード as SST.曲.MusicNode )?.sstfファイルパス );
			this._結果ステージ.演奏ステージインスタンスを取得する = () => ( this._演奏ステージ );
			this._結果ステージ.BGMを停止する = () => { this._演奏ステージ.BGMを停止する(); };
			//----------------
			#endregion
			#region " ユーザを初期化する。"
			//-----------------
			// Users.xml を読み込む。
			StrokeStyleT.ユーザ管理.UsersXmlを読み込む();

			// ユーザ別の初期化。
			foreach( var ユーザ in StrokeStyleT.ユーザ管理.ユーザリスト )
				ユーザ.SourcesXmlを読み込む();
			//-----------------
			#endregion
			#region " サウンドデバイスを初期化する。"
			//----------------
			StrokeStyleT._サウンドデバイス = new FDK.メディア.サウンド.WASAPI.Device( CSCore.CoreAudioAPI.AudioClientShareMode.Shared );
			//----------------
			#endregion
			#region " キーボード入力 を初期化する。"
			//-----------------
			StrokeStyleT._キーボード入力 = new FDK.入力.Keyboard( this._MainForm.Handle );
			//-----------------
			#endregion
			#region " MIDI入力 を初期化する。"
			//-----------------
			StrokeStyleT._MIDI入力 = new FDK.入力.MidiIn();
			//-----------------
			#endregion

			FDK.Log.Info( "最初のダミーステージを開始します。" );
			this._現在のステージ = this._最初のダミーステージ;

			//#warning 全画面モード切替えを仮実装。
			this._MainForm.KeyDown += ( target, arg ) => {

				// F11 → 画面モードの切り替え
				if( arg.KeyCode == System.Windows.Forms.Keys.F11 )
				{
					this._全画面モードとウィンドウモードを切り替える();
					arg.Handled = true;
				}
			};

			// 終了条件チェック。
			Debug.Assert( SharpDX.Size2F.Empty != this._設計画面サイズdpx, "設計画面サイズが設定されていません。" );

			FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		private void _終了する()
		{
			FDK.Log.BeginInfo( $"{FDK.Utilities.現在のメソッド名}" );
			Debug.Assert( null != this._デバイスリソース, "デバイスリソースが解放される前であること。" );

			this._State = ApplicationState.終了;

			#region " ステージを終了し、解放する。"
			//----------------
			if( ( null != this._現在のステージ ) && ( this._現在のステージ.活性化している ) )      // 念のため
				this._現在のステージ.非活性化する( this._デバイスリソース );

			this._最初のダミーステージ = null;
			this._起動ステージ = null;
			this._タイトルステージ = null;
			this._ログインステージ = null;
			this._選曲ステージ = null;
			this._曲読込ステージ = null;
			this._演奏ステージ.BGMを停止する();
			this._演奏ステージ?.BGMのキャッシュを解放する();	// BGMのキャッシュ（デコード済み WaveSource）を解放。
			this._演奏ステージ = null;
			this._結果ステージ = null;
			//----------------
			#endregion

			#region " ユーザ情報を保存する。"
			//----------------
			// 要素が増えたときのため、変更がなくても保存する。
			StrokeStyleT.ユーザ管理.UsersXmlを保存する();
			//----------------
			#endregion

			#region " デバイスリソースを解放する。"
			//----------------
			FDK.Utilities.解放する( ref this._デバイスリソース );
			//----------------
			#endregion

			#region " MIDI入力デバイスを解放する。"
			//-----------------
			FDK.Log.Info( "MIDI入力デバイスを解放します。" );
			FDK.Utilities.解放する( ref StrokeStyleT._MIDI入力 );
			//-----------------
			#endregion
			#region " キーボード入力デバイスを解放する。"
			//-----------------
			FDK.Log.Info( "キーボード入力デバイスを解放します。" );
			FDK.Utilities.解放する( ref StrokeStyleT._キーボード入力 );
			//-----------------
			#endregion
			#region " サウンドデバイスを解放する。"
			//----------------
			FDK.Log.Info( "サウンドデバイスを解放します。" );
			FDK.Utilities.解放する( ref StrokeStyleT._サウンドデバイス );
			//----------------
			#endregion
			
			#region " コンフィグを保存し、解放する。"
			//----------------
			FDK.Log.Info( "コンフィグを解放します。" );
			StrokeStyleT._Config.ConfigXmlを保存する();
			StrokeStyleT._Config = null;
			//----------------
			#endregion
			#region " フォームを閉じる。"
			//----------------
			FDK.Utilities.解放する( ref this._MainForm );
			//----------------
			#endregion

			#region " MediaFoundation を終了する。"
			//-----------------
			SharpDX.MediaFoundation.MediaManager.Shutdown();
			//-----------------
			#endregion

			FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		private void _進行描画する()
		{
			#region " D3Dデバイスが消失していれば再構築する。"
			//----------------
			bool 異常発生 = false;
			this._デバイスリソース.D3Dデバイスが消失していれば再構築する( out 異常発生 );

			// 再構築不可能な異常の場合は、即終了する。
			if( 異常発生 )
			{
				this._State = ApplicationState.終了;
				return;
			}
			//----------------
			#endregion
			#region " 描画の前処理を行う。"
			//----------------
			{
				var d3dDevice = (SharpDX.Direct3D11.Device) null;
				using( var d3dLock = new FDK.同期.AutoD3DDeviceLock( this._デバイスリソース.DXGIDeviceManager, out d3dDevice ) )
				using( d3dDevice )
				using( var d3dContext = d3dDevice.ImmediateContext )
				{
					// 既定のD3Dレンダーターゲットビューを黒でクリアする。
					d3dContext.ClearRenderTargetView( this._デバイスリソース.D3DRenderTargetView, SharpDX.Color4.Black );

					// 深度バッファを 1.0f でクリアする。
					d3dContext.ClearDepthStencilView(
						this._デバイスリソース.D3DDepthStencilView,
						SharpDX.Direct3D11.DepthStencilClearFlags.Depth,
						depth: 1.0f,
						stencil: 0 );
				}
			}
			//----------------
			#endregion
			#region " 現在のステージを進行描画する。"
			//----------------
			this._現在のステージ?.進行描画する( this._デバイスリソース );
			//----------------
			#endregion
			#region " スワップチェーンを表示する。"
			//----------------
			if( StrokeStyleT.Config.垂直帰線待ちを行う )
			{
				// We recommend that you use Flush when the CPU waits for an arbitrary amount of time
				// (such as when you call the Sleep function). 
				// → https://msdn.microsoft.com/ja-jp/library/windows/desktop/ff476425(v=vs.85).aspx

				var d3dDevice = (SharpDX.Direct3D11.Device) null;
				using( var d3dLock = new FDK.同期.AutoD3DDeviceLock( this._デバイスリソース.DXGIDeviceManager, out d3dDevice ) )
				using( d3dDevice )
				using( var d3dContext = d3dDevice.ImmediateContext )
				{
					d3dContext.Flush();
				}
			}

			this._デバイスリソース.SwapChain1.Present(
				( StrokeStyleT.Config.垂直帰線待ちを行う ) ? 1 : 0,
				SharpDX.DXGI.PresentFlags.None );
			//----------------
			#endregion

			#region " ステージの状態をチェックし、必要あれば遷移またはアプリを終了する。また、必要あればビュアーメッセージキューの処理も行う。"
			//----------------
			if( null != this._現在のステージ )
			{
				switch( this._現在のステージ.GetType().Name )
				{
					case nameof( ステージ.ステージ ):
						#region " ビュアーモード → AutoPlayerでログインして演奏ステージへ。"
						//----------------
						if( StrokeStyleT.ビュアーモードである )
						{
							// ビュアー用ユーザでログインする。
							FDK.Log.Info( "ビュアーモード: AutoPlayer ユーザでログインします。" );
							this._ログインする( Properties.Resources.AUTOPLAYER );

							// 曲読込ステージ向けの初期設定。
							StrokeStyleT.曲ツリー管理.現在の管理対象ツリー = null;
							StrokeStyleT.曲ツリー管理.現在選択されているノード = null;
							
							// 演奏ステージ向けの初期設定。
							StrokeStyleT.演奏スコア = null;

							// 演奏ステージへ。
							this._演奏ステージ.活性化する( this._デバイスリソース );
							this._現在のステージ = this._演奏ステージ;
						}
						//----------------
						#endregion
						#region " 通常モード → 起動ステージへ。"
						//----------------
						else
						{
							this._起動ステージ.活性化する( this._デバイスリソース );
							this._現在のステージ = this._起動ステージ;
						}
						//----------------
						#endregion
						break;

					case nameof( ステージ.起動.起動ステージ ):
						#region " 終了 → タイトルステージへ。"
						//---------------
						if( this._起動ステージ.現在のフェーズ == ステージ.起動.起動ステージ.フェーズ.終了 )
						{
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = this._タイトルステージ;
							this._現在のステージ.活性化する( this._デバイスリソース );
						}
						//---------------
						#endregion
						break;

					case nameof( ステージ.タイトル.タイトルステージ ):
						#region " 確定 → ログインステージへ。"
						//---------------
						if( this._タイトルステージ.現在のフェーズ == ステージ.タイトル.タイトルステージ.フェーズ.確定 )
						{
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = this._ログインステージ;
							this._現在のステージ.活性化する( this._デバイスリソース );
						}
						//---------------
						#endregion
						#region " キャンセル → アプリを終了する。"
						//---------------
						else if( this._タイトルステージ.現在のフェーズ == ステージ.タイトル.タイトルステージ.フェーズ.キャンセル )
						{
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = null;
							this._State = ApplicationState.終了;
						}
						//---------------
						#endregion
						break;

					case nameof( ステージ.ログイン.ログインステージ ):
						#region " 確定 → ログイン処理を行って、選曲ステージへ。"
						//---------------
						if( this._ログインステージ.現在のフェーズ == ステージ.ログイン.ログインステージ.フェーズ.確定 )
						{
							var user = StrokeStyleT.ユーザ管理.現在選択されているユーザ;

							if( null != user )
							{
								foreach( var path in user.曲の検索元フォルダパスのリスト )
									SST.曲.曲ツリー管理.フォルダから曲を再帰的に検索して子ノードリストに追加する( user.曲ツリーのルートノード, path );

								StrokeStyleT.曲ツリー管理.現在の管理対象ツリー = StrokeStyleT.ユーザ管理.現在選択されているユーザ.曲ツリーのルートノード;
							}

							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = this._選曲ステージ;
							this._現在のステージ.活性化する( this._デバイスリソース );
						}
						//---------------
						#endregion
						#region " キャンセル → タイトルステージへ。"
						//---------------
						else if( this._ログインステージ.現在のフェーズ == ステージ.ログイン.ログインステージ.フェーズ.キャンセル )
						{
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = this._タイトルステージ;
							this._現在のステージ.活性化する( this._デバイスリソース );
						}
						//---------------
						#endregion
						break;

					case nameof( ステージ.選曲.選曲ステージ ):
						#region " 曲確定 → 曲読込ステージへ。"
						//---------------
						if( this._選曲ステージ.現在のフェーズ == ステージ.選曲.選曲ステージ.フェーズ.曲確定 )
						{
							// 曲ノードが選択されていることを確認。
							Trace.Assert( null != StrokeStyleT.曲ツリー管理.現在選択されているノード, "[バグあり] 選択曲が null です。" );
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = this._曲読込ステージ;
							this._現在のステージ.活性化する( this._デバイスリソース );
						}
						//---------------
						#endregion
						#region " キャンセル → アプリを終了する。"
						//---------------
						else if( this._選曲ステージ.現在のフェーズ == ステージ.選曲.選曲ステージ.フェーズ.キャンセル )
						{
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = null;
							this._State = ApplicationState.終了;
						}
						//---------------
						#endregion
						break;

					case nameof( ステージ.曲読込.曲読込ステージ ):
						#region " 終了 → 演奏ステージへ。"
						//--------------------
						if( this._曲読込ステージ.現在のフェーズ == ステージ.曲読込.曲読込ステージ.フェーズ.終了 )
						{
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = this._演奏ステージ;
							this._現在のステージ.活性化する( this._デバイスリソース );
						}
						//--------------------
						#endregion
						break;

					case nameof( ステージ.演奏.演奏ステージ ):
						if( StrokeStyleT.ビュアーモードである )
						{
							// (A) ビュアーモード

							#region " ビュアーメッセージがあれば処理する。"
							//----------------
							var msg = this._ビュアーメッセージを取得する();

							StrokeStyleT.最後に取得したビュアーメッセージ = msg;	// 保存する。（演奏ステージに対して公開する）

							if( null != msg )
							{
								FDK.Log.Info( msg.ToString() );

								#region " (A) 演奏開始 "
								//----------------
								if( msg.種別 == ViewerMessageType.演奏開始 &&
									( this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.ビュアーメッセージ待機中 ||
									  this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.演奏中 ) )
								{
									try
									{
										// MusicNode を作成し、現在選択されているノードとして登録する。
										StrokeStyleT.曲ツリー管理.現在選択されているノード = new 曲.MusicNode( msg.曲ファイルパス );

										// 演奏を停止する。
										this._演奏ステージ.非活性化する( this._デバイスリソース );
										this._演奏ステージ.BGMを停止する();

										// 曲読込ステージへ。
										this._現在のステージ = this._曲読込ステージ;
										this._現在のステージ.活性化する( this._デバイスリソース );
									}
									catch
									{
										FDK.Log.ERROR( $"MusicNode の作成に失敗しました。" );
									}
								}
								//----------------
								#endregion
								#region " (B) 演奏停止 "
								//----------------
								else if( msg.種別 == ViewerMessageType.演奏停止 && (
									this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.ビュアーメッセージ待機中 ||
									this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.演奏中 )
									)
								{
									this._演奏ステージ.演奏を停止する();
									this._演奏ステージ.現在のフェーズ.Value = ステージ.演奏.演奏ステージ.フェーズ.ビュアーメッセージ待機中;
								}
								//----------------
								#endregion
								#region " その他 "
								//----------------
								else
								{
									// その他のメッセージは無視。
								}
								//----------------
								#endregion
							}
							//----------------
							#endregion
							#region " クリア/キャンセル → メッセージ待機へ。"
							//----------------
							if( this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.クリアor失敗 ||
								this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.キャンセル )
							{
								// フェーズのみ変更。BGM は止めない。
								this._演奏ステージ.現在のフェーズ.Value = ステージ.演奏.演奏ステージ.フェーズ.ビュアーメッセージ待機中;
							}
							//----------------
							#endregion
						}
						else
						{
							// (B) 通常モード

							#region " 演奏終了 → 結果ステージへ。"
							//--------------------
							if( this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.クリアor失敗 )
							{
								this._現在のステージ.非活性化する( this._デバイスリソース );
								this._現在のステージ = this._結果ステージ;
								this._現在のステージ.活性化する( this._デバイスリソース );
							}
							//--------------------
							#endregion
							#region " キャンセル → 選曲ステージへ。"
							//--------------------
							if( this._演奏ステージ.現在のフェーズ.Value == ステージ.演奏.演奏ステージ.フェーズ.キャンセル )
							{
								this._現在のステージ.非活性化する( this._デバイスリソース );
								this._現在のステージ = this._選曲ステージ;
								this._現在のステージ.活性化する( this._デバイスリソース );
							}
							//--------------------
							#endregion
						}
						break;

					case nameof( ステージ.結果.結果ステージ ):
						#region " 終了 → 選曲ステージへ。"
						//--------------------
						if( this._結果ステージ.現在のフェーズ == ステージ.結果.結果ステージ.フェーズ.終了 )
						{
							this._現在のステージ.非活性化する( this._デバイスリソース );
							this._現在のステージ = this._選曲ステージ;
							this._現在のステージ.活性化する( this._デバイスリソース );
						}
						//--------------------
						#endregion
						break;
				}
			}
			//----------------
			#endregion
		}

		private void _デバイス依存リソースを解放する()
		{
			FDK.Log.BeginInfo( $"{FDK.Utilities.現在のメソッド名}" );
			Debug.Assert( null != this._デバイスリソース );  // 解放前であること。

			this._現在のステージ?.デバイス依存リソースを解放する( this._デバイスリソース );

			FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		private void _デバイス依存リソースを再構築する()
		{
			FDK.Log.BeginInfo( $"{FDK.Utilities.現在のメソッド名}" );
			Debug.Assert( null != this._デバイスリソース );  // 再生成済みであること。

			this._現在のステージ?.デバイス依存リソースを作成する( this._デバイスリソース );

			FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		private void _フォームサイズが変更された( object sender, EventArgs e )
		{
			FDK.Log.BeginInfo( $"{FDK.Utilities.現在のメソッド名}" );
			FDK.Log.Info( $"新しいクライアントサイズ = {this._MainForm.ClientSize}" );

			#region " 実行条件チェック。"
			//----------------
			if( null == this._デバイスリソース )
			{
				FDK.Log.Info( " まだ初期化されてないので、何もしません。" );
				FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
				return;
			}
			if( this._MainForm.WindowState == FormWindowState.Minimized )
			{
				FDK.Log.Info( "最小化されました。" );
				FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
				return; // 何もしない
			}
			if( this._State != ApplicationState.進行描画 )
			{
				FDK.Log.Info( " アプリケーションの状態が進行描画じゃないので、何もしません。" );
				FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
				return;
			}
			//----------------
			#endregion

			Debug.Assert( null != this._デバイスリソース, "デバイスリソースが作成済みであること。" );

			// 現在の画面モードを取得しておく。（Alt+TABなど、勝手に全画面を解除されることもあるので。）
			SharpDX.Mathematics.Interop.RawBool fullscreen;
			SharpDX.DXGI.Output outputTarget;
			this._デバイスリソース.SwapChain1.GetFullscreenState( out fullscreen, out outputTarget );
			this._MainForm.IsFullscreen = fullscreen;
			outputTarget?.Dispose();
			FDK.Log.Info( $"現在、全画面モードである = {this._MainForm.IsFullscreen}" );

			// (1) リソースを解放して、
			this._デバイス依存リソースを解放する();
			this._デバイスリソース.サイズに依存するリソースを解放する();

			// (2) 物理画面サイズを変更して、
			this._デバイスリソース.物理画面サイズpx = new SharpDX.Size2F( this._MainForm.ClientSize.Width, this._MainForm.ClientSize.Height );

			// (3) リソースを再構築する。
			this._デバイスリソース.サイズに依存するリソースを作成する();
			this._デバイス依存リソースを再構築する();

			FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		private void _全画面モードとウィンドウモードを切り替える()
		{
			FDK.Log.BeginInfo( $"{FDK.Utilities.現在のメソッド名}" );

			this._MainForm.IsFullscreen = !this._MainForm.IsFullscreen;   // 切り替え
			this._デバイスリソース.SwapChain1.SetFullscreenState( this._MainForm.IsFullscreen, null );

			FDK.Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		private void _ログインする( string ユーザ名 )
		{
			StrokeStyleT.ユーザ管理.ユーザを選択する( ユーザ名 );
			FDK.Log.Info( $"ユーザが選択されました。[{StrokeStyleT.ユーザ管理.現在選択されているユーザ.名前}]" );
		}

		/// <returns>
		///		メッセージがない場合は null を返す。
		///	</returns>
		private ViewerMessage _ビュアーメッセージを取得する()
		{
			var msg = (ViewerMessage) null;

			if( StrokeStyleT.ビュアーモードである &&
				( 0 < this._ビュアーメッセージキュー.Count ) &&
				this._ビュアーメッセージキュー.TryDequeue( out msg ) )
			{
				return msg;
			}

			return null;
		}
	}
}
