﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using CSCore;

namespace FDK.メディア.サウンド.WASAPI
{
	/// <summary>
	///		指定されたメディアファイル（動画, 音楽）をデコードして、CSCore.IWaveSource オブジェクトを生成する。
	/// </summary>
	public class DecodedWaveSource : CSCore.IWaveSource
	{
		/// <summary>
		///		シークは常にサポートする。
		/// </summary>
		public bool CanSeek => ( true );

		/// <summary>
		///		デコード後のオーディオデータの長さ[byte]。
		/// </summary>
		public long Length
		{
			get { return this._EncodedWaveData.Length; }
		}

		/// <summary>
		///		現在の位置。
		///		先頭からのオフセット[byte]で表す。
		/// </summary>
		public long Position
		{
			get { return this._Position; }
			set
			{
				if( ( 0 > value ) || ( ( this.Length > 0 ) && ( this.Length <= value ) ) )	// Length == 0 なら例外は出さない
					throw new ArgumentOutOfRangeException();

				this._Position = value;
			}
		}

		/// <summary>
		///		デコード後のオーディオデータのフォーマット。
		/// </summary>
		public CSCore.WaveFormat WaveFormat
		{
			get;
			protected set;
		}

		public string Path { get; } = null;

		/// <summary>
		///		メディアファイル（動画、音楽）をデコードする。
		/// </summary>
		public DecodedWaveSource( string path, CSCore.WaveFormat targetFormat )
		{
			this.Path = path;

			// ISampleSource は IWaveSource を 32bit-float に変換して出力する仕様なので、
			// 最初からその形式でデコードして ISampleSource.Read() の変換負荷を下げる。
			this.WaveFormat = new CSCore.WaveFormat(
				targetFormat.SampleRate,	// サンプルレートと
				32, 
				targetFormat.Channels,		// チャンネルは、指定されたものを使う。
				AudioEncoding.IeeeFloat );

			this._初期化する( path );
		}

		public void Dispose()
		{
			this._解放する();
		}

		/// <summary>
		///		連続した要素を読み込み、this.Position を読み込んだ要素の数だけ進める。
		/// </summary>
		/// <param name="buffer">
		///		読み込んだ要素を格納するための配列。
		///		このメソッドから戻ると、buffer には offset ～ (offset + count - 1) の数の要素が格納されている。
		/// </param>
		/// <param name="offset">
		///		buffer に格納を始める位置。
		/// </param>
		/// <param name="count">
		///		読み込む最大の要素数。
		/// </param>
		/// <returns>
		///		buffer に読み込んだ要素の総数。
		/// </returns>
		public int Read( byte[] buffer, int offset, int count )
		{
			// 音がめちゃくちゃになるとうざいので、このメソッド内では例外を出さないこと。

			if( ( null == this._EncodedWaveData ) && ( null == buffer ) )
				return 0;

			// offset は、0～buffer.Length-1 に収める。
			offset = Math.Max( 0, Math.Min( buffer.Length - 1, offset ) );

			// count は、_EncodeWaveData.Length-Position, buffer.Length-offset, count のうちの最小値とする。
			count = Math.Min( Math.Min( this._EncodedWaveData.Length - (int) this._Position, count ), buffer.Length - offset );

			if( 0 < count )
			{
				Array.Copy(
					sourceArray: this._EncodedWaveData,
					sourceIndex: this._Position,
					destinationArray: buffer,
					destinationIndex: offset,
					length: count );

				this._Position += count;
			}

			return count;
		}

		private CSCore.MediaFoundation.MFMediaType _MediaType = null;

		private byte[] _EncodedWaveData = null;

		private long _Position = 0;

		private void _初期化する( string path )
		{
			try
			{
				// SourceReader は、SharpDX ではなく CSCore のものを使う。（MediaType から WaveFormat に一発で変換できるので。）
				using( var sourceReader = new CSCore.MediaFoundation.MFSourceReader( path ) )
				using( var waveStream = new System.IO.MemoryStream() )
				{
					#region " 最初のオーディオストリームを選択し、その他のすべてのストリームを非選択にする。"
					//----------------
					sourceReader.SetStreamSelection( (int) SharpDX.MediaFoundation.SourceReaderIndex.AllStreams, false );
					sourceReader.SetStreamSelection( (int) SharpDX.MediaFoundation.SourceReaderIndex.FirstAudioStream, true );
					//----------------
					#endregion
					#region " デコード後フォーマットを持つメディアタイプを作成し、SourceReader に登録する。"
					//----------------
					using( var partialMediaType = CSCore.MediaFoundation.MFMediaType.FromWaveFormat( this.WaveFormat ) )	// WaveFormatEx にも対応。
					{
						// 作成したメディアタイプを sourceReader にセットする。必要なデコーダが見つからなかったら、ここで例外が発生する。
						sourceReader.SetCurrentMediaType( (int) SharpDX.MediaFoundation.SourceReaderIndex.FirstAudioStream, partialMediaType );

						// 完成されたメディアタイプを取得する。
						this._MediaType = sourceReader.GetCurrentMediaType( (int) SharpDX.MediaFoundation.SourceReaderIndex.FirstAudioStream );

						// メディアタイプからフォーマットを取得する。（同じであるはずだが念のため）
						this.WaveFormat = this._MediaType.ToWaveFormat( CSCore.MediaFoundation.MFWaveFormatExConvertFlags.Normal );

						// 最初のオーディオストリームが選択されていることを保証する。
						sourceReader.SetStreamSelection( (int) SharpDX.MediaFoundation.SourceReaderIndex.FirstAudioStream, true );
					}
					//----------------
					#endregion
					#region " sourceReader からサンプルを取得してデコードし、waveStream へ書き込んだのち、byte[] _EncodedWaveData へ出力する。"
					//-----------------
					while( true )
					{
						// 次のサンプルを読み込む。
						int actualStreamIndexRef = 0;
						var dwStreamFlagsRef = CSCore.MediaFoundation.MFSourceReaderFlags.None;
						Int64 llTimestampRef = 0;

						using( var sample = sourceReader.ReadSample(
							(int) SharpDX.MediaFoundation.SourceReaderIndex.FirstAudioStream,
							(int) CSCore.MediaFoundation.SourceReaderControlFlags.None,
							out actualStreamIndexRef,
							out dwStreamFlagsRef,
							out llTimestampRef ) )
						{
							if( null == sample )
								break;      // EndOfStream やエラーのときも null になる。

							// sample をロックし、オーディオデータへのポインタを取得する。
							int cbMaxLengthRef = 0;
							int cbCurrentLengthRef = 0;
							using( var mediaBuffer = sample.ConvertToContiguousBuffer() )
							{
								// オーディオデータをメモリストリームに書き込む。
								var audioData = mediaBuffer.Lock( out cbMaxLengthRef, out cbCurrentLengthRef );
								try
								{
									byte[] dstData = new byte[ cbCurrentLengthRef ];
									System.Runtime.InteropServices.Marshal.Copy( audioData, dstData, 0, cbCurrentLengthRef );
									waveStream.Write( dstData, 0, cbCurrentLengthRef );
								}
								finally
								{
									mediaBuffer.Unlock();
								}
							}
						}
					}

					// ストリームの内容を byte 配列に出力。
					this._EncodedWaveData = waveStream.ToArray();
					//-----------------
					#endregion
				}
			}
			catch( Exception e )
			{
				FDK.Log.ERROR( $"MediaFoundation SourceReader の初期化に失敗しました。[{e.Message}]" );
				this._EncodedWaveData = new byte[] { };
			}

			this._Position = 0;
		}

		private void _解放する()
		{
			FDK.Utilities.解放する( ref this._MediaType );
		}

		#region " Win32 "
		//----------------
		private const int MF_E_INVALIDMEDIATYPE = unchecked((int) 0xC00D36B4);
		//----------------
		#endregion
	}
}
