﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using FDK;
using FDK.メディア;
using FDK.メディア.サウンド.WASAPI;
using SSTFormat.v2;

namespace SST.ステージ.演奏
{
	/// <summary>
	///		AutoPlayer用のドラムサウンド。
	/// </summary>
	class ドラムサウンド : Activity
	{
		public ドラムサウンド()
		{
		}

		protected override void On活性化( グラフィックデバイス gd )
		{
			this._KitXmlを読み込む();
		}

		protected override void On非活性化( グラフィックデバイス gd )
		{
			this._すべてのコンテキストを初期化する();
		}

		public void 発声する( チップ種別 chipType, float 音量0to1 )
		{
			lock( this._スレッド間同期 )
			{
				if( false == this._チップtoコンテキスト.ContainsKey( chipType ) )
					return; // コンテキスト未登録のチップなので何もしない。

				var context = this._チップtoコンテキスト[ chipType ];

				// 現在発声中のサウンドを全部止めるチップ種別の場合は止める。
				if( 0 != chipType.排他発声グループID() ) // ID = 0 は対象外。
				{
					// 消音対象のコンテキストの Sounds[] を select する。
					var 停止するサウンド群 =
						from kvp in this._チップtoコンテキスト
						where ( chipType.直前のチップを消音する( kvp.Key ) )
						select kvp.Value.Sounds;

					// 集めた Sounds[] をすべて停止する。
					foreach( var sounds in 停止するサウンド群 )
					{
						foreach( var sound in sounds )
						{
							sound.Stop();
						}
					}
				}

				// 再生する。
				if( null != context.Sounds[ context.次に再生するSound番号 ] )
				{
					context.Sounds[ context.次に再生するSound番号 ].Volume = 音量0to1;
					context.Sounds[ context.次に再生するSound番号 ].Play();
				}

				// サウンドローテーション。
				context.次に再生するSound番号++;

				if( context.次に再生するSound番号 >= ドラムサウンド._多重度 )
					context.次に再生するSound番号 = 0;
			}
		}


		private const int _多重度 = 2;

		private class Cコンテキスト : IDisposable
		{
			public Sound[] Sounds = new Sound[ _多重度 ];
			public int 次に再生するSound番号 = 0;
			public void Dispose()
			{
				if( null != this.Sounds )
				{
					for( int i = 0; i < this.Sounds.Length; i++ )
					{
						this.Sounds[ i ].Stop();
						FDKUtilities.解放する( ref this.Sounds[ i ] );
					}
				}
			}
		};

		private Dictionary<チップ種別, Cコンテキスト> _チップtoコンテキスト = null;

		private readonly string _KitXmlファイルパス = @"$(System)sounds\Kit.xml";

		private readonly object _スレッド間同期 = new object();


		private void _すべてのコンテキストを初期化する()
		{
			lock( this._スレッド間同期 )
			{
				// コンテキストがすでに存在しているなら解放する。
				if( null != this._チップtoコンテキスト )
				{
					foreach( var kvp in this._チップtoコンテキスト )
						kvp.Value.Dispose();
				}

				// 生成する。
				this._チップtoコンテキスト = new Dictionary<チップ種別, Cコンテキスト>();
			}
		}

		private void _KitXmlを読み込む()
		{
			lock( this._スレッド間同期 )
			{
				this._すべてのコンテキストを初期化する();

				string xmlPath = Folder.絶対パスに含まれるフォルダ変数を展開して返す( this._KitXmlファイルパス );

				if( false == File.Exists( xmlPath ) )
				{
					Log.WARNING( $"Kit ファイルが存在しません。ドラムサウンドは生成されません。[{this._KitXmlファイルパス}]" );
					return;
				}

				try
				{
					var xml文書 = XDocument.Load( xmlPath );

					// <Root>
					var Root要素 = xml文書.Element( "Root" );
					{
						// <DrumKit>
						var DrumKit要素 = Root要素.Element( "DrumKit" );    // 最初の１つのみサポート。
						var DrumKit要素のVersion属性 = DrumKit要素.Attribute( "Version" );
						var DrumKit要素のName属性 = DrumKit要素.Attribute( "Name" );   // 現在は未使用。
						if( "1.2" == DrumKit要素のVersion属性.Value )
						{
							#region " DrumKit ver1.2 "
							//----------------
							foreach( var DrumKitの子要素 in DrumKit要素.Elements() )
							{
								switch( DrumKitの子要素.Name.LocalName )
								{
									// <Sound>
									case "Sound":
										// Name="..." 属性の値は、SSTFormat.チップ種別 の各メンバの名前に等しいものとする。
										var Sound要素のName属性 = DrumKitの子要素.Attribute( "Name" );
										if( Enum.TryParse( Sound要素のName属性.Value, out チップ種別 chipType ) )
										{
											string サウンドファイルパス = Path.Combine( SST.IO.Folder.System, @"sounds\", DrumKitの子要素.Value );

											if( File.Exists( サウンドファイルパス ) )
											{
												// すでに辞書に存在してるなら、解放して削除する。
												if( this._チップtoコンテキスト.ContainsKey( chipType ) )
												{
													this._チップtoコンテキスト[ chipType ]?.Dispose();
													this._チップtoコンテキスト.Remove( chipType );
												}

												// コンテキストを作成する。
												var context = new Cコンテキスト() {
													Sounds = new Sound[ ドラムサウンド._多重度 ],
													次に再生するSound番号 = 0,
												};

												// 多重度分のサウンドを生成する。
												for( int i = 0; i < context.Sounds.Length; i++ )
													context.Sounds[ i ] = App.サウンドデバイス.サウンドを生成する( サウンドファイルパス );

												// コンテキストを辞書に追加する。
												this._チップtoコンテキスト.Add( chipType, context );
											}
											else
											{
												Log.WARNING( $"サウンドファイル {Folder.絶対パスをフォルダ変数付き絶対パスに変換して返す( サウンドファイルパス )} が存在しません。[{this._KitXmlファイルパス}]" );
											}
										}
										else
										{
											Log.WARNING( $"未知の要素 {Sound要素のName属性.Value} をスキップします。[{this._KitXmlファイルパス}]" );
										}
										break;
								}
							}
							//----------------
							#endregion
						}
					}
				}
				catch( Exception e )
				{
					Log.ERROR( $"Kitファイルの読み込みに失敗しました。{e.Message}[{this._KitXmlファイルパス}]" );
				}
			}
		}
	}
}
