﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

namespace SSTFEditor
{
	class 選択モード : IDisposable
	{
		public 選択モード( メインフォーム form )
		{
			this.Form = form;
		}
		public void Dispose()
		{
			FDK.Utilities.解放する( ref this.選択領域用のブラシ );
			FDK.Utilities.解放する( ref this.選択領域用のペン );
		}
		public void 個別選択を解除する( SSTFormat.チップ chip )
		{
			var cell = new UndoRedo.セル<SSTFormat.チップ>(
				所有者ID: null,
				undoアクション: ( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {
					変更対象.選択が確定している = true;
					this.Form.未保存である = true;
				},
				redoアクション: ( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {
					変更対象.選択が確定している = false;
					this.Form.未保存である = true;
				},
				変更対象: chip,
				変更前の値: null, 
				変更後の値: null );

			this.Form.UndoRedo管理.セルを追加する( cell );
			cell.Redoを実行する();
			this.Form.UndoRedo用GUIのEnabledを設定する();
		}
		public void 全チップを選択する()
		{
			this.Form.UndoRedo管理.トランザクション記録を開始する();

			foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
			{
				if( chip.選択が確定していない )	// 選択されていないすべてのチップを選択する。
				{
					var 変更前のチップ = new SSTFormat.チップ( chip );
					var 変更後のチップ = new SSTFormat.チップ( chip ) {
						ドラッグ操作により選択中である = false,
						選択が確定している = true,
					};

					var cell = new UndoRedo.セル<SSTFormat.チップ>(
						null,
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {	// Undo
							変更対象.CopyFrom( 変更前 );
							this.Form.未保存である = true;
						},
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {	// Redo
							変更対象.CopyFrom( 変更後 );
							this.Form.未保存である = true;
						},
						chip, 変更前のチップ, 変更後のチップ, null, null );

					this.Form.UndoRedo管理.セルを追加する( cell );
					cell.Redoを実行する();
				}
			}

			this.Form.UndoRedo管理.トランザクション記録を終了する();
			this.Form.UndoRedo用GUIのEnabledを設定する();
			this.Form.選択チップの有無に応じて編集用GUIのEnabledを設定する();
			this.Form.譜面をリフレッシュする();
		}
		public void 全チップの選択を解除する()
		{
			this.Form.UndoRedo管理.トランザクション記録を開始する();

			foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
			{
				if( chip.枠外レーン数 != 0 || chip.譜面内絶対位置grid < 0 )
				{
					#region [ 譜面範囲外に出たチップがあれば削除する。]
					//-----------------
					var chip変更前 = this.移動開始時のチップ状態[ chip ];
					
					var cell = new UndoRedo.セル<SSTFormat.チップ>(
						null,
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {	// Undo
							変更対象.CopyFrom( chip変更前 );
							this.Form.譜面.SSTFormatScore.listチップ.Add( 変更対象 );
							this.Form.譜面.SSTFormatScore.listチップ.Sort();
							this.Form.未保存である = true;
						},
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {	// Redo
							this.Form.譜面.SSTFormatScore.listチップ.Remove( 変更対象 );
							this.Form.未保存である = true;
						},
						chip, chip変更前, null, null, null );

					this.Form.UndoRedo管理.セルを追加する( cell );
					cell.Redoを実行する();
					//-----------------
					#endregion
				}
				else if( chip.ドラッグ操作により選択中である || chip.選択が確定している )
				{
					#region [ チップの選択を解除する。]
					//-----------------
					var chip変更前 = new SSTFormat.チップ( chip );

					var cell = new UndoRedo.セル<SSTFormat.チップ>(
						null,
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {	// Undo
							変更対象.CopyFrom( 変更前 );
						},
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => {	// Redo
							変更対象.選択が確定している = false;
							変更対象.ドラッグ操作により選択中である = false;
							変更対象.移動済みである = false;
						},
						chip, chip変更前, null, null, null );

					this.Form.UndoRedo管理.セルを追加する( cell );
					cell.Redoを実行する();
					//-----------------
					#endregion
				}
			}

			this.Form.UndoRedo管理.トランザクション記録を終了する();
			this.Form.UndoRedo用GUIのEnabledを設定する();
			this.Form.未保存である = true;
		}
		public void 検索する()
		{
			var dialog = new 検索条件入力ダイアログ();

			if( dialog.ShowDialog( this.Form ) != DialogResult.OK )
				return;

			int n開始小節番号 = dialog.小節範囲指定CheckBoxがチェックされている ? dialog.小節範囲開始番号 : 0;
			int n終了小節番号 = dialog.小節範囲指定CheckBoxがチェックされている ? dialog.小節範囲終了番号 : this.Form.譜面.SSTFormatScore.最大小節番号;

			#region [ 開始・終了が省略されているときは、それぞれ 0・最大小節番号 とみなす。]
			//-----------------
			if( n開始小節番号 < 0 )
				n開始小節番号 = 0;

			if( n終了小節番号 < 0 )
				n終了小節番号 = this.Form.譜面.SSTFormatScore.最大小節番号;
			//-----------------
			#endregion


			this.Form.UndoRedo管理.トランザクション記録を開始する();

			int n選択チップ数 = 0;

			foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
			{
				var e編集レーン = this.Form.譜面.dicチップ編集レーン対応表[ chip.チップ種別 ];
				
				if( e編集レーン == 編集レーン種別.Unknown )
					continue;	// 編集レーンを持たないチップは無視する。

				if( chip.小節番号 >= n開始小節番号 && chip.小節番号 <= n終了小節番号 )
				{
					if( dialog.選択されている( e編集レーン ) ||
						dialog.選択されている( chip.チップ種別 ) )
					{
						// チップを選択する。

						var chip変更前 = new SSTFormat.チップ( chip );
						var cell = new UndoRedo.セル<SSTFormat.チップ>(
							null,
							( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => { 変更対象.選択が確定している = 変更前.選択が確定している; },	// Undo
							( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => { 変更対象.選択が確定している = true; },	// Redo
							chip, chip変更前, null, null, null );

						this.Form.UndoRedo管理.セルを追加する( cell );
						cell.Redoを実行する();

						n選択チップ数++;
					}
				}
			}

			this.Form.UndoRedo管理.トランザクション記録を終了する();
			this.Form.UndoRedo用GUIのEnabledを設定する();
			this.Form.譜面をリフレッシュする();

			#region [ チップ数に応じて結果を表示する。]
			//-----------------
			if( n選択チップ数 > 0 )
			{
				this.Form.選択チップの有無に応じて編集用GUIのEnabledを設定する();

				MessageBox.Show(
					n選択チップ数 + Properties.Resources.MSG_個のチップが選択されました,
					Properties.Resources.MSG_検索結果ダイアログのタイトル,
					MessageBoxButtons.OK, MessageBoxIcon.Asterisk, MessageBoxDefaultButton.Button1 );
			}
			else
			{
				MessageBox.Show(
					Properties.Resources.MSG_該当するチップはありませんでした,
					Properties.Resources.MSG_検索結果ダイアログのタイトル,
					MessageBoxButtons.OK, MessageBoxIcon.Asterisk, MessageBoxDefaultButton.Button1 );
			}
			//-----------------
			#endregion
		}

		// イベント

		public void MouseClick( MouseEventArgs e )
		{
			if( e.Button == MouseButtons.Right )
				this.Form.選択モードのコンテクストメニューを表示する( e.X, e.Y );
		}
		public void MouseDown( MouseEventArgs e )
		{
			if( e.Button == MouseButtons.Left )
			{
				SSTFormat.チップ chip = this.Form.譜面.t譜面パネル内座標pxに存在するチップがあれば返す( e.X, e.Y );


				// (A) チップがないか、未選択のチップがあった場合。

				if( chip == null || !chip.選択が確定している )
					this.範囲選択の開始処理( e );


				// (B) 選択状態のチップがあり、かつ、クリック時に CTRL が押されている場合。

				else if( ( Control.ModifierKeys & Keys.Control ) == Keys.Control )
					this.個別選択を解除する( chip );


				// (C) 選択状態のチップがあり、かつ、クリック時に CTRL が押されていない場合。

				else
					this.移動の開始処理( e );
			}
		}
		public void MouseMove( MouseEventArgs e )
		{
			// (A) 左ボタンが押されながら移動している場合 → 継続処理

			if( e.Button == MouseButtons.Left )
			{
				if( this.範囲選択のためにドラッグ中である )
					this.範囲選択の継続処理( e );

				else if( this.移動のためにドラッグ中である )
					this.移動の継続処理( e );
			}


			// (B) 左ボタンが押されずに移動した場合 → 終了処理

			else
			{
				if( this.範囲選択のためにドラッグ中である )
					this.範囲選択の終了処理( e );

				else if( this.移動のためにドラッグ中である )
					this.移動の終了処理( e );
			}


			// 譜面を再描画。

			this.Form.譜面をリフレッシュする();
		}
		public void Paint( PaintEventArgs e )
		{
			if( this.範囲選択のためにドラッグ中である )
				this.現在の選択範囲を描画する( e.Graphics );
		}

		// 全般

		protected メインフォーム Form;
		protected SolidBrush 選択領域用のブラシ = new SolidBrush( Color.FromArgb( 80, 55, 55, 255 ) );
		protected Pen 選択領域用のペン = new Pen( Color.LightBlue );
		protected void 現在の選択範囲を描画する( Graphics g )
		{
			var rc現在の選択領域px = new Rectangle() {
				X = Math.Min( this.現在の範囲選択用ドラッグ開始位置px.X, this.現在の範囲選択用ドラッグ終了位置px.X ),
				Y = Math.Min( this.現在の範囲選択用ドラッグ開始位置px.Y, this.現在の範囲選択用ドラッグ終了位置px.Y ),
				Width = Math.Abs( (int) ( this.現在の範囲選択用ドラッグ開始位置px.X - this.現在の範囲選択用ドラッグ終了位置px.X ) ),
				Height = Math.Abs( (int) ( this.現在の範囲選択用ドラッグ開始位置px.Y - this.現在の範囲選択用ドラッグ終了位置px.Y ) ),
			};

			#region [ クリッピング ]
			//-----------------
			if( rc現在の選択領域px.Width < 0 )
			{
				rc現在の選択領域px.X = this.現在の移動用ドラッグ開始位置px.X;
				rc現在の選択領域px.Width = this.現在の移動用ドラッグ開始位置px.X - rc現在の選択領域px.X;
			}
			if( rc現在の選択領域px.Height < 0 )
			{
				rc現在の選択領域px.Y = this.現在の移動用ドラッグ開始位置px.Y;
				rc現在の選択領域px.Height = this.現在の移動用ドラッグ開始位置px.Y - rc現在の選択領域px.Y;
			}
			//-----------------
			#endregion

			if( ( rc現在の選択領域px.Width != 0 ) && ( rc現在の選択領域px.Height != 0 ) )
			{
				g.FillRectangle( this.選択領域用のブラシ, rc現在の選択領域px );
				g.DrawRectangle( Pens.LightBlue, rc現在の選択領域px );
			}
		}
		protected void 譜面パネルの上下端にきたならスクロールする( MouseEventArgs e )
		{
			const int n上端スクロール発動幅px = 70;
			const int n下端スクロール発動幅px = 50;

			if( e.Y <= n上端スクロール発動幅px )
			{
				double db速度係数X = ( Math.Max( ( n上端スクロール発動幅px - e.Y ), 0 ) ) / (double) n上端スクロール発動幅px;
				double db速度係数Y = ( 1.0 - Math.Cos( ( Math.PI / 2.0 ) * db速度係数X ) ) * 2.0 + 1.0;
				int nスクロール量grid =	(int) ( - db速度係数Y * 180.0 );

				this.Form.譜面を縦スクロールする( nスクロール量grid );

				if( this.移動のためにドラッグ中である )
					this.現在の移動用ドラッグ開始位置px.Y -= nスクロール量grid / this.Form.GRID_PER_PIXEL;

				if( this.範囲選択のためにドラッグ中である )
					this.現在の範囲選択用ドラッグ開始位置px.Y -= nスクロール量grid / this.Form.GRID_PER_PIXEL;
			}
			else if( e.Y >= ( this.Form.譜面パネルサイズ.Height - n下端スクロール発動幅px ) )
			{
				double db速度係数X = ( Math.Max( ( e.Y - ( this.Form.譜面パネルサイズ.Height - n上端スクロール発動幅px ) ), 0 ) ) / (double) n下端スクロール発動幅px;
				double db速度係数Y = ( 1.0 - Math.Cos( ( Math.PI / 2.0 ) * db速度係数X ) ) * 2.0 + 1.0;
				int nスクロール量grid =	(int) ( db速度係数Y * 180.0 );

				this.Form.譜面を縦スクロールする( nスクロール量grid );
				
				if( this.移動のためにドラッグ中である )
					this.現在の移動用ドラッグ開始位置px.Y -= nスクロール量grid / this.Form.GRID_PER_PIXEL;

				if( this.範囲選択のためにドラッグ中である )
					this.現在の範囲選択用ドラッグ開始位置px.Y -= nスクロール量grid / this.Form.GRID_PER_PIXEL;
			}
		}
		
		// 移動関連

		protected bool 移動のためにドラッグ中である = false;
		protected Point 現在の移動用ドラッグ開始位置px = new Point( 0, 0 );
		protected Point 現在の移動用ドラッグ終了位置px = new Point( 0, 0 );
		protected Dictionary<SSTFormat.チップ, SSTFormat.チップ> 移動開始時のチップ状態 = new Dictionary<SSTFormat.チップ, SSTFormat.チップ>();
		protected struct STレーングリッド座標
		{
			public int 編集レーン番号;         // X座標に相当。
			public int 譜面内絶対位置grid;     // Y座標に相当。
		};
		protected STレーングリッド座標 前回のマウス位置LaneGrid = new STレーングリッド座標();
		protected void 移動の開始処理( MouseEventArgs e )
		{
			this.移動のためにドラッグ中である = true;


			// ドラッグ範囲の初期化。

			this.現在の移動用ドラッグ開始位置px.X = this.現在の移動用ドラッグ終了位置px.X = e.X;
			this.現在の移動用ドラッグ開始位置px.Y = this.現在の移動用ドラッグ終了位置px.Y = e.Y;


			// マウス位置（lane×grid）の初期化。

			this.前回のマウス位置LaneGrid = new STレーングリッド座標() {
				編集レーン番号 = this.Form.譜面.dicレーン番号[ this.Form.譜面.譜面パネル内X座標pxにある編集レーンを返す( e.X ) ],
				譜面内絶対位置grid = this.Form.譜面.譜面パネル内Y座標pxにおける譜面内絶対位置gridをガイド幅単位で返す( e.Y ),
			};


			// 移動対象チップ（現在選択が確定しているチップ）の「移動開始時のチップの全状態」を、ローカルの Dictionary に控えておく。
			// これは t移動終了処理() で使用する。

			this.移動開始時のチップ状態.Clear();
			foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
			{
				if( chip.選択が確定している )
					this.移動開始時のチップ状態.Add( chip, new SSTFormat.チップ( chip ) );
			}
		}
		protected void 移動の継続処理( MouseEventArgs e )
		{
			// ドラッグ終了位置を現在のマウスの位置に更新。

			this.現在の移動用ドラッグ終了位置px.X = e.X;
			this.現在の移動用ドラッグ終了位置px.Y = e.Y;


			// スクロールチェック。

			this.譜面パネルの上下端にきたならスクロールする( e );


			// チップの移動。

			#region [ 現在確定中のチップを移動する。]
			//-----------------

			// 現在の位置を算出。

			var st現在のドラッグ終了位置LaneGrid = new STレーングリッド座標() {
				編集レーン番号 = this.Form.譜面.dicレーン番号[ this.Form.譜面.譜面パネル内X座標pxにある編集レーンを返す( this.現在の移動用ドラッグ終了位置px.X ) ],
				譜面内絶対位置grid = this.Form.譜面.譜面パネル内Y座標pxにおける譜面内絶対位置gridをガイド幅単位で返す( this.現在の移動用ドラッグ終了位置px.Y ),
			};


			// 前回位置からの移動量を算出。

			var st移動量LaneGrid = new STレーングリッド座標() {
				編集レーン番号 = st現在のドラッグ終了位置LaneGrid.編集レーン番号 - this.前回のマウス位置LaneGrid.編集レーン番号,
				譜面内絶対位置grid = st現在のドラッグ終了位置LaneGrid.譜面内絶対位置grid - this.前回のマウス位置LaneGrid.譜面内絶対位置grid,
			};


			// 前回位置から移動していれば、選択されているすべてのチップを移動させる。

			if( st移動量LaneGrid.編集レーン番号 != 0 || st移動量LaneGrid.譜面内絶対位置grid != 0 )
			{
				#region [ 全チップの移動済フラグをリセットする。]
				//-----------------
				foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
					chip.移動済みである = false;
				//-----------------
				#endregion

				foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
				{
					if( chip.選択が確定している && !chip.移動済みである )
					{
						if( st移動量LaneGrid.編集レーン番号 != 0 )
						{
							#region [ チップを横に移動する。]
							//-----------------
							int nレーン数 = Enum.GetValues( typeof( 編集レーン種別 ) ).Length;
							int nチップの現在のレーン番号 = this.Form.譜面.dicレーン番号[ this.Form.譜面.dicチップ編集レーン対応表[ chip.チップ種別 ] ];
							int nチップの移動後のレーン番号;

							#region [ nチップの移動後のレーン番号 を算出。]
							//-----------------
							if( chip.枠外レーン数 < 0 )
							{
								nチップの移動後のレーン番号 = nチップの現在のレーン番号 + st移動量LaneGrid.編集レーン番号;
							}
							else if( chip.枠外レーン数 > 0 )
							{
								nチップの移動後のレーン番号 = ( ( nレーン数 - 1 ) + chip.枠外レーン数 ) + st移動量LaneGrid.編集レーン番号;
							}
							else
							{
								nチップの移動後のレーン番号 = nチップの現在のレーン番号;
							}
							//-----------------
							#endregion
							#region [ nチップの移動後のレーン番号 から、チップのチップ種別 or 枠外レーン数 を修正する。]
							//-----------------
							if( nチップの移動後のレーン番号 < 0 )				// 左にはみ出している
							{
								chip.枠外レーン数 = nチップの移動後のレーン番号;
							}
							else if( nチップの移動後のレーン番号 >= nレーン数 )	// 右にはみ出している
							{
								chip.枠外レーン数 = nチップの移動後のレーン番号 - ( nレーン数 - 1 );
							}
							else
							{
								var eチップの移動後の編集レーン = this.Form.譜面.dicレーン番号逆引き[ nチップの移動後のレーン番号 ];

								foreach( var kvp in this.Form.譜面.dicチップ編集レーン対応表 )
								{
									if( kvp.Value == eチップの移動後の編集レーン )	// 対応表で最初に見つけた kvp のチップ種別を採択する。
									{
										chip.チップ種別 = kvp.Key;
										break;
									}
								}
								chip.枠外レーン数 = 0;
							}
							//-----------------
							#endregion

							chip.移動済みである = true;
							this.Form.未保存である = true;
							//-----------------
							#endregion
						}

						if( st移動量LaneGrid.譜面内絶対位置grid != 0 )
						{
							#region [ チップを縦に移動する。]
							//-----------------
							int n移動後の譜面内絶対位置grid = chip.譜面内絶対位置grid + st移動量LaneGrid.譜面内絶対位置grid;

							if( n移動後の譜面内絶対位置grid < 0 )
							{
								// 画面外なので何もしない。
							}
							else if( n移動後の譜面内絶対位置grid >= this.Form.譜面.全小節の高さgrid )
							{
								chip.譜面内絶対位置grid = n移動後の譜面内絶対位置grid;		// そこに小節はないが、一応描画する。
							}
							else
							{
								chip.譜面内絶対位置grid = n移動後の譜面内絶対位置grid;
							}

							chip.移動済みである = true;
							this.Form.未保存である = true;
							//-----------------
							#endregion
						}
					}
				}


				// 前回位置を現在の位置に更新。

				this.前回のマウス位置LaneGrid = new STレーングリッド座標() {
					編集レーン番号 = st現在のドラッグ終了位置LaneGrid.編集レーン番号,
					譜面内絶対位置grid = st現在のドラッグ終了位置LaneGrid.譜面内絶対位置grid,
				};
			}
			//-----------------
			#endregion
		}
		protected void 移動の終了処理( MouseEventArgs e )
		{
			this.Form.UndoRedo管理.トランザクション記録を開始する();

			foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
			{
				// 選択が確定かつ初期位置から移動しているチップのみ対象とする。

				if( chip.選択が確定している &&
					( chip.譜面内絶対位置grid != this.移動開始時のチップ状態[ chip ].譜面内絶対位置grid || chip.チップ種別 != this.移動開始時のチップ状態[ chip ].チップ種別 ) )
				{
					// ここではまだ、譜面範囲外に出ている（＝枠外レーン数がゼロでない）チップの削除は行わない。（その後再び移動される可能性があるため。）
					// これらの削除処理は「t全チップの選択を解除する()」で行う。

					var chip変更前 = this.移動開始時のチップ状態[ chip ];

					int n小節先頭位置grid = 0;
					int n小節番号 = this.Form.譜面.譜面内絶対位置gridにおける小節の番号と小節先頭の位置gridを返す( chip.譜面内絶対位置grid, out n小節先頭位置grid );

					var chip変更後 = new SSTFormat.チップ( chip ) {
						小節番号 = n小節番号,
						小節解像度 = (int) ( this.Form.GRID_PER_PART * this.Form.譜面.SSTFormatScore.小節長倍率を取得する( n小節番号 ) ),
						小節内位置 = chip.譜面内絶対位置grid - n小節先頭位置grid,
					};

					var cell = new UndoRedo.セル<SSTFormat.チップ>(
						null,
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => { 変更対象.CopyFrom( 変更前 ); },	// Undo
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => { 変更対象.CopyFrom( 変更後 ); },	// Redo
						chip, chip変更前, chip変更後, null, null );

					this.Form.UndoRedo管理.セルを追加する( cell );
					cell.Redoを実行する();
				}
			}

			this.Form.UndoRedo管理.トランザクション記録を終了する();
			this.Form.UndoRedo用GUIのEnabledを設定する();
			this.移動のためにドラッグ中である = false;
		}

		// 範囲選択関連

		protected bool 範囲選択のためにドラッグ中である = false;
		protected Point 現在の範囲選択用ドラッグ開始位置px = new Point( 0, 0 );
		protected Point 現在の範囲選択用ドラッグ終了位置px = new Point( 0, 0 );
		protected void 範囲選択の開始処理( MouseEventArgs e )
		{
			this.範囲選択のためにドラッグ中である = true;


			// ドラッグ範囲の初期化。

			this.現在の範囲選択用ドラッグ開始位置px.X = this.現在の範囲選択用ドラッグ終了位置px.X = e.X;
			this.現在の範囲選択用ドラッグ開始位置px.Y = this.現在の範囲選択用ドラッグ終了位置px.Y = e.Y;


			// CTRL が押されていない場合、いったん全チップの選択を解除する。

			if( ( Control.ModifierKeys & Keys.Control ) != Keys.Control )
				this.全チップの選択を解除する();


			// 全チップについて、選択・選択解除の取捨選択。

			this.現在のドラッグ範囲中のチップをすべて選択状態にしそれ以外は選択を解除する();
		}
		protected void 範囲選択の継続処理( MouseEventArgs e )
		{
			// クリッピング。

			int x = e.X;
			int y = e.Y;

			if( x < 0 ) x = 0;
			if( x >= this.Form.譜面パネルサイズ.Width ) x = this.Form.譜面パネルサイズ.Width;
			if( y < 0 ) y = 0;
			if( y >= this.Form.譜面パネルサイズ.Height ) y = this.Form.譜面パネルサイズ.Height;


			// ドラッグ終了位置を現在のマウスの位置に更新。

			this.現在の範囲選択用ドラッグ終了位置px.X = x;
			this.現在の範囲選択用ドラッグ終了位置px.Y = y;


			// スクロールチェック。

			this.譜面パネルの上下端にきたならスクロールする( e );


			// チップの選択 or 選択解除。

			this.現在のドラッグ範囲中のチップをすべて選択状態にしそれ以外は選択を解除する();
		}
		protected void 範囲選択の終了処理( MouseEventArgs e )
		{
			this.範囲選択のためにドラッグ中である = false;
			this.Form.UndoRedo管理.トランザクション記録を開始する();


			// ドラック選択範囲に入っているがまだ選択が確定していないチップを選択確定させる。

			foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
			{
				if( chip.ドラッグ操作により選択中である && !chip.選択が確定している )		// ドラッグ選択範囲内にあって既に選択が確定されているものについては何もしない。
				{
					var chip変更前 = new SSTFormat.チップ( chip );
					var chip変更後 = new SSTFormat.チップ( chip ) {
						ドラッグ操作により選択中である = false,
						選択が確定している = true,
					};
					var cell = new UndoRedo.セル<SSTFormat.チップ>(
						null,
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => { 変更対象.CopyFrom( 変更前 ); },	// Undo
						( 変更対象, 変更前, 変更後, 任意1, 任意2 ) => { 変更対象.CopyFrom( 変更後 ); },	// Redo
						chip, chip変更前, chip変更後, null, null );

					this.Form.UndoRedo管理.セルを追加する( cell );
					cell.Redoを実行する();
				}
			}

			this.Form.UndoRedo管理.トランザクション記録を終了する();
			this.Form.UndoRedo用GUIのEnabledを設定する();
			this.Form.選択チップの有無に応じて編集用GUIのEnabledを設定する();
		}
		protected void 現在のドラッグ範囲中のチップをすべて選択状態にしそれ以外は選択を解除する()
		{
			// 現在のドラッグ範囲を lane×grid 座標で算出する。
			// Y座標について： px は上→下、grid は下→上に向かって増加するので注意！

			var rc現在のドラッグ範囲px = new Rectangle() {
				X = Math.Min( this.現在の範囲選択用ドラッグ開始位置px.X, this.現在の範囲選択用ドラッグ終了位置px.X ),
				Y = Math.Min( this.現在の範囲選択用ドラッグ開始位置px.Y, this.現在の範囲選択用ドラッグ終了位置px.Y ),
				Width = Math.Abs( (int) ( this.現在の範囲選択用ドラッグ開始位置px.X - this.現在の範囲選択用ドラッグ終了位置px.X ) ),
				Height = Math.Abs( (int) ( this.現在の範囲選択用ドラッグ開始位置px.Y - this.現在の範囲選択用ドラッグ終了位置px.Y ) ),
			};
			var rc現在のドラッグ範囲LaneGrid = new Rectangle() {
				X = this.Form.譜面.dicレーン番号[ this.Form.譜面.譜面パネル内X座標pxにある編集レーンを返す( rc現在のドラッグ範囲px.Left ) ],
				Y = this.Form.譜面.t譜面パネル内Y座標pxにおける譜面内絶対位置gridを返す( rc現在のドラッグ範囲px.Bottom ),
			};
			rc現在のドラッグ範囲LaneGrid.Width = Math.Abs( this.Form.譜面.dicレーン番号[ this.Form.譜面.譜面パネル内X座標pxにある編集レーンを返す( rc現在のドラッグ範囲px.Right ) ] - rc現在のドラッグ範囲LaneGrid.X );
			rc現在のドラッグ範囲LaneGrid.Height = Math.Abs( rc現在のドラッグ範囲px.Height * this.Form.GRID_PER_PIXEL );


			// すべてのチップについて、現在のドラッグ範囲 内に存在しているチップは「ドラッグ操作により選択中」フラグを立てる。

			foreach( var chip in this.Form.譜面.SSTFormatScore.listチップ )
			{
				int nチップのレーン番号 = this.Form.譜面.dicレーン番号[ this.Form.譜面.dicチップ編集レーン対応表[ chip.チップ種別 ] ];
				int nチップの厚さgrid = this.Form.譜面.szチップpx.Height * this.Form.GRID_PER_PIXEL;

				var rcチップLaneGrid = new Rectangle() {
					X = nチップのレーン番号,
					Y = chip.譜面内絶対位置grid,
					Width = 0,
					Height = nチップの厚さgrid,
				};

				if( rcチップLaneGrid.Right < rc現在のドラッグ範囲LaneGrid.Left || rc現在のドラッグ範囲LaneGrid.Right < rcチップLaneGrid.Left )
				{
					chip.ドラッグ操作により選択中である = false;		// チップは範囲外のレーンにいる
				}
				else if( rcチップLaneGrid.Bottom < rc現在のドラッグ範囲LaneGrid.Top || rc現在のドラッグ範囲LaneGrid.Bottom < rcチップLaneGrid.Top )
				{
					chip.ドラッグ操作により選択中である = false;		// チップは範囲外のグリッドにいる
				}
				else
					chip.ドラッグ操作により選択中である = true;		// チップは範囲内である
			}
		}
	}
}
