﻿using System;
using System.Collections.Generic;

namespace FDK.カウンタ
{
	/// <summary>
	/// パフォーマンスカウンタを使用した高精度タイマ。
	/// </summary>
	/// <remarks>
	/// 以下の2種類の使い方を想定する。
	/// (A) 正確に同一の時刻を複数の処理で共有できるように、現在時刻をキャプチャしてから取得する方法。
	///    1. 最初に「現在のカウントをキャプチャする()」を呼び出し、その時点での時刻を内部に保存する。
	///    2. キャプチャされたカウントを、「現在のキャプチャカウントを……取得する()」を呼び出して、希望する単位で取得する。
	///      （次に1.を行うまで、2.はずっと同じ時刻を返し続ける。）
	/// (B) 常に現時刻（メソッド呼び出し時点の時刻）を取得する方法。
	///    a. 「現在のリアルタイムカウントを……取得する()」を呼び出して、希望する単位で取得する。
	///    または、
	///    b. 「生カウントを取得する()」を呼び出して、生カウンタを取得する。
	///
	/// 時刻の単位としては、[カウント], [秒], [100ナノ秒] を用意する。
	/// 
	/// 用語：
	/// "カウント" …………………… タイマインスタンスの生成時（または前回のリセット時）から「前回キャプチャされた時点」までの、パフォーマンスカウンタの差分（相対値）。
	/// "リアルタイムカウント" …… タイマインスタンスの生成時（または前回のリセット時）から「現時点」までの、パフォーマンスカウンタの差分（相対値）。
	/// "生カウント" ………………… パフォーマンスカウンタの生の値。::QueryPerformanceCounter() で取得できる値に等しい。システム依存の絶対値。
	/// </remarks>
	public class QPCTimer
	{
		/// <summary>
		/// カウントが無効であることを示す定数。
		/// </summary>
		public const long 未使用 = -1;

		public QPCTimer()
		{
			var dummy = QPCTimer.周波数;  // プロパティの get() 内で qpc周波数 を更新させるためのダミーアクセス。
			var now = QPCTimer.生カウント;
			this.bs_前回リセットした時点の生カウント = now;
			this.最後にキャプチャされたカウント = now;
			this.稼働中に一時停止した時点のキャプチャカウント = now;
			this.一時停止回数 = 0;
		}

		public static long 周波数
		{
			get
			{
				if( 0 > QPCTimer.qpc周波数Hz )
				{
					// 初めてのアクセスならシステムから値を取得する。
					QPCTimer.qpc周波数Hz = System.Diagnostics.Stopwatch.Frequency;
				}
				return QPCTimer.qpc周波数Hz;
			}
		}
		public static long 生カウント
			=> System.Diagnostics.Stopwatch.GetTimestamp();
		public static double 生カウント相対値を秒へ変換して返す( long 生カウント相対値 )
			=> (double) 生カウント相対値 / (double) QPCTimer.qpc周波数Hz;

		public long 現在のカウントをキャプチャする()
		{
			lock( this.排他利用 )
			{
				this.最後にキャプチャされたカウント = QPCTimer.生カウント;
				return this.最後にキャプチャされたカウント;
			}
		}

		public long 現在のキャプチャカウント
		{
			get
			{
				lock( this.排他利用 )
				{
					long count = ( 0 != this.一時停止回数 ) ?
						( this.稼働中に一時停止した時点のキャプチャカウント - this.bs_前回リセットした時点の生カウント ) :    // 停止中
						( this.最後にキャプチャされたカウント - this.bs_前回リセットした時点の生カウント );                    // 稼働中
					return count;
				}
			}
		}
		public double 現在のキャプチャカウント秒単位
			=> ( (double) this.現在のキャプチャカウント / (double) QPCTimer.周波数 );
		public long 現在のキャプチャカウント100ns単位
			=> ( 1000 * 1000 * 10 * this.現在のキャプチャカウント / QPCTimer.周波数 );

		public long 現在のリアルタイムカウント
		{
			get
			{
				lock( this.排他利用 )
				{
					long count = ( 0 != this.一時停止回数 ) ?
						( this.稼働中に一時停止した時点のキャプチャカウント - this.bs_前回リセットした時点の生カウント ) :    // 停止中
						( QPCTimer.生カウント - this.bs_前回リセットした時点の生カウント );                             // 稼働中
					return count;
				}
			}
		}
		public double 現在のリアルタイムカウント秒単位
			=> ( (double) this.現在のリアルタイムカウント / (double) QPCTimer.周波数 );
		public long 現在のリアルタイムカウント100ns単位
			=> ( 1000 * 1000 * 10 * this.現在のリアルタイムカウント / QPCTimer.周波数 );

		public long 前回リセットした時点の生カウント
		{
			get
			{
				lock( this.排他利用 )
				{
					return this.bs_前回リセットした時点の生カウント;
				}
			}
		}

		public bool 停止中である
		{
			get
			{
				lock( this.排他利用 )
				{
					return ( 0 != this.一時停止回数 ) ? true : false;
				}
			}
		}
		public bool 稼働中である
		{
			get
			{
				lock( this.排他利用 )
				{
					return ( 0 == this.一時停止回数 ) ? true : false;
				}
			}
		}

		public void リセットする()
		{
			lock( this.排他利用 )
			{
				this.bs_前回リセットした時点の生カウント = QPCTimer.生カウント;
			}
		}
		public void リセットする( long 新しいカウント )
		{
			lock( this.排他利用 )
			{
				this.bs_前回リセットした時点の生カウント = QPCTimer.生カウント - 新しいカウント;
			}
		}
		public void 一時停止する()
		{
			lock( this.排他利用 )
			{
				if( 0 == this.一時停止回数 ) // 稼働中である
				{
					this.稼働中に一時停止した時点のキャプチャカウント = this.最後にキャプチャされたカウント;
				}
				this.一時停止回数++;
			}
		}
		public void 再開する()
		{
			lock( this.排他利用 )
			{
				if( 0 != this.一時停止回数 ) // 停止中である
				{
					this.一時停止回数--;

					if( 0 == this.一時停止回数 )
					{
						this.最後にキャプチャされたカウント = QPCTimer.生カウント;
						this.bs_前回リセットした時点の生カウント = this.最後にキャプチャされたカウント - this.稼働中に一時停止した時点のキャプチャカウント;
					}
				}
			}
		}

		private static long qpc周波数Hz = -1;      // キャッシュ
		private long bs_前回リセットした時点の生カウント = 0;
		private long 最後にキャプチャされたカウント = 0;
		private long 稼働中に一時停止した時点のキャプチャカウント = 0;
		private int 一時停止回数 = 0;
		private readonly object 排他利用 = new object();
	}
}
