﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using SharpDX;
using SharpDX.Direct2D1;
using SharpDX.WIC;

namespace FDK.メディア
{
	/// <summary>
	///		Direct2D の Bitmap を使って描画する画像。
	/// </summary>
	public class 画像 : Activity
	{
		/// <summary>
		///		画像の生成に成功していれば true 。
		/// </summary>
		public bool 生成成功
		{
			get
				=> ( null != this.Bitmap );
		}

		/// <summary>
		///		画像の生成に失敗していれば true 。
		/// </summary>
		public bool 生成失敗
		{
			get
				=> !( this.生成成功 );
		}

		/// <summary>
		///		画像は、設計単位で作成される。
		/// </summary>
		public Size2F サイズdpx
		{
			get
			{
				if( this.生成成功 )
				{
					return new Size2F( this.Bitmap.PixelSize.Width, this.Bitmap.PixelSize.Height );
				}
				else
				{
					return Size2F.Zero;
				}
			}
		}

		public InterpolationMode 補正モード
		{
			get;
			set;
		} = InterpolationMode.Linear;

		public bool 加算合成
		{
			get;
			set;
		} = false;


		public 画像( string 画像ファイルパス )
		{
			this.画像ファイルパス = FDK.フォルダ.絶対パスに含まれるフォルダ変数を展開して返す( 画像ファイルパス );
		}

		protected override void Onデバイス依存リソースの作成( デバイスリソース dr )
		{
			this.画像を生成する( dr );
		}

		protected override void Onデバイス依存リソースの解放( デバイスリソース dr )
		{
			Utilities.解放する( ref this.Bitmap );
		}

		/// <summary>
		///		画像を描画する。
		/// </summary>
		/// <param name="dr">デバイスリソース。</param>
		/// <param name="左位置dpx">画像の描画先範囲の左上隅X座標。</param>
		/// <param name="上位置dpx">画像の描画先範囲の左上隅Y座標。</param>
		/// <param name="不透明度0to1">不透明度。(0:透明～1:不透明)</param>
		/// <param name="X方向拡大率">画像の横方向の拡大率。</param>
		/// <param name="Y方向拡大率">画像の縦方向の拡大率。</param>
		/// <param name="転送元矩形dpx">画像の転送元範囲。画像は設計単位で作成されている。</param>
		/// <param name="変換行列3Dpx">射影行列。Direct2D の仕様により、設計単位ではなく物理単位で構築すること。</param>
		public virtual void 描画する(
			デバイスリソース dr,
			float 左位置dpx,
			float 上位置dpx,
			float 不透明度0to1 = 1.0f,
			float X方向拡大率 = 1.0f,
			float Y方向拡大率 = 1.0f,
			RectangleF? 転送元矩形dpx = null,
			Matrix? 変換行列3Dpx = null )
		{
			Debug.Assert( this.活性化している );

			if( null == this.Bitmap )
				return;

			if( ( null == 転送元矩形dpx ) || ( false == 転送元矩形dpx.HasValue ) )
			{
				// Bitmap.PixelSize は設計単位(dpx)なので注意。
				転送元矩形dpx = new RectangleF( 0f, 0f, this.Bitmap.PixelSize.Width, this.Bitmap.PixelSize.Height );
			}

			//var 転送先矩形px = new RectangleF(
			//	x: 左位置dpx * dr.拡大率DPXtoPX横方向 + 0.5f,
			//	y: 上位置dpx * dr.拡大率DPXtoPX縦方向 + 0.5f,
			//	width: 転送元矩形dpx.Value.Width * X方向拡大率 * dr.拡大率DPXtoPX横方向 + 0.5f,
			//	height: 転送元矩形dpx.Value.Height * Y方向拡大率 * dr.拡大率DPXtoPX縦方向 + 0.5f );
			//		↓
			// ※転送先矩形px は float で指定できるが、非整数の値（＝物理ピクセル単位じゃない座標）を渡すと、表示画像がプラスマイナス1pxの範囲で乱れる。
			// 　これにより、数px程度の大きさの画像を移動させるとチカチカする原因になる。
			// 　そのため、転送先矩形はすべて整数で指定することにする。
			//		↓
			var 転送先矩形px = new RectangleF(
				x: (float) Math.Round( 左位置dpx * dr.拡大率DPXtoPX横方向 + 0.5f ),
				y: (float) Math.Round( 上位置dpx * dr.拡大率DPXtoPX縦方向 + 0.5f ),
				width: (float) Math.Round( 転送元矩形dpx.Value.Width * X方向拡大率 * dr.拡大率DPXtoPX横方向 + 0.5f ),
				height: (float) Math.Round( 転送元矩形dpx.Value.Height * Y方向拡大率 * dr.拡大率DPXtoPX縦方向 + 0.5f ) );

			Utilities.D2DBatchDraw( dr.D2DContext1, () => {

				// 変換行列とブレンドモードをD2Dレンダーターゲットに設定する。
				dr.D2DContext1.Transform = Matrix3x2.Identity;
				dr.D2DContext1.PrimitiveBlend = ( this.加算合成 ) ? PrimitiveBlend.Add : PrimitiveBlend.SourceOver;

				// D2Dレンダーターゲットに this.Bitmap を描画する。
				dr.D2DContext1.DrawBitmap(
					bitmap: this.Bitmap,
					destinationRectangle: 転送先矩形px,
					opacity: 不透明度0to1,
					interpolationMode: this.補正モード,
					sourceRectangle: 転送元矩形dpx,  // this.Bitmap は設計単位で作成されている。
					erspectiveTransformRef: 変換行列3Dpx ); // null 指定可。
			} );
		}

		/// <summary>
		///		画像を描画する。
		/// </summary>
		/// <param name="dr">デバイスリソース。</param>
		/// <param name="変換行列2Dpx">Transform に適用する行列。Direct2D の仕様により、設計単位ではなく物理単位で構築すること。</param>
		/// <param name="変換行列3Dpx">射影行列。Direct2D の仕様により、設計単位ではなく物理単位で構築すること。</param>
		/// <param name="不透明度0to1">不透明度。(0:透明～1:不透明)</param>
		/// <param name="転送元矩形dpx">描画する画像範囲。画像は設計単位で作成されている。</param>
		public virtual void 描画する(
			デバイスリソース dr,
			Matrix3x2? 変換行列2Dpx = null,
			Matrix? 変換行列3Dpx = null,
			float 不透明度0to1 = 1.0f,
			RectangleF? 転送元矩形dpx = null )
		{
			Debug.Assert( this.活性化している );

			if( null == this.Bitmap )
				return;

			Utilities.D2DBatchDraw( dr.D2DContext1, () => {

				// 変換行列とブレンドモードをD2Dレンダーターゲットに設定する。
				dr.D2DContext1.Transform = 変換行列2Dpx ?? Matrix3x2.Identity;
				dr.D2DContext1.PrimitiveBlend = ( this.加算合成 ) ? PrimitiveBlend.Add : PrimitiveBlend.SourceOver;

				// D2Dレンダーターゲットに this.Bitmap を描画する。
				dr.D2DContext1.DrawBitmap(
					bitmap: this.Bitmap,
					destinationRectangle: null,
					opacity: 不透明度0to1,
					interpolationMode: this.補正モード,
					sourceRectangle: 転送元矩形dpx,  // this.Bitmap は設計単位で作成されている。
					erspectiveTransformRef: 変換行列3Dpx ); // null 指定可。
			} );
		}


		protected string 画像ファイルパス = null;

		protected Bitmap1 Bitmap = null;


		protected void 画像を生成する( デバイスリソース dr, BitmapProperties1 bitmapProperties1 = null )
		{
			var decoder = (BitmapDecoder) null;
			var sourceFrame = (BitmapFrameDecode) null;
			var converter = (FormatConverter) null;

			string 変数付きファイルパス = フォルダ.絶対パスをフォルダ変数付き絶対パスに変換して返す( this.画像ファイルパス ); // Log出力用
			try
			{
				#region " 画像ファイルパスの有効性を確認する。"
				//-----------------
				if( this.画像ファイルパス.Nullまたは空である() )
				{
					Log.ERROR( $"画像ファイルパスが null または空文字列です。[{変数付きファイルパス}]" );
					return;
				}
				if( false == System.IO.File.Exists( this.画像ファイルパス ) )
				{
					Log.ERROR( $"画像ファイルが存在しません。[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion

				#region " 画像ファイルに対応できるデコーダを見つける。"
				//-----------------
				try
				{
					decoder = new BitmapDecoder(
						dr.WicImagingFactory2,
						this.画像ファイルパス,
						SharpDX.IO.NativeFileAccess.Read,
						DecodeOptions.CacheOnLoad );
				}
				catch( SharpDXException e )
				{
					Log.ERROR( $"画像ファイルに対応するコーデックが見つかりません。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion

				#region " 最初のフレームをデコードし、取得する。"
				//-----------------
				try
				{
					sourceFrame = decoder.GetFrame( 0 );
				}
				catch( SharpDXException e )
				{
					Log.ERROR( $"画像ファイルの最初のフレームのデコードに失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion

				#region " 32bitPBGRA へのフォーマットコンバータを生成する。"
				//-----------------
				try
				{
					// WICイメージングファクトリから新しいコンバータを生成。
					converter = new FormatConverter( dr.WicImagingFactory2 );

					// コンバータに変換元フレームや変換後フォーマットなどを設定。
					converter.Initialize(
						sourceRef: sourceFrame,
						dstFormat: SharpDX.WIC.PixelFormat.Format32bppPBGRA,
						dither: BitmapDitherType.None,
						paletteRef: null,
						alphaThresholdPercent: 0.0,
						paletteTranslate: BitmapPaletteType.MedianCut );
				}
				catch( SharpDXException e )
				{
					Log.ERROR( $"32bitPBGRA へのフォーマットコンバータの生成または初期化に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion

				#region " コンバータを使って、フレームを WICビットマップ経由で D2D ビットマップに変換する。"
				//-----------------
				try
				{
					// WIC ビットマップを D2D ビットマップに変換する。
					this.Bitmap?.Dispose();
					this.Bitmap = Bitmap1.FromWicBitmap(
						dr.D2DContext1,
						converter,
						bitmapProperties1 );
				}
				catch( SharpDXException e )
				{
					Log.ERROR( $"Direct2D1.Bitmap1 への変換に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion

				//Log.Info( $"{Utilities.現在のメソッド名}: 画像を生成しました。[{変数付きファイルパス}]" );
			}
			finally
			{
				converter?.Dispose();
				sourceFrame?.Dispose();
				decoder?.Dispose();
			}
		}
	}
}
