﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using SharpDX;
using SharpDX.Direct2D1;
using SharpDX.Direct3D11;
using SharpDX.DirectWrite;
using SharpDX.DXGI;
using SharpDX.Mathematics.Interop;
using SharpDX.MediaFoundation;
using SharpDX.WIC;
using D2DDevice1 = SharpDX.Direct2D1.Device1;
using D2DDeviceContext1 = SharpDX.Direct2D1.DeviceContext1;
using D2DFactory2 = SharpDX.Direct2D1.Factory2;
using DWFactory = SharpDX.DirectWrite.Factory;
using FDK.同期;

namespace FDK.メディア
{
	public class グラフィックコンテキスト : IDisposable
	{
		/// <summary>
		///		設計時（コーディング時や画像作成時）における画面サイズ。
		///		実際の物理画面サイズ（ウィンドウサイズやモニタサイズ）には依存しない。
		/// </summary>
		public Size2F 設計画面サイズdpx
		{
			get;
			set;
		} = new Size2F( 1920f, 1080f );

		/// <summary>
		///		実際の画面（ウィンドウやモニタ）のサイズ。
		///		設計画面サイズには依存しない。
		/// </summary>
		public Size2F 物理画面サイズpx
		{
			get;
			set;
		} = new Size2F( 0f, 0f );   // (0, 0) は、サイズ依存リソース無効の印。

		public IntPtr ウィンドウハンドル
		{
			get;
			set;
		} = IntPtr.Zero;

		public float 視野角deg
		{
			get;
			set;
		} = 45f;

		public Matrix ビュー変換行列
		{
			get
			{
				var カメラの位置 = new Vector3( 0f, 0f, ( -2f * this._dz( this.設計画面サイズdpx.Height, this.視野角deg ) ) );
				var カメラの注視点 = new Vector3( 0f, 0f, 0f );
				var カメラの上方向 = new Vector3( 0f, 1f, 0f );

				var mat = Matrix.LookAtLH( カメラの位置, カメラの注視点, カメラの上方向 );

				mat.Transpose();  // 転置

				return mat;
			}
		}

		public Matrix 射影変換行列
		{
			get
			{
				float dz = this._dz( this.設計画面サイズdpx.Height, this.視野角deg );

				var mat = Matrix.PerspectiveFovLH(
					MathUtil.DegreesToRadians( 視野角deg ),
					設計画面サイズdpx.Width / 設計画面サイズdpx.Height,  // アスペクト比
					-dz,  // 前方投影面までの距離
					dz ); // 後方投影面までの距離

				mat.Transpose();  // 転置

				return mat;
			}
		}

		public DXGIDeviceManager DXGIDeviceManager
		{
			get
				=> this._DXGIDeviceManager;
		}

		public SwapChain1 SwapChain1
		{
			get
				=> this._SwapChain1;
		}

		public RenderTargetView D3DRenderTargetView
		{
			get
				=> this._D3DRenderTargetView;
		}

		public RawViewportF[] D3DViewPort
		{
			get
				=> this._D3DViewPort;
		}

		public Texture2D D3DDepthStencil
		{
			get
				=> this._D3DDepthStencil;
		}

		public DepthStencilView D3DDepthStencilView
		{
			get
				=> this._D3DDepthStencilView;
		}

		public DepthStencilState D3DDepthStencilState
		{
			get
				=> this._D3DDepthStencilState;
		}

		public DeviceDebug D3DDeviceDebug
		{
			get
				=> this._D3DDeviceDebug;
		}

		public D2DFactory2 D2DFactory2
		{
			get
				=> this._D2DFactory2;
		}

		public DWFactory DWriteFactory
		{
			get
				=> this._DWriteFactory;
		}

		public ImagingFactory2 WicImagingFactory2
		{
			get
				=> this._WicImagingFactory2;
		}

		public D2DDevice1 D2DDevice1
		{
			get
				=> this._D2DDevice1;
		}

		public D2DDeviceContext1 D2DContext1
		{
			get
				=> this._D2DContext1;
		}

		public Bitmap1 D2DRenderTargetBitmap
		{
			get
				=> this._D2DRenderTargetBitmap;
		}

		public float 拡大率DPXtoPX横方向
		{
			get
				=> this.物理画面サイズpx.Width / this.設計画面サイズdpx.Width;
		}

		public float 拡大率DPXtoPX縦方向
		{
			get
				=> this.物理画面サイズpx.Height / this.設計画面サイズdpx.Height;
		}

		public Matrix3x2 拡大行列DPXtoPX
		{
			get
				=> Matrix3x2.Scaling( this.拡大率DPXtoPX横方向, this.拡大率DPXtoPX縦方向 );
		}

		public float 拡大率PXtoDPX横方向
		{
			get
				=> 1f / this.拡大率DPXtoPX横方向;
		}

		public float 拡大率PXtoDPX縦方向
		{
			get
				=> 1f / this.拡大率DPXtoPX縦方向;
		}

		public Matrix3x2 拡大行列PXtoDPX
		{
			get
				=> Matrix3x2.Scaling( this.拡大率PXtoDPX横方向, this.拡大率PXtoDPX縦方向 );
		}


		public グラフィックコンテキスト( Size2F 設計画面サイズdpx )
		{
			this.設計画面サイズdpx = 設計画面サイズdpx;
		}

		public void Dispose()
		{
			this.すべてのリソースを解放する();
		}

		public void すべてのリソースを作成する( System.Drawing.Size バックバッファサイズpx, IntPtr ウィンドウハンドル )
		{
			Log.BeginInfo( $"{Utilities.現在のメソッド名}" );

			this.物理画面サイズpx = new Size2F( バックバッファサイズpx.Width, バックバッファサイズpx.Height );
			this.ウィンドウハンドル = ウィンドウハンドル;

			this._すべてのリソースを作成する();

			Log.EndInfo( $"{Utilities.現在のメソッド名}" );
		}

		public void すべてのリソースを解放する()
		{
			Log.BeginInfo( $"{Utilities.現在のメソッド名}" );

			// D3D 関連
			this._SwapChain1?.SetFullscreenState( fullscreen: false, targetRef: null );    // スワップチェインをウインドウモードにする。
			this.サイズに依存するリソースを解放する();

			Utilities.解放する( ref this._SwapChain1 );
			Utilities.解放する( ref this._DXGIDeviceManager );

			// その他
			Utilities.解放する( ref this._WicImagingFactory2 );
			Utilities.解放する( ref this._DWriteFactory );
			Utilities.解放する( ref this._D2DFactory2 );

			this._D3DDeviceDebug?.ReportLiveDeviceObjects( ReportingLevel.Detail );
			Utilities.解放する( ref this._D3DDeviceDebug );

			Log.EndInfo( $"{Utilities.現在のメソッド名}" );
		}

		public void サイズに依存するリソースを作成する()
		{
			Log.BeginInfo( $"{Utilities.現在のメソッド名}" );

			#region " スワップチェーンのサイズを変更する。"
			//----------------
			{
				Debug.Assert( null != this.SwapChain1 ); // スワップチェーンは（デバイスとともに）すでに生成されていること。

				// ResizeTarget は、全画面モードにしたとき、モニタ画面の解像度も変更する。
				//var mode = new SharpDX.DXGI.ModeDescription(
				//	width: (int) this.物理画面サイズpx.Width,
				//	height: (int) this.物理画面サイズpx.Height,
				//	refreshRate: new SharpDX.DXGI.Rational( 60000, 1001 ),	// 多くのモニタは(60,1)じゃない；一致しないとFlipじゃなくBlitになるので注意(MSDN)。
				//	format: SharpDX.DXGI.Format.B8G8R8A8_UNorm );
				//this.SwapChain.ResizeTarget( ref mode );

				// ResizeBuffers は、全画面モードにしたとき、モニタの解像度を変更しない。
				// ただし、swapChainFlags に AllowModeSwitch を指定すると変更される（ResizeTargetと同じ挙動になる）。
				this.SwapChain1.ResizeBuffers(
					bufferCount: 2,
					width: ( int ) this.物理画面サイズpx.Width,
					height: ( int ) this.物理画面サイズpx.Height,
					newFormat: Format.B8G8R8A8_UNorm,
					swapChainFlags: SwapChainFlags.None );
				//swapChainFlags: SwapChainFlags.AllowModeSwitch );
			}
			//----------------
			#endregion

			// バックバッファを使って、D3D / D2D 関連のリソースを作成する。
			using( var backBuffer = Texture2D.FromSwapChain<Texture2D>( this._SwapChain1, 0 ) )
			{
				using( var d3dLock = new AutoD3DDeviceLock( this.DXGIDeviceManager, out SharpDX.Direct3D11.Device d3dDevice ) )
				using( d3dDevice )
				{
					// D3D 関連

					#region " レンダーターゲットビューを作成する。"
					//----------------
					this._D3DRenderTargetView = new RenderTargetView( d3dDevice, backBuffer );
					//----------------
					#endregion

					#region " 深度ステンシルテクスチャを作成する。"
					//----------------
					var descDepth = backBuffer.Description;
					//descDepth.Width = backBuffer.Description.Width;	→ backBuffer に同じ
					//descDepth.Height = backBuffer.Description.Height;	→ 同上
					descDepth.MipLevels = 1;    // ミップマップレベル数
					descDepth.ArraySize = 1;    // 配列サイズ
					descDepth.Format = Format.D32_Float;   // フォーマット（深度のみ）
					descDepth.Usage = ResourceUsage.Default; // デフォルト使用法
					descDepth.BindFlags = BindFlags.DepthStencil;    // 深度ステンシル
					descDepth.CpuAccessFlags = CpuAccessFlags.None;  // CPUからはアクセスしない
					descDepth.OptionFlags = ResourceOptionFlags.None;    // その他の設定なし
					this._D3DDepthStencil = new Texture2D( d3dDevice, descDepth );
					//----------------
					#endregion

					#region " 深度ステンシルビューを作成する。"
					//----------------
					var descDSV = new DepthStencilViewDescription() {
						Format = descDepth.Format,
						Dimension = DepthStencilViewDimension.Texture2D,
						Flags = DepthStencilViewFlags.None,
						Texture2D = new DepthStencilViewDescription.Texture2DResource() {
							MipSlice = 0,
						},
					};
					this._D3DDepthStencilView = new DepthStencilView( d3dDevice, this._D3DDepthStencil, descDSV );
					//----------------
					#endregion

					#region " 深度ステンシルステートを作成する。"
					//----------------
					var DepthSencil = new DepthStencilStateDescription() {
						IsDepthEnabled = true,  // 深度テストあり
						DepthWriteMask = DepthWriteMask.All,     // 書き込む
						DepthComparison = Comparison.Less,   // 手前の物体を描画
						IsStencilEnabled = false,   // ステンシルテストなし。
						StencilReadMask = 0,    // ステンシル読み込みマスク。
						StencilWriteMask = 0,   // ステンシル書き込みマスク。

						// 面が表を向いている場合のステンシル・テストの設定
						FrontFace = new DepthStencilOperationDescription() {
							FailOperation = StencilOperation.Keep,   // 維持
							DepthFailOperation = StencilOperation.Keep,  // 維持
							PassOperation = StencilOperation.Keep,   // 維持
							Comparison = Comparison.Never,   // 常に失敗
						},

						// 面が裏を向いている場合のステンシル・テストの設定
						BackFace = new DepthStencilOperationDescription() {
							FailOperation = StencilOperation.Keep,   // 維持
							DepthFailOperation = StencilOperation.Keep,  // 維持
							PassOperation = StencilOperation.Keep,   // 維持
							Comparison = Comparison.Always,  // 常に成功
						},
					};
					this._D3DDepthStencilState = new DepthStencilState( d3dDevice, DepthSencil );
					//----------------
					#endregion

					#region " ビューポートを作成する。"
					//----------------
					this._D3DViewPort[ 0 ] = new RawViewportF() {
						X = 0.0f,
						Y = 0.0f,
						Width = ( float ) backBuffer.Description.Width, // 物理画面単位[px]
						Height = ( float ) backBuffer.Description.Height,   // 物理画面単位[px]
						MinDepth = 0.0f,
						MaxDepth = 1.0f,
					};
					//----------------
					#endregion

					テクスチャ.全インスタンスで共有するリソースを作成する( this );

					// D2D 関連

					using( var backsurface = Surface.FromSwapChain( this._SwapChain1, 0 ) )
					{
						#region " D2DDevice を作成する。"
						//----------------
						using( var dxgiDevice2 = d3dDevice.QueryInterfaceOrNull<SharpDX.DXGI.Device2>() )
						{
							if( null == dxgiDevice2 )
								throw new FDKException( "Direct3D11デバイスが、IDXGIDevice2 をサポートしていません。" );

							this._D2DDevice1 = new D2DDevice1( this.D2DFactory2, dxgiDevice2 );
						}
						//----------------
						#endregion

						#region " D2DDeviceを作成する。"
						//----------------
						using( var dxgiDevice2 = d3dDevice.QueryInterfaceOrNull<SharpDX.DXGI.Device2>() )
						{
							if( null == dxgiDevice2 )
								throw new FDKException( "Direct3D11デバイスが、IDXGIDevice2 をサポートしていません。" );

							this._D2DDevice1 = new D2DDevice1( this.D2DFactory2, dxgiDevice2 );
						}
						//----------------
						#endregion

						#region " D2Dの既定のデバイスコンテキストを作成する。"
						//----------------
						this._D2DContext1 = new D2DDeviceContext1( this.D2DDevice1, DeviceContextOptions.EnableMultithreadedOptimizations );

						// 現在のディスプレイDPI を取得し、D2DContext に設定する。
						//this.D2DContext1.DotsPerInch = this.D2DFactory2.DesktopDpi;
						this.D2DContext1.DotsPerInch = new Size2F( 96f, 96f );
						//----------------
						#endregion

						#region " D2Dの既定のレンダーターゲットビットマップを作成する。"
						//----------------
						var dpi = this.D2DContext1.DotsPerInch;

						// DXGIスワップチェーンのバックバッファとデータを共有するD2Dターゲットビットマップを作成する。ビューではなく共有リソース。
						this._D2DRenderTargetBitmap = new Bitmap1(
							this.D2DContext1,   // このコンテキストを通じて、
							backsurface, // バックバッファとデータを共有する。
							new BitmapProperties1(
								new SharpDX.Direct2D1.PixelFormat( backsurface.Description.Format, SharpDX.Direct2D1.AlphaMode.Ignore ),
								dpi.Width,
								dpi.Height,
								BitmapOptions.Target | BitmapOptions.CannotDraw ) );

						// ここでもうD2Dのレンダーターゲットとして登録しておく。
						this.D2DContext1.Target = this._D2DRenderTargetBitmap;
						//----------------
						#endregion

						#region " テキストのアンチエイリアシングを設定する。"
						//----------------
						// Grayscale が、すべての Windows ストアアプリで推奨される。らしい。
						this.D2DContext1.TextAntialiasMode = SharpDX.Direct2D1.TextAntialiasMode.Grayscale;
						//----------------
						#endregion
					}
				}
			}

			Log.EndInfo( $"{FDK.Utilities.現在のメソッド名}" );
		}

		public void サイズに依存するリソースを解放する()
		{
			Log.BeginInfo( $"{Utilities.現在のメソッド名}" );

			// D2D 関連

			if( null != this._D2DContext1 )
				this._D2DContext1.Target = null;

			Utilities.解放する( ref this._D2DRenderTargetBitmap );
			Utilities.解放する( ref this._D2DContext1 );
			Utilities.解放する( ref this._D2DDevice1 );

			// D3D 関連

			using( var d3dLock = new AutoD3DDeviceLock( this.DXGIDeviceManager, out SharpDX.Direct3D11.Device d3dDevice ) )
			using( d3dDevice )
			using( var d3dContext = d3dDevice.ImmediateContext )
			{
				d3dContext.ClearState();
				d3dContext.OutputMerger.ResetTargets();

				テクスチャ.全インスタンスで共有するリソースを解放する();

				Utilities.解放する( ref this._D3DDepthStencilState );
				Utilities.解放する( ref this._D3DDepthStencilView );
				Utilities.解放する( ref this._D3DDepthStencil );
				Utilities.解放する( ref this._D3DRenderTargetView );
				//Utilities.解放する( ref this.bs_SwapChain ); → スワップチェーンは解放しない（これの生成・解放はデバイスとセットで行う）。

				// (0,0)は、サイズ依存リソース無効の印。
				this.物理画面サイズpx = new Size2F( 0f, 0f );
			}

			Log.EndInfo( $"{Utilities.現在のメソッド名}" );
		}

		public void D3Dデバイスが消失していれば再構築する( out bool 異常状態なのでアプリを終了せよ )
		{
			異常状態なのでアプリを終了せよ = false;
			Result 削除理由;

			using( var d3dLock = new AutoD3DDeviceLock( this.DXGIDeviceManager, out SharpDX.Direct3D11.Device d3dDevice ) )
			using( d3dDevice )
			{
				削除理由 = d3dDevice.DeviceRemovedReason;
			}

			if( 削除理由.Success )
				return; // デバイスは消失していない。

			var エラー詳細 = new[] {
					new { Code = SharpDX.DXGI.ResultCode.DeviceHung.Code, Info = SharpDX.DXGI.ResultCode.DeviceHung.ApiCode, Rebuild = true },
					new { Code = SharpDX.DXGI.ResultCode.DeviceReset.Code, Info = SharpDX.DXGI.ResultCode.DeviceReset.ApiCode, Rebuild = true },
					new { Code = SharpDX.DXGI.ResultCode.DeviceRemoved.Code, Info = SharpDX.DXGI.ResultCode.DeviceRemoved.ApiCode, Rebuild = false },
					new { Code = SharpDX.DXGI.ResultCode.DriverInternalError.Code, Info = SharpDX.DXGI.ResultCode.DriverInternalError.ApiCode, Rebuild = false },
					new { Code = SharpDX.DXGI.ResultCode.InvalidCall.Code, Info = SharpDX.DXGI.ResultCode.InvalidCall.ApiCode, Rebuild = false },
				}.First( ( エラー ) => エラー.Code == 削除理由.Code );  // 見つからないなら System.InvalidOperationException 。

			Trace.WriteLine( $"D3Dデバイスが消失しました: {エラー詳細.Info}" );

			if( エラー詳細.Rebuild )
			{
				this.すべてのリソースを解放する();
				this._すべてのリソースを作成する();
			}
			else
			{
				異常状態なのでアプリを終了せよ = true;
			}
		}

		public void D2DContextの設定をリセットする( D2DDeviceContext1 context1 )
		{
			context1.Transform = Matrix3x2.Identity;
			context1.PrimitiveBlend = PrimitiveBlend.SourceOver;
		}


		private D2DFactory2 _D2DFactory2 = null;

		private DWFactory _DWriteFactory = null;

		private ImagingFactory2 _WicImagingFactory2 = null;

		private DXGIDeviceManager _DXGIDeviceManager = null;

		private SwapChain1 _SwapChain1 = null;

		private RawViewportF[] _D3DViewPort = new RawViewportF[ 1 ];

		private DepthStencilState _D3DDepthStencilState = null;

		private RenderTargetView _D3DRenderTargetView = null;

		private Texture2D _D3DDepthStencil = null;

		private DepthStencilView _D3DDepthStencilView = null;

		private DeviceDebug _D3DDeviceDebug = null;

		private D2DDevice1 _D2DDevice1 = null;

		private D2DDeviceContext1 _D2DContext1 = null;

		private Bitmap1 _D2DRenderTargetBitmap = null;


		private void _すべてのリソースを作成する()
		{
			Log.BeginInfo( $"{Utilities.現在のメソッド名}" );

			// これらが呼び出し前に設定されていること。
			Debug.Assert( ( 0f < this.物理画面サイズpx.Width ) && ( 0f < this.物理画面サイズpx.Height ) );
			Debug.Assert( IntPtr.Zero != this.ウィンドウハンドル );

			#region " D2DFactory2を作成する。"
			//----------------
			var デバッグレベル = DebugLevel.None;
#if DEBUG
			// プロジェクトがデバッグビルドに含まれている場合は、Direct2D デバッグレイヤーを SDK レイヤーを介して有効にする。
			デバッグレベル = DebugLevel.Information;
#endif
			this._D2DFactory2 = new D2DFactory2( SharpDX.Direct2D1.FactoryType.MultiThreaded, デバッグレベル );
			//----------------
			#endregion

			this._DWriteFactory = new DWFactory( SharpDX.DirectWrite.FactoryType.Shared );
			this._WicImagingFactory2 = new ImagingFactory2();
			this._DXGIDeviceManager = new DXGIDeviceManager();

			var d3dDevice = (SharpDX.Direct3D11.Device) null;

			#region " D3Dデバイスを作成する。"
			//----------------
			{
				// 作成フラグ
				var creationFlags = DeviceCreationFlags.BgraSupport; // D2Dをサポートするなら BgraSupport フラグが必須。
#if DEBUG
				// D3D11 Debugメッセージは、プロジェクトプロパティで「ネイティブコードのデバッグを有効にする」を ON にしないと表示されないので注意。
				//creationFlags |= DeviceCreationFlags.Debug;	// デバッグ レイヤーを有効にしてアプリケーションを実行すると、速度が大幅に低下します。(MSDN)
#endif
				d3dDevice = new SharpDX.Direct3D11.Device(
					SharpDX.Direct3D.DriverType.Hardware,
					creationFlags,
					new SharpDX.Direct3D.FeatureLevel[] {
						SharpDX.Direct3D.FeatureLevel.Level_11_1,
						SharpDX.Direct3D.FeatureLevel.Level_11_0,
						//SharpDX.Direct3D.FeatureLevel.Level_9_1,	// Windowsストアゲームの最小要件
					} );

				Log.Info( "D3Dデバイスを生成しました。" );
				Log.Info( $"機能レベル: {d3dDevice.FeatureLevel.ToString()}" );
			}
			//----------------
			#endregion

			using( d3dDevice )
			{
				// D3D 関連

				#region " スワップチェーンを作成する。"
				//----------------
				using( var dxgiDevice1 = d3dDevice.QueryInterface<SharpDX.DXGI.Device1>() )
				using( var dxgiAdapter = dxgiDevice1.Adapter )
				using( var dxgiFactory2 = dxgiAdapter.GetParent<SharpDX.DXGI.Factory2>() )
				{
					var swapChainDesc1 = new SwapChainDescription1() {
						BufferCount = 2,
						Width = (int) this.物理画面サイズpx.Width,
						Height = (int) this.物理画面サイズpx.Height,
						Format = Format.B8G8R8A8_UNorm,  // D2D をサポートするなら B8G8R8A8 を使う必要がある。
						Stereo = false,
						SampleDescription = new SampleDescription( 1, 0 ), // マルチサンプリングは使わない。
						SwapEffect = SwapEffect.FlipSequential,    // ストアアプリの認定要件。
						Scaling = Scaling.None,    // SwapEffect に Flip なんちゃらを使う場合は None にする必要がある。
						Usage = Usage.RenderTargetOutput,
						Flags = SwapChainFlags.None,
						//Flags = SwapChainFlags.AllowModeSwitch,
					};
					this._SwapChain1 = new SwapChain1(
						dxgiFactory2,
						d3dDevice,
						this.ウィンドウハンドル,
						ref swapChainDesc1 );

					dxgiDevice1.MaximumFrameLatency = 1;    // ストアアプリの認定要件
				}
				//----------------
				#endregion

				// デバイスからデバッグオブジェクトを取得する。
				this._D3DDeviceDebug = d3dDevice.QueryInterfaceOrNull<DeviceDebug>();

				// D3DDevice が ID3D11VideoDevice を実装してないならエラー。（Win8以降のPCでは実装されているはず。） 
				using( var videoDevice = d3dDevice.QueryInterfaceOrNull<VideoDevice>() )
				{
					if( null == videoDevice )
						throw new FDKException( "Direct3D11デバイスが、ID3D11VideoDevice をサポートしていません。" );
				}

				// マルチスレッドモードを ON に設定する。DXVAを使う場合は必須。
				using( var multithread = d3dDevice.QueryInterfaceOrNull<SharpDX.Direct3D.DeviceMultithread>() )
				{
					if( null == multithread )
						throw new FDKException( "Direct3D11デバイスが、ID3D10Multithread をサポートしていません。" );

					multithread.SetMultithreadProtected( true );
				}

				// DXGIデバイスマネージャに D3Dデバイスを登録する。
				this.DXGIDeviceManager.ResetDevice( d3dDevice );

				// すべての Windows イベントを無視する。具体的には PrintScreen と Alt+Enter 。
				using( var factory = this._SwapChain1.GetParent<SharpDX.DXGI.Factory>() )
				{
					factory.MakeWindowAssociation( ウィンドウハンドル, WindowAssociationFlags.IgnoreAll );
				}
			}

			this.サイズに依存するリソースを作成する();

			Log.EndInfo( $"{Utilities.現在のメソッド名}" );
		}

		private float _dz( float 高さdpx, float 視野角deg )
		{
			return ( float ) ( 高さdpx / ( 4.0 * Math.Tan( MathUtil.DegreesToRadians( 視野角deg / 2.0f ) ) ) );
		}
	}
}
