package org.sqlite.jdbc;

import java.io.IOException;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class JdbcPreparedStatementTest {

    public JdbcPreparedStatementTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    private static final String DRIVER_CLASS = "org.sqlite.Driver";
    private static final String DATABASE = System.getProperty("user.dir") + "/test/unittest.db";
    
    private static Connection newConnection() throws ClassNotFoundException, SQLException {
        Class.forName(DRIVER_CLASS);
        return DriverManager.getConnection("jdbc:sqlite:file:" + DATABASE);
    }
    
    @Test
    public void execute() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            stmt.close();
            
            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(?, ?, ?, ?)";
            PreparedStatement pstmt = conn.prepareStatement(sql);
            pstmt.setInt(1, 1);
            pstmt.setDouble(2, 2.22d);
            pstmt.setBytes(3, new byte[] { Byte.MAX_VALUE, Byte.MIN_VALUE, 1, 2, 3, });
            pstmt.setString(4, "TEST");
            assertFalse(pstmt.execute());
            assertEquals(1, pstmt.getUpdateCount());
            
            pstmt.setInt(1, 2);
            assertFalse(pstmt.execute());
            assertEquals(1, pstmt.getUpdateCount());

            pstmt.close();

            sql = "SELECT * FROM temp_tbl_1 WHERE ID = ?";
            pstmt = conn.prepareStatement(sql);
            pstmt.setInt(1, 1);
            assertTrue(pstmt.execute());
            assertEquals(-1, pstmt.getUpdateCount());
            ResultSet rs = pstmt.getResultSet();
            assertTrue(rs.next());
            assertEquals(1, rs.getInt(1));
            assertFalse(rs.next());
            rs.close();
            
            pstmt.setInt(1, 2);
            assertTrue(pstmt.execute());
            assertEquals(-1, pstmt.getUpdateCount());
            rs = pstmt.getResultSet();
            assertTrue(rs.next());
            assertEquals(2, rs.getInt(1));
            assertFalse(rs.next());
            rs.close();
            
            pstmt.close();
            
            
        } finally {
            conn.close();
        }
    }
    
    @Test
    public void detach() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            stmt.close();
            
            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(?, ?, ?, ?)";
            JdbcPreparedStatement pstmt = (JdbcPreparedStatement) conn.prepareStatement(sql);
            pstmt.setInt(1, 1);
            pstmt.setDouble(2, 2.22d);
            pstmt.setBytes(3, new byte[] { Byte.MAX_VALUE, Byte.MIN_VALUE, 1, 2, 3, });
            pstmt.setString(4, "TEST");
            assertFalse(pstmt.execute());
            assertEquals(1, pstmt.getUpdateCount());
            
            pstmt.setInt(1, 2);
            assertFalse(pstmt.execute());
            assertEquals(1, pstmt.getUpdateCount());

            pstmt.close();

            sql = "SELECT * FROM temp_tbl_1 WHERE ID = ?";
            pstmt = (JdbcPreparedStatement) conn.prepareStatement(sql);
            pstmt.setInt(1, 1);
            assertTrue(pstmt.execute());
            assertEquals(-1, pstmt.getUpdateCount());
            ResultSet rs = pstmt.getResultSet();
            assertNotNull(rs);
            assertTrue(rs.next());
            assertEquals(1, rs.getInt("ID"));
            assertEquals(2, 2.22d, rs.getDouble("rVALUE"));
            assertEquals("TEST", rs.getString("nVALUE"));
            
            pstmt.detach(rs);
            rs.beforeFirst();
            
            assertTrue(rs.next());
            assertEquals(1, rs.getInt(1));
            assertFalse(rs.next());
            rs.close();
            
            // detachするとパラメータにセットした値がクリアされるため rs.next()がfalseになる
            assertTrue(pstmt.execute());
            assertEquals(-1, pstmt.getUpdateCount());
            rs = pstmt.getResultSet();
            assertFalse(rs.next());
            rs.close();
            
            pstmt.close();
            
            
        } finally {
            conn.close();
        }
    }
    
    @Test
    public void setBlob() throws ClassNotFoundException, SQLException, IOException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", bVALUE BLOB"
                    + ")";
            stmt.executeUpdate(sql);
            
            sql = "INSERT INTO temp_tbl_2 VALUES(1, 'Hello, SQLite!')";
            stmt.executeUpdate(sql);
            assertEquals(1, stmt.getUpdateCount());

            sql = "SELECT ROWID, bVALUE FROM temp_tbl_2";
            JdbcResultSet rs = (JdbcResultSet) stmt.executeQuery(sql);
            
            assertTrue(rs.next());
            Blob blob = rs.getBlob(2, rs.getLong(1), true);
            rs.close();
            
            blob.setBinaryStream(1).write("Hello, Blob!".getBytes("UTF8"));

            sql = "INSERT INTO temp_tbl_2 VALUES(2, ?)";
            JdbcPreparedStatement pstmt = (JdbcPreparedStatement) conn.prepareStatement(sql);
            pstmt.setBlob(1, blob);
            assertFalse(pstmt.execute());
            assertEquals(1, pstmt.getUpdateCount());
            pstmt.close();
            
            sql = "SELECT bVALUE FROM temp_tbl_2 WHERE ROWID = 2";
            rs = (JdbcResultSet) stmt.executeQuery(sql);
            
            assertTrue(rs.next());
            assertEquals("Hello, Blob!e!", rs.getString(1));
            rs.close();
            pstmt.close();
            
        } finally {
            conn.close();
        }        
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void executeQuery() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            stmt.close();
            
            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(?, ?, ?, ?)";
            PreparedStatement pstmt = conn.prepareStatement(sql);
            pstmt.setInt(1, 1);
            pstmt.setDouble(2, 2.22d);
            pstmt.setBytes(3, new byte[] { Byte.MAX_VALUE, Byte.MIN_VALUE, 1, 2, 3, });
            pstmt.setString(4, "TEST");
            assertFalse(pstmt.execute());
            assertEquals(1, pstmt.getUpdateCount());
            
            pstmt.setInt(1, 2);
            assertFalse(pstmt.execute());
            assertEquals(1, pstmt.getUpdateCount());

            pstmt.close();

            sql = "SELECT * FROM temp_tbl_1 WHERE ID = ?";
            pstmt = conn.prepareStatement(sql);
            pstmt.setInt(1, 1);
            ResultSet rs = pstmt.executeQuery();
            assertTrue(rs.next());
            assertEquals(1, rs.getInt(1));
            assertFalse(rs.next());
//            rs.close();
            
            pstmt.setInt(1, 2); // library routine called out of sequence
            rs = pstmt.executeQuery();
            assertTrue(rs.next());
            assertEquals(2, rs.getInt(1));
            assertFalse(rs.next());
            rs.close();
            
            pstmt.close();
            
            
        } finally {
            conn.close();
        }
    }
}