package org.sqlite;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.sqlite.jdbc.TransactionType;
import org.sqlite.udf.Context;
import org.sqlite.schema.ColumnMetaData;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.sqlite.jdbc.JdbcConnection;
import org.sqlite.swig.SQLite3;
import org.sqlite.swig.SQLite3.SQLite3StmtPtrPtr;
import static org.sqlite.swig.SQLite3Constants.*;
import org.sqlite.swig.SWIGTYPE_p_p_char;
import org.sqlite.udf.Function;
import org.sqlite.udf.ScalarFunction;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class DatabaseTest {

    public DatabaseTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    @Test(expected = java.lang.UnsatisfiedLinkError.class)
    public void newThrowUnsatisfiedLinkError() throws IOException, SQLException {
        new Database(null, null).close();
    }

    @Test
    public void openWithTempDir() throws ClassNotFoundException, SQLException, IOException {
        Class.forName("org.sqlite.Driver");
        final Map<String, String> info = new HashMap<String, String>();
        info.put("TEMP_DIR", "C:\\temp");
        final Database db = new Database(DATABASE, info);
        try {
            List<String[]> ret = db.getTable("PRAGMA temp_store_directory", null);
            assertEquals(2, ret.size());
            String[] values = ret.get(0);
            assertEquals(1, values.length);
            assertEquals("temp_store_directory", values[0]);
            
            values = ret.get(1);
            assertEquals(1, values.length);
            assertEquals("C:\\temp", values[0]);
            
        } finally {
            db.close();
        }
    }

    @Test
    public void openWithPRAGMA() throws ClassNotFoundException, SQLException, IOException {
        Class.forName("org.sqlite.Driver");
        final Map<String, String> info = new HashMap<String, String>();
        info.put("PRAGMA temp_store_directory ", " 'C:\\WINDOWS\\Temp'");
        final Database db = new Database(DATABASE, info);
        try {
            List<String[]> ret = db.getTable("PRAGMA temp_store_directory", null);
            assertEquals(2, ret.size());
            String[] values = ret.get(0);
            assertEquals(1, values.length);
            assertEquals("temp_store_directory", values[0]);
            
            values = ret.get(1);
            assertEquals(1, values.length);
            assertEquals("C:\\WINDOWS\\Temp", values[0]);
            
        } finally {
            db.close();
        }
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void openFailed() throws SQLException, ClassNotFoundException, IOException {
        Class.forName("org.sqlite.Driver");
        final Database db = new Database("\\Program Files", null);
        db.close();
    }
    
    private static final String DATABASE = System.getProperty("user.dir") + "/test/unittest.db";
    
    private static Database newDatabase() throws ClassNotFoundException, SQLException {
        Class.forName("org.sqlite.Driver");
        return new Database(DATABASE, null);
    }
    
//    @Test
//    public void getURL() throws SQLException, ClassNotFoundException {
//        final Database db = newDatabase();
//        
//        assertEquals(SCHEME + DATABASE, db.getURL());
//        assertFalse((SCHEME.toUpperCase() + DATABASE).equals(db.getURL()));
//        assertFalse((SCHEME + DATABASE.toUpperCase()).equals(db.getURL()));
//
//        db.close();
//    }
    
    @Test
    public void getProductName() {
        assertEquals("SQLite", Database.getProductName());
    }
    
    @Test
    public void isClosed() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        assertFalse(db.isClosed());
        
        db.close();
        assertTrue(db.isClosed());
    }
    
    @Test
    public void transaction() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        assertTrue(db.getAutoCommit());
        
        db.beginTransaction(null);
        assertFalse(db.getAutoCommit());
        
        db.rollbackTransaction();
        assertTrue(db.getAutoCommit());
        
        db.beginTransaction(null);
        assertFalse(db.getAutoCommit());
        
        db.commitTransaction();
        assertTrue(db.getAutoCommit());
        
        db.close();
    }
    
    @Test
    public void busyTimeout() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        assertEquals(0, db.getBusyTimeout());
        
        db.setBusyTimeout(500);
        assertEquals(500, db.getBusyTimeout());
        
        db.setBusyTimeout(-500);
        assertEquals(0, db.getBusyTimeout());
        
        db.close();
    }
    
    @Test
    public void execute() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        String sql= null;
        db.execute(sql);
        
        sql= "";
        db.execute(sql);
        
        sql= "SELECT * FROM sqlite_master";
        db.execute(sql);
        
        sql= "SELECT * FROM sqlite_master WHERE type = ?";
        db.execute(sql);
        
        db.close();
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void executeSyntaxError() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        String sql= "SELECT * FROM";
        db.execute(sql);
        
        db.close();
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void executeTableNotFound() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        String sql= "SELECT * FROM _NOT_FOUND_TABLE_";
        db.execute(sql);
        
        db.close();
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void executeColumnNotFound() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        String sql = "SELECT * FROM sqlite_master WHERE _NOT_FOUND_COLUMN_ = ?";
        db.execute(sql);
        
        db.close();
    }
    
    @Test(expected = java.lang.NullPointerException.class)
    public void pragmaThrowNullPointerException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        db.pragma(null);
        db.close();
    }
    
    @Test
    public void pragma() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        
        // confirm default valie
        stmt = db.prepare("PRAGMA read_uncommitted");
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(0, stmt.getInt(1));
        stmt.close();
        
        stmt = db.prepare("PRAGMA full_column_names");
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(0, stmt.getInt(1));
        stmt.close();

        stmt = db.prepare("PRAGMA cache_size");
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(2000, stmt.getInt(1));
        stmt.close();

        // execute PRAGMA
        final String[] commands
                = new String[] {
                        "read_uncommitted = on",
                        "full_column_names = on",
                        "cache_size = 9876",
                    };
        db.pragma(commands);
        
        // validate pragma() funcation.
        stmt = db.prepare("PRAGMA read_uncommitted");
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(1, stmt.getInt(1));
        stmt.close();
        
        stmt = db.prepare("PRAGMA full_column_names");
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(1, stmt.getInt(1));
        stmt.close();

        stmt = db.prepare("PRAGMA cache_size");
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(9876, stmt.getInt(1));
        stmt.close();
        
        db.close();
    }
    
    @Test(expected = java.lang.NullPointerException.class)
    public void prepareThrowSqlNullPointerException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
        Statement stmt = db.prepare(null, ppStmt);
        stmt.close();
        ppStmt.delete();
        
        db.close();
    }
    
    @Test(expected = java.lang.NullPointerException.class)
    public void prepareThrowSQLite3StmtPtrPtrNullPointerException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        Statement stmt = db.prepare("SELECT * FROM sqlite_master", null);
        stmt.close();

        db.close();
    }
    
    @Test
    public void prepareWithSQLite3StmtPtrPtr() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
        Statement stmt = db.prepare("SELECT * FROM sqlite_master", ppStmt);
        stmt.step();
        stmt.close();
        ppStmt.delete();
        
        db.close();
    }
    
    @Test(expected = java.lang.NullPointerException.class)
    public void prepareThrowNullPointerException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        Statement stmt = db.prepare(null);
        stmt.close();

        db.close();
    }
    
    @Test
    public void prepare() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        
        Statement stmt = db.prepare("SELECT * FROM sqlite_master");
        stmt.step();
        stmt.close();
        
        db.close();
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void prepareMultiple() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        
        try {
            String sql = null;

            sql = "SELECT * FROM sqlite_master;"
                + "SELECT * FROM sqlite_temp_master;"
                + ";";
            List<Statement> stmts = db.prepareMultiple(sql);
            assertEquals(3, stmts.size());
            for (final Statement stmt : stmts) {
                stmt.step();
                stmt.close();
            }
        } finally {
            db.close();
        }
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void interruptThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        try {
            String sql
                    = "SELECT * FROM("
                        + "SELECT 'One' UNION ALL "
                        + "SELECT 'Two' UNION ALL "
                        + "SELECT 'Three' UNION ALL "
                        + "SELECT 'Four' UNION ALL "
                        + "SELECT 'Five'"
                    +")";
            Statement stmt = db.prepare(sql);
            stmt.step();
            db.interrupt();
            stmt.step();
            stmt.close();
        } catch (SQLException ex) {
            assertEquals(SQLITE_INTERRUPT, ex.getErrorCode());
            throw ex;
            
        } finally {        
            db.close();
        }
    }
    
    @Test
    public void interrup() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        Statement stmt = db.prepare("SELECT * FROM sqlite_master");
        db.interrupt();
        stmt.close();
        
        db.close();
    }
    
    @Test
    public void changes() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")"
            );
        assertEquals(0, db.changes());
        
        db.beginTransaction(TransactionType.IMMEDIATE);
        Statement stmt = db.prepare("INSERT INTO temp_tbl_1(ID, VALUE) VALUES(@id, @value)");
        for (int i = 0; i < 5; ++i) {
            stmt.reset();
            stmt.bindInt(stmt.getParameterIndex("@id"), i);
            
            byte[] text = new byte[5];
            for (int j = 0; j < text.length; ++j) {
                text[j] = (byte) ('A' + i + j * 5);
            }
            stmt.bindText(stmt.getParameterIndex("@value"), new String(text));
            stmt.execute();
            assertEquals(1, db.changes());
        }
        stmt.close();
        db.commitTransaction();

        stmt = db.prepare("UPDATE temp_tbl_1 SET VALUE = lower(VALUE) WHERE ID BETWEEN @from AND @to");
        stmt.bindInt(stmt.getParameterIndex("@from"), 1);
        stmt.bindInt(stmt.getParameterIndex("@to"), 4);
        stmt.execute();
        assertEquals(4, db.changes());
        stmt.close();
        
        db.close();
    }
    
    @Test
    public void totalChanges() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")"
            );
        assertEquals(0, db.totalChanges());

        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2_bak("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")"
            );
        assertEquals(0, db.totalChanges());

        db.execute(
                "CREATE TEMPORARY TRIGGER backup_temp_tbl_2 UPDATE OF VALUE ON temp_tbl_2"
                    + " FOR EACH ROW WHEN NOT EXISTS(SELECT * FROM temp_tbl_2_bak WHERE ID = OLD.ID)"
                    + " BEGIN"
                        + " INSERT INTO temp_tbl_2_bak VALUES(OLD.ID, OLD.VALUE);"
                    + " END;"
            );
        assertEquals(0, db.totalChanges());
        
        db.beginTransaction(TransactionType.IMMEDIATE);
        Statement stmt = db.prepare("INSERT INTO temp_tbl_2(ID, VALUE) VALUES(@id, @value)");
        int totalChanges = 0;
        for (int i = 0; i < 5; ++i) {
            stmt.reset();
            stmt.bindInt(stmt.getParameterIndex("@id"), i);
            
            byte[] text = new byte[5];
            for (int j = 0; j < text.length; ++j) {
                text[j] = (byte) ('A' + i + j * 5);
            }
            stmt.bindText(stmt.getParameterIndex("@value"), new String(text));
            stmt.execute();
            assertEquals(1, db.changes());
            totalChanges += db.changes();
            assertEquals(totalChanges, db.totalChanges());
        }
        stmt.close();
        db.commitTransaction();

        stmt = db.prepare("UPDATE temp_tbl_2 SET VALUE = lower(VALUE) WHERE ID BETWEEN @from AND @to");
        stmt.bindInt(stmt.getParameterIndex("@from"), 1);
        stmt.bindInt(stmt.getParameterIndex("@to"), 4);
        stmt.execute();
        assertEquals(4, db.changes());
        totalChanges += (db.changes() * 2); // Triger
        assertEquals(totalChanges, db.totalChanges());
        stmt.close();
        
        db.close();
    }
    
    @Test
    public void lastInsertRowId() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_3("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")"
            );
        
        db.beginTransaction(TransactionType.IMMEDIATE);
        Statement stmt = db.prepare("INSERT INTO temp_tbl_3(VALUE) VALUES(?1)");
        for (int i = 0; i < 5; ++i) {
            stmt.reset();
            stmt.bindText(1, "qwerty");
            stmt.execute();
            assertEquals((long) (i + 1), db.lastInsertRowId());
        }
        stmt.close();
        db.commitTransaction();
        
        db.close();
    }
    
    @Test(expected = java.lang.NullPointerException.class)
    public void addStatementThrowNullPointerException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        db.add(null);
        
        db.close();
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void addStatementThrowIllegalArgumentException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        final Statement stmt = new Statement(db, new SQLite3StmtPtrPtr());
        db.add(stmt);
        
        db.close();
    }
    
    @Test
    public void addStatement() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        final Database db2 = newDatabase();

        final Statement stmt = new Statement(db, new SQLite3StmtPtrPtr());
        db2.add(stmt);
        db2.remove(stmt);
        
        db2.close();
        db.close();

        assertTrue(stmt.isClosed());        
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void removeStatementThrowIllegalArgumentException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        final Statement stmt = new Statement(db, new SQLite3StmtPtrPtr());
        db.remove(null);
        
        db.close();
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void removeStatementThrowIllegalArgumentException2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        final Database db2 = newDatabase();
        final Statement stmt = new Statement(db, new SQLite3StmtPtrPtr());
        db2.add(stmt);

        final Statement stmt2 = new Statement(db, new SQLite3StmtPtrPtr());
        db2.remove(stmt2);
        
        db2.close();
        db.close();
    }
    
    @Test
    public void removeStatement() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();

        final Statement stmt = new Statement(db, new SQLite3StmtPtrPtr());
        db.remove(stmt);
        
        db.close();
        
        assertTrue(stmt.isClosed());
    }

    @Test
    public void getColumnMetaData() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        try {
            final Connection conn = new JdbcConnection(db, null);
            
            final java.sql.Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank' COLLATE NOCASE"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
        
            ColumnMetaData meta = db.getColumnMetaData(null, "temp_tbl_1", "ROWID");
            assertEquals("INTEGER", meta.dataType);
            assertEquals("BINARY", meta.collationSequenceName);
            assertFalse(meta.isNotNull);
            assertTrue(meta.isPrimaryKey);
            assertTrue(meta.isAutoIncrement);
            
            meta = db.getColumnMetaData(null, "temp_tbl_1", "ID");
            assertEquals("INTEGER", meta.dataType);
            assertEquals("BINARY", meta.collationSequenceName);
            assertTrue(meta.isNotNull);
            assertFalse(meta.isPrimaryKey);
            assertFalse(meta.isAutoIncrement);
            
            meta = db.getColumnMetaData(null, "temp_tbl_1", "tVALUE");
            assertEquals("TEXT", meta.dataType);
            assertEquals("NOCASE", meta.collationSequenceName);
            assertFalse(meta.isNotNull);
            assertFalse(meta.isPrimaryKey);
            assertFalse(meta.isAutoIncrement);
            
            
            conn.close();
            
        } finally {
            db.close();
        }
    }

    private static Connection newConnection(Database db) throws ClassNotFoundException, SQLException {
        return new JdbcConnection(db, null);
    }

    @Test
    public void createFunctions() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        try {
            final Function myFunc
                    = new ScalarFunction("myFunc") {
                            @Override
                            public void xFunc(Context ctx) throws SQLException {
                                System.out.println("called xFunc()!");
                                String val = ctx.getString(1);
                                ctx.result(val);
                            }
                        };
            assertFalse(myFunc.isRegistered());
            db.createFunction(myFunc);
            assertTrue(myFunc.isRegistered());
            
            final Connection conn = newConnection(db);
            final java.sql.Statement stmt = conn.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT myFunc('TEST') LIMIT 1");
            assertTrue(rs.next());
            assertEquals("TEST", rs.getString(1));
            rs.close();
            
            stmt.close();
            conn.close();
            
        } finally {
            db.close();
        }
    }
    
    @Test
    public void close() throws ClassNotFoundException, SQLException {
        final Database db = newDatabase();
        try {
            // nothing
        } finally {
            new Thread() {
                @Override
                public void run() {
                    try {
                        db.close();
                        System.out.println("close database.");
                    } catch (SQLException ex) {
                        Logger.getLogger(DatabaseTest.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }
            }.start();
        }
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void loadExtentionNotAuthorized() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        SWIGTYPE_p_p_char errmsg = null;
        try {
            final String ext = "C:\\Java\\sqlite3-onig_20070329\\sqlite3-onig.dll";
            errmsg = SQLite3.new_p_p_char();
            db.loadExtention(ext, null, errmsg);
            
        } catch (SQLException ex) {
            assertEquals(SQLITE_ERROR, ex.getErrorCode());
            final String msg = SQLite3.get_p_char(errmsg);
            assertNotNull(msg);
            System.out.println("errmsg is '" + msg + "'");
            throw ex;
            
        } finally {
            if (errmsg != null) {
                SQLite3.delete_p_p_char(errmsg);
            }
            db.close();
        }
    }
    
    /**
     * 
     * @throws java.lang.ClassNotFoundException
     * @throws java.sql.SQLException
     * @see <a href="http://d.hatena.ne.jp/yotaropg/20070329/1175181139">SQLite3と鬼車で正規表現検索</a>
     */
    @Test
    public void loadExtention() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        SWIGTYPE_p_p_char errmsg = null;
        try {
            db.enableLoadExtention();
            
            final String ext = "C:\\Java\\sqlite3-onig_20070329\\sqlite3-onig.dll";
//            final String ext = "C:/Documents and Settings/user/My Documents/Visual Studio 2008/Projects/sqlite3oniguruma/sqlite3oniguruma/Release/sqlite3oniguruma.dll";
//            final String ext = "C:/Documents and Settings/user/My Documents/Visual Studio 2008/Projects/sqlite3extensions/sqlite3regex/Release/sqlite3regex.dll";
            
            errmsg = SQLite3.new_p_p_char();
            db.loadExtention(ext, null, errmsg);

            final Connection conn = newConnection(db);
            final java.sql.Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank' COLLATE NOCASE"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            
            sql = "SELECT * FROM sqlite_temp_master WHERE name REGEXP 'tbl'";
            for (final ResultSet rs = stmt.executeQuery(sql); rs.next(); ) {
                final String name = rs.getString("name");
                System.out.println("name is '" + name + "'");
                assertEquals("temp_tbl_1", name);
            }
            
            stmt.close();
            conn.close();
            
        } catch (SQLException ex) {
            final String msg = SQLite3.get_p_char(errmsg);
            assertNotNull(msg);
            System.out.println("errmsg is '" + msg + "'");
            throw ex;
            
        } finally {
            if (errmsg != null) {
                SQLite3.delete_p_p_char(errmsg);
            }
            db.close();
        }
    }
    
    @Test
    public void isThreadSafe() {
        assertTrue(Database.isThreadSafe());
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void openReadOnly() throws ClassNotFoundException, SQLException, IOException {
        Class.forName("org.sqlite.Driver");
        final Map<String, String> info = new HashMap<String, String>();
        info.put("READ_ONLY", "");
        final Database db = new Database(DATABASE, info);
        try {
            final String sql = "CREATE TABLE tbl_temp(ID INTEGER PRIMARY KEY, VALUE TEXT)";
            final Statement stmt = db.prepare(sql);
            stmt.step();
            stmt.close();
            
        } catch (SQLException ex) {
            assertEquals(SQLITE_READONLY, ex.getErrorCode());
            throw ex;
            
        } finally {
            db.close();
        }
        
    }
    
    @Test
    public void openBlob() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        final byte[] x
                = new byte[] {
                        (byte) 0xE3, (byte) 0x81, (byte) 0x82, (byte) 0xE3, (byte) 0x81,
                        (byte) 0x84, (byte) 0xE3, (byte) 0x81, (byte) 0x86, (byte) 0xE3,
                        (byte) 0x81, (byte) 0x88, (byte) 0xE3, (byte) 0x81, (byte) 0x8A, (byte) 0x00,
                        (byte) 0xE3, (byte) 0x81, (byte) 0x82, (byte) 0xE3, (byte) 0x81,
                        (byte) 0x84, (byte) 0xE3, (byte) 0x81, (byte) 0x86, (byte) 0xE3,
                        (byte) 0x81, (byte) 0x88, (byte) 0xE3, (byte) 0x81, (byte) 0x8A, (byte) 0x00,
                    };
        org.sqlite.Blob blob = null;
        try {
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", VALUE BLOB"
                    + ")";
            db.execute(sql);
            sql = "INSERT INTO temp_tbl_1(VALUE) VALUES(?)";
            Statement stmt = db.prepare(sql);
            stmt.bindBytes(1, x);
            stmt.execute();
            assertEquals(1, db.changes());
            stmt.close();
            
            blob = db.openBlob("temp", "temp_tbl_1", "VALUE", 1, 0);
            final int len = blob.length();
            assertEquals(x.length, len);
            
            byte[] buff = new byte[len];
            blob.read(buff, buff.length, 0);
            for (int i = 0; i < len; ++i) {
                assertEquals(x[i], buff[i]);
                System.out.printf("%2X", buff[i]);
            }
            System.out.println();
            
        } finally {
            db.close();
            if (blob != null) {
                assertTrue(blob.isClosed());
            }
        }
    }
    
    @Test
    public void openBlobWritable() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        byte[] x
                = new byte[] {
                        (byte) 0xE3, (byte) 0x81, (byte) 0x82, (byte) 0xE3, (byte) 0x81,
                        (byte) 0x84, (byte) 0xE3, (byte) 0x81, (byte) 0x86, (byte) 0xE3,
                        (byte) 0x81, (byte) 0x88, (byte) 0xE3, (byte) 0x81, (byte) 0x8A, (byte) 0x00,
                        (byte) 0xE3, (byte) 0x81, (byte) 0x82, (byte) 0xE3, (byte) 0x81,
                        (byte) 0x84, (byte) 0xE3, (byte) 0x81, (byte) 0x86, (byte) 0xE3,
                        (byte) 0x81, (byte) 0x88, (byte) 0xE3, (byte) 0x81, (byte) 0x8A, (byte) 0x00,
                    };
        final int length = x.length;
        org.sqlite.Blob blob = null;
        try {
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", VALUE BLOB"
                    + ")";
            db.execute(sql);
            sql = "INSERT INTO temp_tbl_1(VALUE) VALUES(?)";
            Statement stmt = db.prepare(sql);
            stmt.bindBytes(1, x);
            stmt.execute();
            assertEquals(1, db.changes());
            stmt.close();
            
            blob = db.openBlob("temp", "temp_tbl_1", "VALUE", 1, 1);
            int len = blob.length();
            assertEquals(length, len);
            
            byte[] buff = new byte[len];
            blob.read(buff, buff.length, 0);
            for (int i = 0; i < len; ++i) {
                assertEquals(x[i], buff[i]);
                System.out.printf("%2X", buff[i]);
            }
            System.out.println();
            
            x = "Hello, SQLite!".getBytes("UTF8");
            blob.write(x, x.length, 0);
            blob.close();

            blob = db.openBlob("temp", "temp_tbl_1", "VALUE", 1, 1);
            len = blob.length();
            assertEquals(length, len);
            
            buff = new byte[len];
            blob.read(buff, buff.length, 0);
            for (int i = 0; i < len; ++i) {
                if (i < x.length) {
                    assertEquals(x[i], buff[i]);
                }
                System.out.printf("%2X", buff[i]);
            }
            System.out.println();
            
        } finally {
            db.close();
            if (blob != null) {
                assertTrue(blob.isClosed());
            }
        }
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void openBlobThrowSQLException() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        byte[] x
                = new byte[] {
                        (byte) 0xE3, (byte) 0x81, (byte) 0x82, (byte) 0xE3, (byte) 0x81,
                        (byte) 0x84, (byte) 0xE3, (byte) 0x81, (byte) 0x86, (byte) 0xE3,
                        (byte) 0x81, (byte) 0x88, (byte) 0xE3, (byte) 0x81, (byte) 0x8A, (byte) 0x00,
                        (byte) 0xE3, (byte) 0x81, (byte) 0x82, (byte) 0xE3, (byte) 0x81,
                        (byte) 0x84, (byte) 0xE3, (byte) 0x81, (byte) 0x86, (byte) 0xE3,
                        (byte) 0x81, (byte) 0x88, (byte) 0xE3, (byte) 0x81, (byte) 0x8A, (byte) 0x00,
                    };
        final int length = x.length;
        org.sqlite.Blob blob = null;
        try {
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", VALUE BLOB"
                    + ")";
            db.execute(sql);
            sql = "INSERT INTO temp_tbl_1(VALUE) VALUES(?)";
            Statement stmt = db.prepare(sql);
            stmt.bindBytes(1, x);
            stmt.execute();
            assertEquals(1, db.changes());
            stmt.close();
            
            blob = db.openBlob("temp", "temp_tbl_1", "VALUE", 1, 0);
            int len = blob.length();
            assertEquals(length, len);
            
            x = "Hello, SQLite!".getBytes("UTF8");
            blob.write(x, x.length, 0);
            blob.close();

        } catch (SQLException ex) {
            assertEquals(SQLITE_READONLY, ex.getErrorCode());
            throw ex;
                
        } finally {
            db.close();
            if (blob != null) {
                assertTrue(blob.isClosed());
            }
        }
    }
    
    @Test
    public void limit() throws ClassNotFoundException, SQLException {
        final Database db = newDatabase();
        try {
            int old = db.limit(SQLITE_LIMIT_LENGTH, -1);
            int newVal = old * 2;
            assertEquals(old, db.limit(SQLITE_LIMIT_LENGTH, newVal));
            
        } finally {
            db.close();
        }
    }

    @Test
    public void nextStatement() throws ClassNotFoundException, SQLException {
        final Database db = newDatabase();
        try {
            Statement stmt = db.nextStatement(null);
            assertNull(stmt);

            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", VALUE BLOB"
                    + ")";
            db.execute(sql);
            sql = "INSERT INTO temp_tbl_1(VALUE) VALUES(?)";
            Statement stmt2 = db.prepare(sql);

            stmt = db.nextStatement(null);
            assertNotNull(stmt);
            stmt.close();
            stmt = null;
            
            stmt = stmt2.nextStatement();
            assertNotNull(stmt);
            stmt.close();
            stmt2.close();

        } finally {
            db.close();
        }
    }

    @Test
    public void status() throws SQLException {
        final int[] params
                = new int[] {
                    SQLITE_STATUS_MEMORY_USED,
                    SQLITE_STATUS_PAGECACHE_USED,
                    SQLITE_STATUS_PAGECACHE_OVERFLOW,
                    SQLITE_STATUS_SCRATCH_USED,
                    SQLITE_STATUS_SCRATCH_OVERFLOW,
                    SQLITE_STATUS_MALLOC_SIZE,
                };
        for (final int op : params) {
            final int[] status = Database.status(op, false);
            System.out.println(
                        "op = " + op
                        + " / current = " + status[0]
                        + " / highwater = " + status[1]
                    );
//            assertTrue(status[0] != 0);
//            assertTrue(status[1] != 0);
        }
    }

    @Test
    public void connectionStatus() throws ClassNotFoundException, SQLException {
        final Database db = newDatabase();
        final int[] params
                = new int[] {
                    SQLITE_DBSTATUS_LOOKASIDE_USED,
                };
        for (final int op : params) {
            final int[] status = db.connectionStatus(op, false);
            System.out.println(
                        "op = " + op
                        + " / current = " + status[0]
                        + " / highwater = " + status[1]
                    );
//            assertTrue(status[0] != 0);
//            assertTrue(status[1] != 0);
        }
    }
}