package org.sqlite.text;

import java.io.UnsupportedEncodingException;
import java.util.Comparator;
import org.sqlite.callback.NamedCallback;
import org.sqlite.swig.SQLite3;
import static org.sqlite.swig.SQLite3Constants.SQLITE_UTF8;

/**
 * User-Defined Collating Sequence class.
 * @author calico
 * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
 * @see org.sqlite.jdbc.JdbcConnection#createCollationSequence(org.sqlite.text.Collator)
 * @see org.sqlite.jdbc.JdbcConnection#dropCollationSequence(org.sqlite.text.Collator)
 * @see org.sqlite.jdbc.JdbcConnection#dropCollationSequence(java.lang.String)
 */
public abstract class Collator extends NamedCallback implements Comparator<String> {
    
    private final int enc;
    
    /**
     * Returns the text encoding code.
     * @return the text encoding code
     */
    public int getEncoding() {
        return enc;
    }
    
    private final String encoding;
    
    /**
     * Create new collation sequence with SQLITE_UTF8.
     * @param name the name of the new collation sequence.
     */
    public Collator(String name) {
        this(name, SQLITE_UTF8);
    }
    
    /**
     * Create new collation sequence.
     * @param name the name of the new collation sequence.
     * @param enc the most desirable encoding may be one of the constants SQLITE_UTF8, SQLITE_UTF16LE, SQLITE_UTF16BE or SQLITE_UTF16_ALIGNED.
     * @see <a href="http://sqlite.org/c3ref/c_any.html">Text Encodings</a>
     */
    public Collator(String name, int enc) {
        super(name);
        this.enc = enc;
        this.encoding = (enc == SQLITE_UTF8 ? null : SQLite3.getEncoding(enc));
    }
    
    /**
     * Called from the sqlite3_step() function.
     * @param source the source bytes to be compared.
     * @param target the target bytes to be compared.
     * @return a negative integer, zero, or a positive integer as the first argument is less than, equal to, or greater than the second.
     * @see java.util.Comparator#compare(Object, Object)
     */
    protected final int xCompare(byte[] source, byte[] target) throws UnsupportedEncodingException {
        // TODO Mysaifu JVMのBug#11980が解決したらアクセス修飾子をprivateに変更すること！
        // @see http://sourceforge.jp/tracker/index.php?func=detail&aid=11980&group_id=1890&atid=7027
        return compare(new String(source, encoding), new String(target, encoding));
    }
}
