package org.sqlite;

import org.sqlite.schema.ColumnMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SWIGTYPE_p_int;
import org.sqlite.swig.SWIGTYPE_p_p_char;
import org.sqlite.swig.SWIGTYPE_p_p_sqlite3;
import org.sqlite.swig.SWIGTYPE_p_sqlite3;
import org.sqlite.text.Collator;
import org.sqlite.types.SQLite3StmtPtrPtr;
import org.sqlite.udf.Function;
import static org.sqlite.swig.SQLite3.*;

/**
 * sqlite3 wrapper class.<br/>
 * NOTE: SQLite 3.3.5 based.
 * @author calico
 */
public class Database {
    /** sqlite3** wrapper class */
    private static class SQLite3PtrPtr extends SWIGTYPE_p_p_sqlite3 {
        private boolean isDeleted = false;
        
        public SQLite3PtrPtr() {
            super(getCPtr(new_p_p_sqlite3()), false);
        }

        public boolean isDeleted() {
            return isDeleted;
        }
        
        public synchronized void delete() {
            if (!isDeleted) {
                delete_p_p_sqlite3(this);
                isDeleted = true;
            }
        }

        public SWIGTYPE_p_sqlite3 getSQLite3Ptr() {
            if (isDeleted) {
                throw new IllegalStateException("SQLite3PtrPtr is already deleted.");
            }
            return p_p_sqlite3_value(this);
        }

        @Override
        protected void finalize() throws Throwable {
            delete();
            super.finalize();
        }
    }
    
    private final SQLite3PtrPtr ppDb =new SQLite3PtrPtr();
    private Map<Long, Statement> statements;
    private List<Function> functions;
    private List<Collator> collators;
    protected final Properties info;
    private final boolean isInMemory;
    
    /** timeout(ms) : sqlite3_busy_timeout */
    private int timeout;
    
    public Database(String filename, Properties info) throws SQLException {
        this.info = info;
        this.isInMemory
                = (filename == null || getInMemoryFileName().equals(filename));
        open(filename);
    }
    
    /**
     * It always returns "SQLite".
     * @return "SQLite"
     */
    public String getProductName() {
        return "SQLite";
    }
    
    public boolean isInMemoryMode() {
        return isInMemory;
    }
    
    private SWIGTYPE_p_sqlite3 getInstance() {
        return ppDb.getSQLite3Ptr();
    }
    
    /**
     * invoke sqlite3_open() function.
     * @param filename  filename
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/open.html">Opening A New Database Connection</a>
     */
    private void open(String filename) throws SQLException {
        int ret = sqlite3_open(filename, ppDb);
        if (ret != SQLITE_OK) {
            SWIGTYPE_p_sqlite3 db = getInstance();
            SQLException ex = new JdbcSQLException(db);
            ppDb.delete();
            throw ex;
        }
    }
    
    public boolean isReadOnly() {
        return false;
    }
    
    /**
     * Retrieves whether this Database object has been closed.
     * @return true if this Database object is closed. false if it is still open.
     */
    public boolean isClosed() {
        return (ppDb.isDeleted());
    }
    
    /**
     * invoke sqlite3_close() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/close.html">Closing A Database Connection</a>
     */
    public void close() throws SQLException {
        if (!isClosed()) {
            // close all statements
            closeStatements();
            // unregister all functions
            unregisterFunctions();
            // unregister all collators
            unregisterCollators();
            
            final SWIGTYPE_p_sqlite3 db = getInstance();
            int ret = sqlite3_close(db);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
            ppDb.delete();
        }
    }
    
    /**
     * invoke sqlite3_get_autocommit() function.
     * @return  true if auto commit mode.
     * @see <a href="http://sqlite.org/c3ref/get_autocommit.html">Test To See If The Database Is In Auto-Commit Mode</a>
     */
    public boolean getAutoCommit() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return (sqlite3_get_autocommit(db) != 0);
    }
    
    /**
     * invoke sqlite3_busy_timeout() function.
     * @param ms    milliseconds
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/busy_timeout.html">Set A Busy Timeout</a>
     */
    public int setBusyTimeout(int ms) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = sqlite3_busy_timeout(db, ms);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        timeout = (ms < 1 ? 0 : ms);
        return ret;
    }

    /**
     * Retrieves the timeout(ms) value.
     * @return  timeout(ms) value.
     */
    public int getBusyTimeout() {
        return timeout;
    }
    
    /**
     * invoke sqlite3_exec() function.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/exec.html">One-Step Query Execution Interface</a>
     */
    public int execute(String sql) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = 0;
        if (timeout == 0) {
            // no limit
            while ((ret = sqlite3_exec(db, sql)) == SQLITE_BUSY) {
                // waiting...
            }
        } else {
            ret = sqlite3_exec(db, sql);
            if (ret == SQLITE_BUSY) {
                // timeout
                throw new JdbcSQLException("Timeout expired.", SQLITE_BUSY);
            }
        }
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return ret;
    }

    /**
     * execute PRAGMA commands by sqlite3_exec() finction.
     * @param commands  command list
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/pragma.html">Pragma statements supported by SQLite</a>
     */
    public void pragma(String[] commands) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        for (final String cmd : commands) {
            int ret = sqlite3_exec(db, "PRAGMA " + cmd);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
        }
    }
    
    /**
     * begin transaction.
     * @param type  transaction type.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void beginTransaction(TransactionType type) throws SQLException {
        closeStatements();
        if (type == null) {
            execute("BEGIN");
        } else {
            execute("BEGIN " + type);
        }
    }
    
    /**
     * commit toransaction.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void commitTransaction() throws SQLException {
        closeStatements();
        execute("COMMIT");
    }
    
    /**
     * rollback transaction.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void rollbackTransaction() throws SQLException {
        closeStatements();
        execute("ROLLBACK");
    }
    
    /**
     * create MANAGED Statement instance.
     * @param sql
     * @param ppStmt
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }
        if (ppStmt == null) {
            throw new NullPointerException("ppStmt is null.");
        }
        
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt.getSQLite3StmtPtr());
    }
    
    /**
     * create UNMANAGED Statement instance.
     * @param sql
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }

        final SWIGTYPE_p_sqlite3 db = getInstance();
        final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            ppStmt.delete();
            throw new JdbcSQLException(db);
        }
        return new Statement(this, ppStmt);
    }
    
    /**
     * invoke sqlite3_interrupt() function.
     * @see <a href="http://sqlite.org/c3ref/interrupt.html">Interrupt A Long-Running Query</a>
     */
    public void interrupt() throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        sqlite3_interrupt(db);
        closeStatements();
    }
    
    /**
     * invoke sqlite3_changes() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/changes.html">Count The Number Of Rows Modified</a>
     */
    public int changes() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return sqlite3_changes(db);
    }
    
    /**
     * invoke sqlite3_total_changes() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/total_changes.html">Total Number Of Rows Modified</a>
     */
    public int totalChanges() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return sqlite3_total_changes(db);
    }
    
    /**
     * invoke sqlite3_last_insert_rowid() function.
     * @return
     * @see <a href="http://sqlite.org/c3ref/last_insert_rowid.html">Last Insert Rowid</a>
     */
    public long lastInsertRowId() {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        return sqlite3_last_insert_rowid(db);
    }
    
    /**
     * invoke sqlite3_table_column_metadata() function.
     * @param dbName
     * @param tableName
     * @param columnName
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://www.sqlite.org/c3ref/table_column_metadata.html">Extract Metadata About A Column Of A Table</a>
     */
    public ColumnMetaData getColumnMetaData(String dbName, String tableName, String columnName) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        SWIGTYPE_p_p_char dataType = null;
        SWIGTYPE_p_p_char collSeq = null;
        SWIGTYPE_p_int notNull = null;
        SWIGTYPE_p_int primaryKey = null;
        SWIGTYPE_p_int autoInc = null;
        try {
            dataType = new_p_p_char();
            collSeq = new_p_p_char();
            notNull = new_p_int();
            primaryKey = new_p_int();
            autoInc = new_p_int();
        
            int ret = sqlite3_table_column_metadata(
                            db, dbName, tableName, columnName,
                            dataType, collSeq, notNull, primaryKey, autoInc
                        );
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(db);
            }
            
            return new ColumnMetaData(
                            p_p_char_value(dataType),
                            p_p_char_value(collSeq),
                            p_int_value(notNull),
                            p_int_value(primaryKey),
                            p_int_value(autoInc)
                        );
        } finally {
            if (dataType != null) {
                delete_p_p_char(dataType);
            }
            if (collSeq != null) {
                delete_p_p_char(collSeq);
            }
            if (notNull != null) {
                delete_p_int(notNull);
            }
            if (primaryKey != null) {
                delete_p_int(primaryKey);
            }
            if (autoInc != null) {
                delete_p_int(autoInc);
            }
        }
    }
    
    void addStatement(Statement stmt) throws SQLException {
        if (statements == null) {
            statements = new HashMap<Long, Statement>();
        }
        final long key = stmt.getHandle();
        if (statements.containsKey(key)) {
            throw new SQLException("Duplicate sqlite3_stmt handle error.", "90J31");
        }
        statements.put(key, stmt);
    }
    
    void removeStatement(Statement stmt) throws SQLException {
        if (statements != null) {
            final long key = stmt.getHandle();
            if (statements.remove(key) == null) {
                throw new SQLException("Unmanaged sqlite3_stmt handle error.", "90J32");
            }
        }
    }
    
    private void closeStatements() {
        if (statements != null) {
            final Collection<Statement> list = statements.values();
            statements = null;
            for (final Statement stmt : list) {
                try {
                    stmt.close();
                } catch (SQLException ex) {
                    Logger.getLogger(Database.class.getName()).info(ex.toString());
                }
            }
        }
    }
    
    private void addFunction(Function func) {
        if (functions == null) {
            functions = new ArrayList<Function>();
        }
        if (!functions.contains(func)) {
            functions.add(func);
        }
    }
    
    private void removeFunction(Function func) {
        if (functions != null) {
            functions.remove(func);
        }
    }
    
    private void unregisterFunctions() {
        if (functions != null) {
            final List<Function> list = functions;
            functions = null;
            for (final Function func : list) {
                try {
                    unregisterFunction(func);
                } catch (SQLException ex) {
                    Logger.getLogger(Database.class.getName()).info(ex.toString());
                }
            }
        }
    }
    
    /**
     * invoke 'sqlite3_create_function()' function and the User-Defined function is registered.
     * @param func User-Defined function
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
     */
    public void registerFunction(Function func) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = register_function(db, func);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        addFunction(func);
    }
    
    /**
     * invoke 'sqlite3_create_function()' function and the User-Defined function is unregistered.
     * @param func User-Defined function
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
     */
    public void unregisterFunction(Function func) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = unregister_function(db, func);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        removeFunction(func);
    }
    
    private void addCollator(Collator col) {
        if (collators == null) {
            collators = new ArrayList<Collator>();
        }
        if (!collators.contains(col)) {
            collators.add(col);
        }
    }
    
    private void removeCollator(Collator col) {
        if (collators != null) {
            collators.remove(col);
        }
    }
    
    private void unregisterCollators() {
        if (collators != null) {
            final List<Collator> list = collators;
            collators = null;
            for (final Collator col : list) {
                try {
                    unregisterCollation(col);
                } catch (SQLException ex) {
                    Logger.getLogger(Database.class.getName()).info(ex.toString());
                }
            }
        }
    }
    
    /**
     * invoke 'sqlite3_create_collation()' function and the User-Defined Collating Sequences is registered.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
     */
    public void registerCollation(Collator col) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = register_collation(db, col);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        addCollator(col);
    }
    
    /**
     * invoke 'sqlite3_create_function()' function and the User-Defined Collating Sequences is unregistered.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
     */
    public void unregisterCollation(Collator col) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getInstance();
        int ret = unregister_collation(db, col);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(db);
        }
        removeCollator(col);
    }

    // TODO Statement毎にtimeoutが設定できるようにする！
    
    @Override
    protected void finalize() throws Throwable {
        if (!isClosed()) {
            // sqlite3_openを呼び出したスレッドと異なるスレッドから
            // sqlite3_closeを呼び出すことは出来ない（呼び出すとJVMが異常終了する）
            Logger.getLogger("global").severe("Database connection has leaked!");
        }
        super.finalize();
    }
    
}