﻿using System.Windows.Forms;
using System.IO;
using System.Media;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Reflection;

namespace SortFiles
{
    public partial class MainForm : Form
    {
        /// <summary>
        /// このアプリケーションからドラッグを始めたことを表す。真の間はドロップを受け付けない
        /// </summary>
        private bool dragedFromMe = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MainForm() {
            InitializeComponent();
        }

        /// <summary>
        /// “名前”カラムをクリックしたとき
        /// </summary>
        private void listView_ColumnClick(object sender, ColumnClickEventArgs e) {
            if (e.Column == 0) {
                // ソート方法の昇順と降順を切り替える
                listView.Sorting = SortOrder.None;
                if (listView.Sorting == SortOrder.Ascending)
                    listView.Sorting = SortOrder.Descending;
                else
                    listView.Sorting = SortOrder.Ascending;
                listView.Sort();
            }
        }

        /// <summary>
        /// ドラッグしたアイテムがListView上にきたとき
        /// </summary>
        private void listView_DragOver(object sender, DragEventArgs e) {
            // ファイルのみ受け付ける
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
                e.Effect = DragDropEffects.Copy;
            else
                e.Effect = DragDropEffects.None;
        }

        /// <summary>
        /// ListViewにアイテムをドロップされたとき
        /// </summary>
        private void listView_DragDrop(object sender, DragEventArgs e) {
            try {
                // ドラッグ元がListViewならドロップさせない
                if (dragedFromMe) return;

                if (e.Data.GetDataPresent(DataFormats.FileDrop)) {
                    // ListViewにドロップされたファイルを追加する
                    foreach (string file in (string[])e.Data.GetData(DataFormats.FileDrop)) {
                        var item = new ListViewItem();
                        item.Text = Path.GetFileName(file);
                        item.SubItems.Add(Path.GetDirectoryName(file));
                        item.Tag = file;
                        listView.Items.Add(item);
                    }
                }
            }
            catch { }
        }

        /// <summary>
        /// ListViewからアイテムをアプリ外にドラッグする
        /// </summary>
        private void listView_ItemDrag(object sender, ItemDragEventArgs e) {
            try {
                string[] files = GetSelectedFiles();
                dragedFromMe = true;
                var dataObj = new DataObject(DataFormats.FileDrop, files);
                ((Control)sender).DoDragDrop(dataObj, DragDropEffects.Copy);
                dragedFromMe = false;
            }
            catch { }
        }

        /// <summary>
        /// ListView上でキーが押されたとき
        /// </summary>
        private void listView_KeyDown(object sender, KeyEventArgs e) {

            if (e.KeyCode == Keys.Delete) {
                // 選択されたアイテムをListViewから削除する
                if (listView.Items.Count > 0) {
                    try {
                        listView.FocusedItem.Remove();
                    }
                    catch {
                        SystemSounds.Beep.Play();
                    }
                }
                else {
                    SystemSounds.Beep.Play();
                }
            }

            // CtrlもShiftもAltも押されていないとき
            if (!e.Control && !e.Shift && !e.Alt) {
                if (e.KeyCode == Keys.F1) {
                    // バージョン情報を表示
                    var v = GetMyVersion();
                    MessageBox.Show(this, "Sort Files ver." + v + "\n© 2011 yuuAn. All rights reserved.", "バージョン情報", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
            }

            // Ctrlだけが押されているとき
            if (e.Control && !e.Shift && !e.Alt) {
                switch (e.KeyCode) {
                    case Keys.A:
                        // すべて選択
                        foreach (ListViewItem item in listView.Items) {
                            item.Selected = true;
                        }
                        break;

                    case Keys.C:
                        try {
                            // 選択されているファイルをクリップボードにコピー
                            var files = new StringCollection();
                            files.AddRange(GetSelectedFiles());
                            Clipboard.SetFileDropList(files);
                        }
                        catch {
                            Clipboard.Clear();
                        }
                        break;

                    case Keys.X:
                        // 切り取りはできない仕様
                        SystemSounds.Beep.Play();
                        break;

                    case Keys.W:
                        this.Close();
                        break;
                }
            }

        }

        /// <summary>
        /// ListViewで選択されているファイルのパスを取得する
        /// </summary>
        /// <returns>ファイルパスの配列</returns>
        private string[] GetSelectedFiles() {
            var files = new string[listView.SelectedItems.Count];
            for (int i = 0; i < listView.SelectedItems.Count; i++) {
                files[i] = (string)listView.SelectedItems[i].Tag;
            }
            return files;
        }

        /// <summary>
        /// このアプリケーションのバージョンナンバーを取得する
        /// </summary>
        /// <returns>バージョンナンバー</returns>
        private string GetMyVersion() {
            var ver = FileVersionInfo.GetVersionInfo(Assembly.GetExecutingAssembly().Location);
            return ver.FileMajorPart + "." + ver.FileMinorPart + ((ver.FileMajorPart < 1) ? " β" : "");
        }
    }
}
