/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: soopyg.h,v 1.41 2004/05/15 12:15:03 randy Exp $
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import "../../soopy.h"

enum Align {
  alNone,
  alTop,
  alBottom,
  alLeft,
  alRight,
  alClient,
};

class SpCaptionException : public SpException {
public:
    SpCaptionException(const char* s)
        : SpException(s) {}
};

/***********************************/
/*          Base Component         */
/***********************************/

/*
 * class SpComponent
 */

class SpComponent : public SpObject {
protected:
    SpValue ns;       // NameSpace
    SpValue parentNameSpace;   // NameSpace
    NSObject* widget;
    NSObject* parent;
    Align align;

    virtual void resetAlign(NSRect& parent_rect);
    static void resetAlignAllComponents(NSView* parent);
    void resetAlignParentComponents();
    void setAlign(Align a);

    virtual SpValue& getCaption(NSObject* w) = 0;
    virtual void setCaption(NSObject* w, const char* str) = 0;

    static SpValue& primGetCaption();
    static SpValue& primSetCaption(SpValue& v);
    static SpValue& primGetParent();
    static SpValue& primGetTop();
    static SpValue& primSetTop(SpValue& v);
    static SpValue& primGetLeft();
    static SpValue& primSetLeft(SpValue& v);
    static SpValue& primGetWidth();
    static SpValue& primSetWidth(SpValue& v);
    static SpValue& primGetHeight();
    static SpValue& primSetHeight(SpValue& v);
    static SpValue& primGetAlign();
    static SpValue& primSetAlign(SpValue& v);
public:
    SpComponent(SpValue& n, NSObject* w=NULL) : ns(n), widget(w) { align = alNone; }
    virtual ~SpComponent(){
        if(widget != NULL){
            [widget release];
        }
    }
    SpValue& toString();
    const char* typeString(){ return "component"; }
    bool isComponent(){ return true; }

    virtual void creation(SpValue& parentNS, NSObject* parent) = 0;

    void after_creation();

    static void init();
    virtual void initProperties(SpNameSpace* ns);

    friend class SpEdit;
    friend class SpField;
    friend class SpMemo;
    friend class SpCheckBox;
    friend class SpSaveDialog;
    friend class SpOpenDialog;
    friend class SpComboBox;
    friend class SpListBox;
    friend class SpRadioGroup;
    friend class SpContainer;
    friend class SpTableView;
};

/***********************************/
/*          Container Base        */
/***********************************/

class SpContainer : public SpComponent {
protected:
    void create_components(NSObject* parent_control,
                           SpValue& parent_NS,
                           SpValue& vcomps);

public:
    SpContainer(SpValue& n) : SpComponent(n) {}

    void resetAlignAllComponents();
};



/***********************************/
/*           Button               */
/***********************************/

/*
 * class SpNSButton
 */
@interface SpNSButton : NSButton
{
  SpNameSpace* ns;
  SpComponent* spcomp;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(void)click:(NSObject*)sender;

@end

/*
 * class SpButton
 */
class SpButton: public SpComponent {
 protected:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);
 public:
    SpButton(SpValue& n) : SpComponent(n) {}
    const char* typeString(){ return "button"; }
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();

    // primitives
    static SpValue& create_and_run();
    static SpValue& make();
};

/*
 * class SpCheckBox
 */
class SpCheckBox : public SpButton {
 private:
    static SpValue& primGetCheck();
    static SpValue& primSetCheck(SpValue& v);
 public:
    SpCheckBox(SpValue& n) : SpButton(n) {}
    const char* typeString(){ return "checkbox"; }
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();
    void initProperties(SpNameSpace* ns);

    // primitives
    static SpValue& create_and_run();
    static SpValue& make();
};

/***********************************/
/*          RadioGroup             */
/***********************************/

/*
 * class SpNSRadioGroup
 */

@interface SpNSRadioGroup : NSBox
{
  SpNameSpace* ns;
  SpComponent* spcomp;
  NSMatrix* matrix;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(void)dealloc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(void)click:(NSObject*)sender;
-(void)setMatrix:(NSMatrix*)mat;
-(void)setIndex:(int)i;
-(int)getIndex;
-(NSString*)getText;

@end

/*
 * class SpRadioGroup
 */

class SpRadioGroup : public SpContainer {
private:
    void resetAlign(NSRect& parent_rect);

    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);

    static SpValue& primGetIndex();
    static SpValue& primSetIndex(SpValue& i);
    static SpValue& primGetText();

    void initProperties(SpNameSpace* pNS);
public:
    SpRadioGroup(SpValue& n) : SpContainer(n) {}
    const char* typeString(){ return "radiogroup"; }

    static void init();
    void creation(SpValue& parentNS, NSObject* parent);

    // primitives
    static SpValue& make();
};



/***********************************/
/*      Base of Label & field     */
/***********************************/
@interface SpNSTextField : NSTextField
{
  SpNameSpace* ns;
  SpComponent* spcomp;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(void)mouseUp:(NSEvent*)event;
-(void)change:(NSObject*)sender;

@end

/*
 * class SpTextField
 */
class SpTextField: public SpComponent {
 private:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);
 public:
    SpTextField(SpValue& n) : SpComponent(n) {}
};



/***********************************/
/*            Label                */
/***********************************/

/*
 * class SpNSLabel
 */
@interface SpNSLabel : SpNSTextField
-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
@end

/*
 * class SpLabel
 */
class SpLabel: public SpTextField {
 public:
    SpLabel(SpValue& n) : SpTextField(n) {}
    const char* typeString(){ return "label"; }
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();

    // primitives
    static SpValue& create_and_run();
    static SpValue& make();
};

/***********************************/
/*            Field                */
/***********************************/

/*
 * class SpNSField
 */
@interface SpNSField : SpNSTextField
-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
@end

/*
 * class SpField
 */
class SpField : public SpTextField {
 private:
    static SpValue& primGetText();
    static SpValue& primSetText(SpValue& v);
 public:
    SpField(SpValue& n) : SpTextField(n) {}
    const char* typeString(){ return "field"; }
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();
    void initProperties(SpNameSpace* pNS);

    // primitives
    static SpValue& create_and_run();
    static SpValue& make();
};

/***********************************/
/*             Edit                */
/***********************************/

/*
 * class SpNSEdit
 */
@interface SpNSTextView : NSTextView
{
  bool isModified;
}

-(void)didChangeText;
-(bool)modified;
-(void)setModified:(bool)p;

@end

@interface SpNSEdit : NSScrollView
{
  SpNameSpace* ns;
  SpComponent* spcomp;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;

@end

/*
 * class SpEdit
 */
class SpEdit : public SpComponent {
 private:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);

    static SpValue& primGetText();
    static SpValue& primSetText(SpValue& v);
    static SpValue& primGetModified();
    static SpValue& primSetModified(SpValue& v);
    static SpValue& primAppend(SpValue& v);
    static SpValue& primClear();
 public:
    SpEdit(SpValue& n) : SpComponent(n) {}
    const char* typeString(){ return "edit"; }
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();
    void initProperties(SpNameSpace* ns);

    // primitives
    static SpValue& create_and_run();
    static SpValue& make();
};

/***********************************/
/*            ComboBox             */
/***********************************/

/*
 * class SpNSComboBox
 */

@interface SpNSComboBox : NSComboBox
{
  SpNameSpace* ns;
  SpComponent* spcomp;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(void)change:(NSObject*)sender;

@end

/*
 * class SpComboBox
 */

class SpComboBox : public SpComponent {
private:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);

    static SpValue& primGetText();
    static SpValue& primSetText(SpValue& v);
public:
    SpComboBox(SpValue& n) : SpComponent(n) {}
    const char* typeString(){ return "combobox"; }

    static void init();
    void creation(SpValue& parentNS, NSObject* parent);

    void initProperties(SpNameSpace* pNS);

    // primitives
    static SpValue& make();
};

/***********************************/
/*            ListBox              */
/***********************************/

/*
 * class SpNSListBox
 */
@interface SpNSListBoxDataSource : NSObject
{
  NSMutableArray* ary;
}

-(id)initWithCapacity:(unsigned)numItems;
-(void)dealloc;

-(void)addObject:(id)aObject;
-(void)replaceObjectAtIndex:index withObject:anObject;
-(int)numberOfRowsInTableView:(NSTableView*)aTableView;
-(id)tableView:(NSTableView*)aTableView objectValueForTableColumn:(NSTableColumn*)column row:(int)rowIndex;
-(void)tableView:(NSTableView*)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn*)column row:(int)rowIndex;

-(int)count;
-(id)objectAtIndex:(int)index;

@end

@interface NSTableViewForListBox : NSTableView

-(int)count;
-(id)objectAtIndex:(int)index;

@end


@interface SpNSListBox : NSScrollView
{
  SpNameSpace* ns;
  SpComponent* spcomp;
  NSTableViewForListBox* tableView;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(void)change:(NSObject*)sender;

-(int)count;
-(int)selectedRow;
-(id)objectAtIndex:(int)index;

@end

/*
 * class SpListBox
 */

class SpListBox : public SpComponent {
private:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);

    static SpValue& primGetText();
public:
    SpListBox(SpValue& n) : SpComponent(n) {}
    const char* typeString(){ return "listbox"; }

    static void init();
    void creation(SpValue& parentNS, NSObject* parent);

    // primitives
    static SpValue& make();

    void initProperties(SpNameSpace* pNS);
};



/***********************************/
/*           TableView             */
/***********************************/

@interface SpNSTableViewDataSource : SpNSListBoxDataSource
{
  unsigned int rows;
  unsigned int cols;
  NSView* view;
}

-(id)initWithRows:(unsigned int)r Columns:(unsigned int)c View:(NSView*)v;
-(int)numberOfRowsInTableView:(NSTableView*)aTableView;
-(id)tableView:(NSTableView*)aTableView objectValueForTableColumn:(NSTableColumn*)column row:(int)rowIndex;
-(void)tableView:(NSTableView*)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn*)column row:(int)rowIndex;
-(id)objectAtRow:(int)row atColumn:(int)col;
-(void)setObject:(id)anObject atRow:(int)row atColumn:(int)col;

@end

@interface NSTableViewForTableView : NSTableViewForListBox

-(id)objectAtRow:(int)row atColumn:(int)col;
-(void)setObject:(id)anObject atRow:(int)row atColumn:(int)col;

@end

@interface SpNSTableView : NSScrollView
{
  SpNameSpace* ns;
  SpComponent* spcomp;
  NSTableViewForTableView* tableView;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(void)change:(NSObject*)sender;

-(int)count;
-(int)selectedColumn;
-(void)selectColumn:(int)col byExtendingSelection:(BOOL)flag;
-(int)selectedRow;
-(void)selectRow:(int)col byExtendingSelection:(BOOL)flag;
-(id)objectAtIndex:(int)index;
-(id)objectAtRow:(int)row atColumn:(int)col;
-(void)setObject:(id)anObject atRow:(int)row atColumn:(int)col;

@end

/*
 * class SpTableView
 */

class SpTableView : public SpComponent {
private:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);
    static SpValue& primGetRow();
    static SpValue& primSetRow(SpValue& value);
    static SpValue& primGetCol();
    static SpValue& primSetCol(SpValue& value);
    static SpValue& primGetTable();

public:
    SpTableView(SpValue& n) : SpComponent(n) {}
    const char* typeString(){ return "tableview"; }

    static void init();
    void creation(SpValue& parentNS, NSObject* parent);

    // primitives
    static SpValue& make();

    void initProperties(SpNameSpace* pNS);

    friend class TSpTableView;
};

class TableOfTableView : public SpArray {
  private:
    SpNSTableView* view;

  public:
    TableOfTableView(SpNSTableView* v){
        view = v;
        depth = 2;
    }

    SpValue& toString();
    const char* typeString(){ return "table"; }
    SpValue& getFeatureType(){ return NilObject; }
    SpValue& operator[](SpValue& tuple);
    void set(SpValue& tuple, SpValue& value);
    SpValue& onMessage(SpValue& rec, SpValue& msg);
};


/***********************************/
/*            Panel                */
/***********************************/

/*
 * class SpNSPanel
 */
@interface SpNSPanel : NSView
{
  SpNameSpace* ns;
  SpComponent* spcomp;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)p comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(id)contentView;
-(void)drawRect:(NSRect)rect;
-(void)mouseUp:(NSEvent*)event;

@end

/*
 * class SpPanel
 */

class SpPanel : public SpContainer {
private:
    void resetAlign(NSRect& parent_rect);

    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);
public:
    SpPanel(SpValue& n) : SpContainer(n) {}
    const char* typeString(){ return "panel"; }
    void run();
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();

    // primitives
    static SpValue& make();
};


/***********************************/
/*           Window                */
/***********************************/

/*
 * class WindowCtrl
 */

@interface WindowCtrl : NSObject
{
  NSWindow* window;
}

-(id)window:(NSWindow*)win;

-(void)dealloc;
-(id)mainMenu;

/* Local Method */
-(void)showTitle:(NSString*)title message:(NSString*)message;

/* Delegate Messages */
-(void)applicationDidFinishLaunching:(NSNotification*)aNotification;
-(void)applicationWillTerminate:(NSNotification*)aNotification;


-(BOOL)windowShouldClose:(id)sender;
-(void)windowWillClose:(NSNotification*)aNotification;

@end

/*
 * class SpNSWindow
 */

@interface SpNSWindow : NSWindow
{
  SpNameSpace* ns;
  SpComponent* spcomp;
}

-(id)initWithNameSpace:(SpNameSpace*)assoc rect:(NSRect)r comp:(SpComponent*)spc;
-(SpNameSpace*)getNameSpace;
-(SpComponent*)getSpComponent;
-(void)setFrame:(NSRect)rect;

@end

/*
 * class SpWindow
 */

class SpWindow : public SpContainer {
private:
    static bool app_running;
    static SpNSWindow* main_window;

    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);
public:
    SpWindow(SpValue& n) : SpContainer(n) {}
    const char* typeString(){ return "window"; }
    void run();
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();

    // primitives
    static SpValue& create_and_run();
    static SpValue& make();
};

/***********************************/
/*           SaveDialog            */
/***********************************/

/*
 * class SpSaveDialog
 */

class SpSaveDialog : public SpContainer {
private:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);

    static SpValue& primGetFilename();
//    static SpValue& primGetFilenames();
public:
    SpSaveDialog(SpValue& n) : SpContainer(n) {}
    const char* typeString(){ return "save_dialog"; }
    void run();
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();
    // primitives
    static SpValue& make();

    void initProperties(SpNameSpace* pNS);
};

/***********************************/
/*           OpenDialog            */
/***********************************/

/*
 * class SpOpenDialog
 */

class SpOpenDialog : public SpContainer {
private:
    SpValue& getCaption(NSObject* w);
    void setCaption(NSObject* w, const char* str);

    static SpValue& primGetFilename();
//    static SpValue& primGetFilenames();
public:
    SpOpenDialog(SpValue& n) : SpContainer(n) {}
    const char* typeString(){ return "open_dialog"; }
    void run();
    void creation(SpValue& parentNS, NSObject* parent);

    static void init();
    // primitives
    static SpValue& make();

    void initProperties(SpNameSpace* pNS);
};

/***********************************/
/*             OkDialog            */
/***********************************/

/*
 * class SpOkDialog
 */

class SpOkDialog : public SpObject {
public:
    static void init();
    static SpValue& make();
};

/***********************************/
/*          OkCancelDialog         */
/***********************************/

/*
 * class SpOkCancelDialog
 */

class SpOkCancelDialog : public SpObject {
public:
    static void init();
    static SpValue& make();
};

/***********************************/
/*            YesNoDialog          */
/***********************************/

/*
 * class SpYesNoDialog
 */

class SpYesNoDialog : public SpObject {
public:
    static void init();
    static SpValue& make();
};



extern SpNameSpace* PComponentNameSpace;
extern SpValue ComponentNS;
extern SpNameSpace* PLabelNameSpace;
extern SpValue LabelNS;
extern SpNameSpace* PWindowNameSpace;
extern SpValue WindowNS;
extern SpNameSpace* PPanelNameSpace;
extern SpValue PanelNS;
extern SpNameSpace* PButtonNameSpace;
extern SpValue ButtonNS;
extern SpNameSpace* PFieldNameSpace;
extern SpValue FieldNS;
extern SpNameSpace* PEditNameSpace;
extern SpValue EditNS;
extern SpNameSpace* PCheckBoxNameSpace;
extern SpValue CheckBoxNS;
extern SpNameSpace* PSaveDialogNameSpace;
extern SpValue SaveDialogNS;
extern SpNameSpace* POpenDialogNameSpace;
extern SpValue OpenDialogNS;
extern SpNameSpace* PComboBoxNameSpace;
extern SpValue ComboBoxNS;
extern SpNameSpace* PListBoxNameSpace;
extern SpValue ListBoxNS;
extern SpNameSpace* PRadioGroupNameSpace;
extern SpValue RadioGroupNS;
extern SpNameSpace* PRadioButtonNameSpace;
extern SpValue RadioButtonNS;
extern SpNameSpace* PTableViewNameSpace;
extern SpValue TableViewNS;

extern SpValue DefaultTop;
extern SpValue DefaultLeft;
extern SpValue DefaultWidth;
extern SpValue DefaultHeight;

extern SpValue SymTop;
extern SpValue SymLeft;
extern SpValue SymWidth;
extern SpValue SymHeight;
extern SpValue SymComponents;
extern SpValue SymTitle;
extern SpValue SymCaption;
extern SpValue SymMake;
extern SpValue SymOnClick;
extern SpValue SymOnDblClick;
extern SpValue SymOnChange;
extern SpValue SymParent;
extern SpValue SymAlign;
extern SpValue SymText;
extern SpValue SymAppend;
extern SpValue SymClear;
extern SpValue SymCheckedP;
extern SpValue SymCanceledP;
extern SpValue SymFilename;
extern SpValue SymFilenames;
extern SpValue SymItems;
extern SpValue SymModifiedP;
extern SpValue SymMessage;
extern SpValue SymIndex;
extern SpValue SymTable;
extern SpValue SymRow;
extern SpValue SymCol;
extern SpValue SymRows;
extern SpValue SymCols;
extern SpValue SymHeader;
extern SpValue SymOnCreate;

