/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 *
 * Copyright (C) 2002 SUZUKI Jun
 *
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 *
 *
 * $Id: String.cpp,v 1.65 2004/05/18 12:47:17 randy Exp $
 */

//
// Soopy String Class
//

#include <limits.h>
#include "soopy.h"

/*
void SpString::append(const char* s)
{
//cout << "spstr.app('" << s << "')" << endl;
    istringstream ist(s);
    StreamReader reader(&ist);
    while(true){
        SpChar ch = reader.ReadChar();
        if(ch == (SpChar)NULL){ break; }
        str.push_back(ch);
    }
}
 */

//void SpString::appendWithEncoder(const char* s, ReadEncoder& encoder)
void SpString::append(const char* s, ReadEncoder& encoder)
{
  //cout << "spstr.app('" << s << "')" << endl;
  istringstream ist(s);
  StreamReader sr(&ist);
  Reader* old = encoder.getReader();
  encoder.setReader(&sr);
  while(!encoder.eof()){
    SpChar ch = encoder.ReadChar();
    if(ch == (SpChar)NULL){ break; }
    str.push_back(ch);
  }
  encoder.setReader(old);
}

SpString::~SpString()
{
#ifdef TEST
  *spout << ":destroy SpString '" << toCStringWithEncoder() << "'" << "\n";
#endif
}

const int Buf_Size = 1024 * 4;

/*
const char* SpString::toCString()
{
    ostringstream ost;
    StreamWriter writer(&ost);
    SpCharVector::iterator it;
    for(it = str.begin(); it < str.end(); it++){
        writer.WriteChar(*it);
    }

    static string str;
    str = ost.str();
    return str.c_str();
}
 */

const char* SpString::toCStringWithEncoder(WriteEncoder& encoder)
{
  ostringstream ost;
  StreamWriter sw(&ost);
  Writer* old = encoder.getWriter();
  encoder.setWriter(&sw);
  SpCharVector::iterator it;
  for(it = str.begin(); it < str.end(); it++){
    encoder.WriteChar(*it);
  }
  encoder.setWriter(old);

  static string str;
  str = ost.str();
  return str.c_str();
}

SpValue& SpString::toString()
{
  SpString* str = new SpString("\"");
  *str += toCStringWithEncoder();
  *str += "\"";
  //    static SpValue val;
  //    val.setNewObject(str);
  //    return val;
  return SpObjectResult(str);
}

bool SpString::operator==(SpObject& obj)
{
  //cout << "str==  " << *this << " <-> " << obj << endl;
  if(SpString* s = dynamic_cast<SpString*>(&obj)){
    return str == s->str;
  }
  return false;
}

bool SpString::operator<(SpObject& obj)
{
  //cout << "str==  " << *this << " <-> " << obj << endl;
  if(typeid(*this) == typeid(obj)){
    if(SpString* s = dynamic_cast<SpString*>(&obj)){
      return str < s->str;
    }
  }
  return SpObject::operator<(obj);
}

SpString& SpString::operator+=(SpString& s)
{
  SpCharVector::iterator it;
  for(it = s.str.begin(); it != s.str.end(); it++){
    str.push_back(*it);
  }
  return *this;
}

SpValue& SpString::plus(SpValue& e1, SpValue& e2)
{
  SpString* s1;
  SpString* s2;
  SpString* s3;

  if(!e2.isString()){
    throw SpException("type mismatch in +(String, ...)");
  }
  s1 = dynamic_cast<SpString*>(e1.getObject());
  s2 = dynamic_cast<SpString*>(e2.getObject());
  s3 = new SpString(*s1);
  *s3 += *s2;
  //    static SpValue v;
  //    v.setNewObject(s3);
  //    return v;
  return SpObjectResult(s3);
}

bool SpString::match(SpValue&, SpValue& val, SpNameSpace*)
{
  if(val.isObject()){
    return operator==(*(val.getObject()));
  }
  return false;
}

SpValue& SpString::eq(SpValue&, SpValue& e2)
{
  if(!e2.isString()){
    //throw SpException("type mismatch in ==(String, ...)");
    return FalseObject;
  }
  SpString* s = e2.asString();
  if(str == s->str){
    return TrueObject;
  }
  return FalseObject;
}

SpValue& SpString::gt(SpValue&, SpValue& e2)
{
  if(!e2.isString()){
    throw SpException("type mismatch in >(String, ...)");
  }
  SpString* s = e2.asString();
  if(str > s->str){
    return TrueObject;
  }
  return FalseObject;
}

SpValue& SpString::lt(SpValue&, SpValue& e2)
{
  if(!e2.isString()){
    throw SpException("type mismatch in <(String, ...)");
  }
  SpString* s = e2.asString();
  if(str < s->str){
    return TrueObject;
  }
  return FalseObject;
}

// pXfBNgƃt@CɕB
SpTuple* SpString::split_path_file()
{
  SpCharVector::iterator it   = begin();
  SpCharVector::iterator last = end();

  for(; it < end(); it++){
#ifdef __WIN32__
    if((*it == JIS('\\')) ||
       (*it == JIS('/'))){
      last = it;
    }
#else
    if(*it == JIS('/')){
      last = it;
    }
#endif
  }
  SpTuple* tuple;
  if(last == end()){
    SpValue temp;
    tuple = new SpTuple(NullString);
    temp.setNewObject(this);
    tuple->append(temp);
  }else{
    SpString* pstr = new SpString; // path
    SpString* fstr = new SpString; // filename
    for(it = begin(); it < last; it++){
      pstr->addSpChar(*it);
    }
    for(it++; it < end(); it++){
      fstr->addSpChar(*it);
    }

    SpValue temp;
    temp.setNewObject(pstr);
    tuple = new SpTuple(temp);
    temp.setNewObject(fstr);
    tuple->append(temp);
  }
  return tuple;
}

// match_glob
static bool string_match_glob(SpCharVector::iterator sbegin,
                              SpCharVector::iterator send,
                              SpCharVector::iterator gbegin,
                              SpCharVector::iterator gend)
{
  SpCharVector::iterator it  = sbegin;
  SpCharVector::iterator git = gbegin;

  for(git; git < gend; git++){
    if(it >= send){
      return false;
    }
    switch(*git){
    case JIS('*'):
      git++;
      if(git >= gend){
        return true;
      }
      if(*git == JIS('*')){
        throw SpException("illegal glob '**'");
      }

    research:
      while(*it != *git){
        if(it >= send){
          return false;
        }
        if(*git == JIS('?')){
          break;
        }
        it++;
      }
      if(string_match_glob(it, send, git, gend)){
        return true;
      }else{
        it++;
        goto research;
      }

      break;
    case JIS('?'):
      it++;
      break;
    case JIS('\\'):
      git++;
      if(git == gend){
        throw SpException("illegal end (glob)");
      }
      if(*git != *it){
        return false;
      }
      it++;
      break;
    default:
      if(*git != *it){
        return false;
      }
      it++;
    }
  }

  if(it == send){
    return true;
  }
  return false;
}

bool SpString::match_glob(SpString* glob)
{
  return string_match_glob(begin(), end(), glob->begin(), glob->end());
}

// split
SpValue& SpString::split(SpChar c)
{
  if(length() == 0){
    return NilObject;
  }

  SpCons* cons = NULL;
  SpString s;
  SpCharVector::iterator it = begin();;
  while(it < end()){
    s.clear();
    for(; it < end(); it++){
      if(*it == c){
        it++;
        break;
      }
      s.addSpChar(*it);
    }
    SpString* s2;
    s2 = new SpString(s);
    SpValue temp;
    temp.setNewObject(s2);
    if(cons == NULL){
      cons = new SpCons(temp);
    }else{
      cons->append(temp);
    }
  }

  //    static SpValue result;
  //    result.setNewObject(cons);
  //    return result;
  return SpObjectResult(cons);
}

/*
 * Message Handler
 */

SpValue& SpString::onMessage(SpValue& rec, SpValue& msg)
{
  return StringMsgHandler(rec, msg);
}

// string Primitives

SpValue& str_length(SpValue& s)
{
  SpString* ptr = dynamic_cast<SpString*>(s.getObject());
  if(ptr == NULL){
    throw SpException("not string (length)");
  }
  //    static SpValue v;
  //    v.setInt(ptr->length());
  //    return v;
  return SpIntResult(ptr->length());
}

SpValue& SpString::prim_nth(SpValue& self, SpValue& index)
{
  if(!index.isInt()){
    throw SpException("not int. (tuple.nth)");
  }
  SpInt i = index.getInt();
  SpString* str = self.asString();
  if(str->length() < i + 1){
    throw SpException("size over (string.nth)");
  }
  //    static SpValue result;
  //    result.setSpChar((*str)[i]);
  //    return result;
  return SpCharResult((*str)[i]);
}

SpValue& SpString::prim_sub(SpValue& self, SpValue& from, SpValue& len)
{
  if(!from.isInt()){
    throw SpException("not int. (string.sub)");
  }
  if(!len.isInt()){
    throw SpException("not int. (string.sub)");
  }
  SpInt i = from.getInt();
  SpInt j = len.getInt();
  if(j == 0){
    return NullString;
  }
  SpString* str = self.asString();
  if(str->length() < i+1){
    throw SpException("string is too short (string.sub)");
  }
  if(str->length() < i+j){
    throw SpException("string is too short. (string.sub)");
  }

  SpCharVector vec;
  for(int k=0; k < j; k++){
    vec.push_back(str->str[i+k]);
  }
  SpString* s = new SpString(vec);
  //    static SpValue result;
  //    result.setNewObject(s);
  //    return result;
  return SpObjectResult(s);
}

SpValue& SpString::prim_split(SpValue& self, SpValue& ch)
{
  if(!ch.isChar()){
    throw SpException("not char(string.split)");
  }
  SpChar c = ch.getChar();
  SpString* str = self.asString();

  //    static SpValue result;
  //    result = str->split(c);
  //    return result;
  return SpValueResult(str->split(c));
}

SpValue& SpString::prim_split1(SpValue& self, SpValue& ch)
{
  if(!ch.isChar()){
    throw SpException("not char(string.split1)");
  }
  SpChar c = ch.getChar();
  SpString* str = self.asString();

  if(str->length() == 0){
    return NilObject;
  }

  SpCons* cons;
  SpString s;
  SpCharVector::iterator it = str->begin();;

  for(; it < str->end(); it++){
    if(*it == c){
      it++;
      break;
    }
    s.addSpChar(*it);
  }
  SpString* s2;
  s2 = new SpString(s);
  SpValue temp;
  temp.setNewObject(s2);
  cons = new SpCons(temp);
  if(it < str->end()){
    s.clear();
    for(; it < str->end(); it++){
      s.addSpChar(*it);
    }
    s2 = new SpString(s);
    temp.setNewObject(s2);
    cons->append(temp);
  }else{
    cons->append(NullString);
  }

  //    static SpValue result;
  //    result.setNewObject(cons);
  //    return result;
  return SpObjectResult(cons);
}

SpValue& SpString::prim_tr(SpValue& self, SpValue& from, SpValue& to)
{
  if(!from.isChar()){
    throw SpException("not char(string.tr)");
  }
  if(!to.isChar()){
    throw SpException("not char(string.tr)");
  }
  SpChar c1 = from.getChar();
  SpChar c2 = to.getChar();
  SpString* str = self.asString();

  SpString* s = new SpString;
  SpCharVector::iterator it;
  for(it = str->begin(); it < str->end(); it++){
    if(*it == c1){
      s->addSpChar(c2);
    }else{
      s->addSpChar(*it);
    }
  }

  //    static SpValue result;
  //    result.setNewObject(s);
  //    return result;
  return SpObjectResult(s);
}

SpValue& SpString::prim_reader(SpValue& self)
{
  SpString* str = self.asString();
  StringReader* reader = new StringReader(str);

  //    static SpValue result;
  //    result.setNewObject(reader);
  //    return result;
  return SpObjectResult(reader);
}

SpValue& SpString::prim_upper(SpValue& self)
{
  SpString* str = self.asString();

  SpString* s = new SpString;
  SpCharVector::iterator it;
  for(it = str->begin(); it < str->end(); it++){
    s->addSpChar(SpChar2Upper(*it));
  }

  //    static SpValue result;
  //    result.setNewObject(s);
  //    return result;
  return SpObjectResult(s);
}

SpValue& SpString::prim_lower(SpValue& self)
{
  SpString* str = self.asString();

  SpString* s = new SpString;
  SpCharVector::iterator it;
  for(it = str->begin(); it < str->end(); it++){
    s->addSpChar(SpChar2Lower(*it));
  }

  //    static SpValue result;
  //    result.setNewObject(s);
  //    return result;
  return SpObjectResult(s);
}

SpValue& str2symbol(SpValue& s)
{
  SpString* ptr = dynamic_cast<SpString*>(s.getObject());
  if(ptr == NULL){
    throw SpException("not string (string.symbol)");
  }
  //    static SpValue v;
  //    v.setNewObject(new SpSymbol(*ptr));
  //    return v;
  return SpObjectResult(new SpSymbol(*ptr));
}

SpValue& SpString::prim_toInt(SpValue& self)
{
  SpString* str = self.asString();
  SpInt n = 0;

  SpCharVector::iterator it = str->str.begin();
  for(; it < str->str.end(); it++){
    SpChar c = *it;
    if(!isDIGIT(c)){
      throw SpException("not digit (string.toInt)");
    }
    n = n * 10 + (c - JIS('0'));
  }

  //    static SpValue result;
  //    result.setInt(n);
  //    return result;
  return SpIntResult(n);
}

SpValue& SpString::prim_replace(SpValue& self, SpValue& from, SpValue& to)
{
  if(!from.isString()){
    throw SpException("not string(string.replace)");
  }
  if(!to.isString()){
    throw SpException("not string(string.replace)");
  }
  SpString* src = self.asString();
  SpString* s1 = from.asString();
  SpString* s2 = to.asString();

  SpString* res = new SpString;
  SpCharVector::iterator sit; // source iterator
  SpCharVector::iterator fit; // from iterator

  sit = src->begin();
  fit = s1->begin();
  while(sit < src->end()){
    if(*sit == *fit){
      bool all_match = true;
      SpCharVector::iterator it1;
      SpCharVector::iterator it2;
      it1 = sit;
      it2 = fit;
      it1++;
      it2++;
      while(it2 < s1->end()){
        if(*it1 != *it2){
          all_match = false;
          break;
        }
        it1++;
        it2++;
      }
      if(all_match){
        SpCharVector::iterator it3 = s2->begin();
        for(; it3 < s2->end(); it3++){
          res->addSpChar(*it3);
        }

        sit += s1->length();
      }else{
        res->addSpChar(*sit);
        sit++;
      }
    }else{
      res->addSpChar(*sit);
      sit++;
    }
  }

  //    static SpValue result;
  //    result.setNewObject(res);
  //    return result;
  return SpObjectResult(res);
}

SpValue& SpString::prim_replace1(SpValue& self, SpValue& from, SpValue& to)
{
  if(!from.isString()){
    throw SpException("not string(string.replace1)");
  }
  if(!to.isString()){
    throw SpException("not string(string.replace1)");
  }
  SpString* src = self.asString();
  SpString* s1 = from.asString();
  SpString* s2 = to.asString();

  SpString* res = new SpString;
  SpCharVector::iterator sit; // source iterator
  SpCharVector::iterator fit; // from iterator

  sit = src->begin();
  fit = s1->begin();
  while(sit < src->end()){
    if(*sit == *fit){
      bool all_match = true;
      SpCharVector::iterator it1;
      SpCharVector::iterator it2;
      it1 = sit;
      it2 = fit;
      it1++;
      it2++;
      while(it2 < s1->end()){
        if(*it1 != *it2){
          all_match = false;
          break;
        }
        it1++;
        it2++;
      }
      if(all_match){
        SpCharVector::iterator it3 = s2->begin();
        for(; it3 < s2->end(); it3++){
          res->addSpChar(*it3);
        }
        sit += s1->length();
        for(; sit < src->end(); sit++){
          res->addSpChar(*sit);
        }
        break;
      }else{
        res->addSpChar(*sit);
        sit++;
      }
    }else{
      res->addSpChar(*sit);
      sit++;
    }
  }

  //    static SpValue result;
  //    result.setNewObject(res);
  //    return result;
  return SpObjectResult(res);
}

SpValue& SpString::prim_find(SpValue& self, SpValue& s, SpValue& from)
{
  //    static SpValue result;
  SpValue result;

  if(!s.isString()){
    throw SpException("not string(string.find)");
  }
  if(!from.isInt()){
    throw SpException("not int(string.find)");
  }
  SpString* src = self.asString();
  SpString* s1 = s.asString();
  SpInt start = from.getInt();

  SpCharVector::iterator it = src->begin();;
  //int i = 0;
  int i = start;
  it += start;
  int j;
cont:
  for(; *it != s1->str[0]; it++, i++){
    if(i >= src->length()){
      result.setInt(-1);
      //            return result;
      return SpValueResult(result);
    }
  }
  // found first char
  SpCharVector::iterator it2 = it;
  it2++;
  for(j=1; j < s1->length(); j++, it2++)
    {
      if(i+j >= src->length()){
        i = -1;
        break;
      }
      if(*it2 != s1->str[j]){
        it++;
        i++;
        goto cont;
      }
    }
  result.setInt(i);
  //    return result;
  return SpValueResult(result);
}

//
// format
//    ex. "format string %s %d %f\n" format aString aInt aReal;
//
static SpValue PrimAdd;
static SpValue& _prim_add(SpValue& a, SpValue& b)
{
  SpValue result, temp1, temp2;
  temp1 = a.eval();
  temp2 = b.eval();
  result = temp1.plus(temp2);
  return SpValueResult(result);
}

// '%s'
class _format_make_string : public SpFunc {
protected:
  bool   flag;     // '-' flag(l)
  int    width;
  SpChar preChar;

public:
  _format_make_string(SpChar ch, int len, bool f)
       : preChar(ch), width(len), flag(f)
  {
    argLen = 1;
  }
  virtual SpValue& operator()(SpValue& arg);
  SpValue& toString();
  const char* typeString(){ return "_format_make_string"; }
};

SpValue& _format_make_string::toString()
{
  SpString* str = new SpString("<make format string>");
  return SpObjectResult(str);
}

SpValue& _format_make_string::operator()(SpValue& arg)
{
  SpValue temp;
  SpValue result;

  SpString* str;
  if(!arg.isString()){
    temp = arg.toString();
    str  = temp.asString();
  }else{
    str = arg.asString();
  }
  int len = str->length();
  if((width == 0) || (width <= len)){
    result.setObject(str);
  }else{  // width > len
    if(flag){ // l
      SpString* str2 = new SpString();
      *str2 += *str;
      for(int i=0; i < width - len; i++){
        //str2->addSpChar(preChar);
        str2->addSpChar(MakeSpChar(CodeJIS, ' '));
      }
      result.setObject(str2);
    }else{
      SpString* str2 = new SpString();
      for(int i=0; i < width - len; i++){
        str2->addSpChar(preChar);
      }
      *str2 += *str;
      result.setObject(str2);
    }
  }

  return SpValueResult(result);
}

class _format_make_string_object : public _format_make_string {
public:
  _format_make_string_object(SpChar ch, int len, bool f) : _format_make_string(ch, len, f) {}
  SpValue& operator()(SpValue& arg){
    SpValue temp = arg.toString();
    return _format_make_string::operator()(temp);
  }
};

// '%u'
class _format_make_unsigned_string : public _format_make_string {
public:
  _format_make_unsigned_string(SpChar ch, int len, bool f) : _format_make_string(ch, len, f) {}
  SpValue& operator()(SpValue& arg);
};

SpValue& _format_make_unsigned_string::operator()(SpValue& arg)
{
  if(!arg.isInt()){
    throw SpException("not int in string.format '%u'");
  }
  SpInt n = arg.getInt();
  char buf[64];
  sprintf(buf, "%u", n);
  SpValue temp;
  temp.setNewObject(new SpString(buf));
  return _format_make_string::operator()(temp);
}

// '%c'
class _format_make_char_string : public _format_make_string {
public:
  _format_make_char_string(SpChar ch, int w, bool f)
       : _format_make_string(ch, w, f) {}
  SpValue& operator()(SpValue& arg);
  SpValue& toString()
  {
    SpString* str = new SpString("<make format string:char>");
    return SpObjectResult(str);
  }
  const char* typeString(){ return "_format_make_char_string"; }
};

SpValue& _format_make_char_string::operator()(SpValue& arg)
{
  SpValue temp;
  SpInt n;
  SpString* str;

  str = new SpString();
  temp.setNewObject(str);
  n = arg.getInt();
  str->addSpChar((SpChar)n);

  return _format_make_string::operator()(temp);
}

// '%f'
class _format_make_real_string : public SpFunc {
  char format[64];

public:
  _format_make_real_string(char* str)
  {
    argLen = 1;
    strncpy(format, str, 63);
  }
  SpValue& operator()(SpValue& arg);
  SpValue& toString()
  {
    SpString* str = new SpString("<make format string:real>");
    return SpObjectResult(str);
  }
  const char* typeString(){ return "_format_make_real_string"; }
};

SpValue& _format_make_real_string::operator()(SpValue& arg)
{
  SpValue result;
  SpReal real;
  char buf[128];

  if(arg.isInt()){
    arg.setReal((SpReal)arg.getInt());
  }
  if(!arg.isReal()){
    throw SpException("arg is not real in format string");
  }
  real = arg.getReal();
  sprintf(buf, format, real);
  result.setNewObject(new SpString(buf));

  return SpValueResult(result);
}

// '%b'
class _format_make_b_string : public SpFunc {
private:
  int width;
  bool fSharp;
  bool fMinus;
  bool fZero;

public:
  _format_make_b_string(int w, bool s, bool m, bool z)
       : width(w), fSharp(s), fMinus(m), fZero(z)
  {
    argLen = 1;
  }
  SpValue& operator()(SpValue& arg);
  SpValue& toString()
  {
    SpString* str = new SpString("<make format string:b>");
    return SpObjectResult(str);
  }
  const char* typeString(){ return "_format_make_b_string"; }
};

#ifndef INT_BIT
#define INT_BIT (CHAR_BIT * (sizeof(int) / sizeof(char)))
#endif

SpValue& _format_make_b_string::operator()(SpValue& arg)
{
  SpValue result;
  SpValue temp;
  SpInt n;
  SpString* str = NULL;

  if(!arg.isInt()){
    throw SpException("not int '%b'");
  }
  n = arg.getInt();

  for(int i = INT_BIT-1; i >= 0; i--){
    if(((n >> i) & 1) == 1){
      if(str == NULL){
        str = new SpString("1");
        result.setObject(str);
      }else{
        str->addSpChar(MakeSpChar(CodeJIS, '1'));
      }
    }else{
      if(str != NULL){
        str->addSpChar(MakeSpChar(CodeJIS, '0'));
      }
    }
  }
  if(str == NULL){
    str = new SpString();
    result.setObject(str);
  }

  int len = str->length();
  if(fSharp){ // '#' flag
    if((width - 2) > len){
      SpString* str2;
      if(fMinus){ // l
        str2 = new SpString("0b");
        *str2 += *str;
        for(int i=0; i < (width - 2) - len; i++){
          str2->addSpChar(MakeSpChar(CodeJIS, ' '));
        }
      }else{
        if(fZero){
          str2 = new SpString("0b");
          for(int i=0; i < (width - 2) - len; i++){
            str2->addSpChar(MakeSpChar(CodeJIS, '0'));
          }
        }else{
          str2 = new SpString();
          for(int i=0; i < (width - 2) - len; i++){
            str2->addSpChar(MakeSpChar(CodeJIS, ' '));
          }
          str2->addSpChar(MakeSpChar(CodeJIS, '0'));
          str2->addSpChar(MakeSpChar(CodeJIS, 'b'));
        }
        *str2 += *str;
      }
      result.setObject(str2);
    }else{
      SpString* str2 = new SpString("0b");
      SpValue temp;
      temp.setObject(str2);
      *str2 += *str;
      result.setObject(str2);
    }
  }else{
    if(width > len){
      SpString* str2 = new SpString();
      if(fMinus){ // l
        *str2 += *str;
        for(int i=0; i < width - len; i++){
          str2->addSpChar(MakeSpChar(CodeJIS, ' '));
        }
      }else{
        SpChar preChar;
        if(fZero){
          preChar = MakeSpChar(CodeJIS, '0');
        }else{
          preChar = MakeSpChar(CodeJIS, ' ');
        }
        for(int i=0; i < width - len; i++){
          str2->addSpChar(preChar);
        }
        *str2 += *str;
      }
      result.setObject(str2);
    }
  }

  return SpValueResult(result);
}

// '%o'
class _format_make_o_string : public SpFunc {
private:
  int width;
  bool fSharp;
  bool fMinus;
  bool fZero;

public:
  _format_make_o_string(int w, bool s, bool m, bool z)
       : width(w), fSharp(s), fMinus(m), fZero(z)
  {
    argLen = 1;
  }
  SpValue& operator()(SpValue& arg);
  SpValue& toString()
  {
    SpString* str = new SpString("<make format string:o>");
    return SpObjectResult(str);
  }
  const char* typeString(){ return "_format_make_o_string"; }
};

SpValue& _format_make_o_string::operator()(SpValue& arg)
{
  SpValue result;
  SpValue temp;
  SpUInt n;
  SpString* str;

  if(!arg.isInt()){
    throw SpException("not int '%o'");
  }
  n = (SpUInt)arg.getInt();

  str = new SpString();
  result.setNewObject(str);
  for(SpUInt i = n; i > 0; i >>= 3){
    int j = i & 0x7;
    str->insertSpChar(MakeSpChar(CodeJIS, j + '0'));
  }
  if(str->length() == 0){
    str->addSpChar(MakeSpChar(CodeJIS, '0'));
  }

  int len = str->length();
  if(fSharp){ // '#' flag
    if((width - 2) > len){
      SpString* str2;
      if(fMinus){ // l
        str2 = new SpString("00");
        *str2 += *str;
        for(int i=0; i < (width - 2) - len; i++){
          str2->addSpChar(MakeSpChar(CodeJIS, ' '));
        }
      }else{
        if(fZero){
          str2 = new SpString("00");
          for(int i=0; i < (width - 2) - len; i++){
            str2->addSpChar(MakeSpChar(CodeJIS, '0'));
          }
        }else{
          str2 = new SpString();
          for(int i=0; i < (width - 2) - len; i++){
            str2->addSpChar(MakeSpChar(CodeJIS, ' '));
          }
          str2->addSpChar(MakeSpChar(CodeJIS, '0'));
          str2->addSpChar(MakeSpChar(CodeJIS, '0'));
        }
        *str2 += *str;
      }
      result.setObject(str2);
    }else{
      SpString* str2 = new SpString("00");
      SpValue temp;
      temp.setObject(str2);
      *str2 += *str;
      result.setObject(str2);
    }
  }else{
    if(width > len){
      SpString* str2 = new SpString();
      if(fMinus){ // l
        *str2 += *str;
        for(int i=0; i < width - len; i++){
          str2->addSpChar(MakeSpChar(CodeJIS, ' '));
        }
      }else{
        SpChar preChar;
        if(fZero){
          preChar = MakeSpChar(CodeJIS, '0');
        }else{
          preChar = MakeSpChar(CodeJIS, ' ');
        }
        for(int i=0; i < width - len; i++){
          str2->addSpChar(preChar);
        }
        *str2 += *str;
      }
      result.setObject(str2);
    }
  }

  return SpValueResult(result);
}

// '%x' '%X'
class _format_make_hex_string : public SpFunc {
private:
  int width;
  bool upper;
  bool fSharp;
  bool fMinus;
  bool fZero;

public:
  _format_make_hex_string(int w, bool up, bool s, bool m, bool z)
       : width(w), upper(up), fSharp(s), fMinus(m), fZero(z)
  {
    argLen = 1;
  }
  SpValue& operator()(SpValue& arg);
  SpValue& toString()
  {
    SpString* str = new SpString("<make format string:x(X)>");
    return SpObjectResult(str);
  }
  const char* typeString(){ return "_format_make_hex_string"; }
};

SpValue& _format_make_hex_string::operator()(SpValue& arg)
{
  SpValue result;
  SpValue temp;
  SpUInt n;
  SpString* str;

  if(!arg.isInt()){
    throw SpException("not int '%x' or '%X'");
  }
  n = (SpUInt)arg.getInt();

  str = new SpString();
  result.setNewObject(str);
  for(SpUInt i = n; i > 0; i >>= 4){
    int j = i & 0xF;
    if(j <= 9){
      str->insertSpChar(MakeSpChar(CodeJIS, j + '0'));
    }else{
      if(upper){
        str->insertSpChar(MakeSpChar(CodeJIS, j - 10 + 'A'));
      }else{
        str->insertSpChar(MakeSpChar(CodeJIS, j - 10 + 'a'));
      }
    }
  }
  if(str->length() == 0){
    str->addSpChar(MakeSpChar(CodeJIS, '0'));
  }

  int len = str->length();
  if(fSharp){ // '#' flag
    if((width - 2) > len){
      SpString* str2;
      if(fMinus){ // l
        if(upper){
          str2 = new SpString("0X");
        }else{
          str2 = new SpString("0x");
        }
        *str2 += *str;
        for(int i=0; i < (width - 2) - len; i++){
          str2->addSpChar(MakeSpChar(CodeJIS, ' '));
        }
      }else{
        if(fZero){
          if(upper){
            str2 = new SpString("0X");
          }else{
            str2 = new SpString("0x");
          }
          for(int i=0; i < (width - 2) - len; i++){
            str2->addSpChar(MakeSpChar(CodeJIS, '0'));
          }
        }else{
          str2 = new SpString();
          for(int i=0; i < (width - 2) - len; i++){
            str2->addSpChar(MakeSpChar(CodeJIS, ' '));
          }
          str2->addSpChar(MakeSpChar(CodeJIS, '0'));
          if(upper){
            str2->addSpChar(MakeSpChar(CodeJIS, 'X'));
          }else{
            str2->addSpChar(MakeSpChar(CodeJIS, 'x'));
          }
        }
        *str2 += *str;
      }
      result.setObject(str2);
    }else{
      SpString* str2;
      if(upper){
        str2 = new SpString("0X");
      }else{
        str2 = new SpString("0x");
      }
      SpValue temp;
      temp.setObject(str2);
      *str2 += *str;
      result.setObject(str2);
    }
  }else{
    if(width > len){
      SpString* str2 = new SpString();
      if(fMinus){ // l
        *str2 += *str;
        for(int i=0; i < width - len; i++){
          str2->addSpChar(MakeSpChar(CodeJIS, ' '));
        }
      }else{
        SpChar preChar;
        if(fZero){
          preChar = MakeSpChar(CodeJIS, '0');
        }else{
          preChar = MakeSpChar(CodeJIS, ' ');
        }
        for(int i=0; i < width - len; i++){
          str2->addSpChar(preChar);
        }
        *str2 += *str;
      }
      result.setObject(str2);
    }
  }

  return SpValueResult(result);
}

// prim_format
SpValue& SpString::prim_format(SpValue& self)
{
  SpCharVector vec;
  SpChar preChar;
  bool flagSharp = false;
  bool flagPlus  = false;
  bool flagSpace = false;
  bool flagMinus = false;
  bool flagZero  = false;
  bool upper_hex = true;
  int  width  = -1;  // -1 ̂Ƃɂ͕̎w͖
  int  width2 = -1;  // -1 ̂Ƃɂ͕̎w͖Ax̏ȗl͂U
  SpValueVector args;
  SpCons* body = NULL;
  SpCons* exps = NULL;
  SpFunc* func = NULL;
  SpValue vArgs;
  SpValue vBody;
  SpValue temp;
  SpValue temp2;
  SpValue temp3;
  SpSymbol* sym;
  int count = 0;
  char buf[64];
  char format[64];

  SpString* ptr = dynamic_cast<SpString*>(self.getObject());
  if(ptr == NULL){
    throw SpException("not string (format)");
  }
  // parse format string
  SpCharVector::iterator it;
  it = ptr->str.begin();
  for(; it != ptr->str.end(); it++){
    flagSharp = false;
    flagPlus  = false;
    flagSpace = false;
    flagMinus = false;
    flagZero  = false;
    upper_hex = true;
    width  = -1;
    width2 = -1;

    if(*it == MakeSpChar(CodeJIS, '%')){    // parse '%'
      it++;
      if(it == ptr->str.end()){
        throw SpException("illegal format string");
      }
      if(*it == MakeSpChar(CodeJIS, '%')){    // parse "%%"
        vec.push_back(*it);
        continue;
      }

      // init preChar & width
      preChar = MakeSpChar(CodeJIS, ' ');
      width = 0;
      temp.setNewObject(new SpString(vec));
      vec.clear();
      if(exps == NULL){
        exps = new SpCons(temp);
      }else{
        exps->append(temp);
      }

      // parse flag ('#', '+', ' ', '-', '0')
parse_flags:
      switch(*it){
      case MakeSpChar(CodeJIS, '#'):
        flagSharp = true;
        it++;
        if(it == ptr->str.end()){
          throw SpException("illegal format string");
        }
        goto parse_flags;
        break;
      case MakeSpChar(CodeJIS, '+'):
        flagPlus = true;
        it++;
        if(it == ptr->str.end()){
          throw SpException("illegal format string");
        }
        goto parse_flags;
        break;
      case MakeSpChar(CodeJIS, ' '):
        flagSpace = true;
        it++;
        if(it == ptr->str.end()){
          throw SpException("illegal format string");
        }
        goto parse_flags;
        break;
      case MakeSpChar(CodeJIS, '-'):
        flagMinus = true;
        it++;
        if(it == ptr->str.end()){
          throw SpException("illegal format string");
        }
        goto parse_flags;
        break;
      case MakeSpChar(CodeJIS, '0'):
        flagZero = true;
        preChar  = *it;
        it++;
        if(it == ptr->str.end()){
          throw SpException("illegal format string");
        }
        break;
      }

      // parse width
      if(isDIGIT(*it)){
        width = toINT(*it);
        for(it++; it != ptr->str.end() && isDIGIT(*it); it++){
          width = width * 10 + toINT(*it);
        }
      }

      // parser width2 '.num'
      if(*it == MakeSpChar(CodeJIS, '.')){
        it++;
        width2 = 0;
        for(; it != ptr->str.end() && isDIGIT(*it); it++){
          width2 = width2 * 10 + toINT(*it);
        }
      }

      switch(*it){
      case MakeSpChar(CodeJIS, 'c'): // char
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        temp.setNewObject(new _format_make_char_string(preChar, width, flagMinus));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      case MakeSpChar(CodeJIS, 's'): // string
      case MakeSpChar(CodeJIS, 'p'):
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        temp.setNewObject(new _format_make_string(preChar, width, flagMinus));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      case MakeSpChar(CodeJIS, 'd'): // digit
      case MakeSpChar(CodeJIS, 'i'):
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        temp.setNewObject(new _format_make_string_object(preChar, width, flagMinus));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      case MakeSpChar(CodeJIS, 'u'): // unsigned int
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        temp.setNewObject(new _format_make_unsigned_string(preChar, width, flagMinus));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      case MakeSpChar(CodeJIS, 'b'): // bin
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        temp.setNewObject(new _format_make_b_string(width, flagSharp, flagMinus, flagZero));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      case MakeSpChar(CodeJIS, 'o'): // oct
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        temp.setNewObject(new _format_make_o_string(width, flagSharp, flagMinus, flagZero));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      case MakeSpChar(CodeJIS, 'x'): // hex
        upper_hex = false;
      case MakeSpChar(CodeJIS, 'X'): // HEX
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        temp.setNewObject(new _format_make_hex_string(width, upper_hex, flagSharp, flagMinus, flagZero));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      case MakeSpChar(CodeJIS, 'f'):
      case MakeSpChar(CodeJIS, 'e'):
      case MakeSpChar(CodeJIS, 'E'):
      case MakeSpChar(CodeJIS, 'g'):
      case MakeSpChar(CodeJIS, 'G'):
#ifdef _MICROSOFTC_
        _snprintf(buf, 60, "a%d", count++);
#else
        snprintf(buf, 60, "a%d", count++);
#endif
        sym = new SpSymbol(buf);
        temp2.setNewObject(sym);
        if(width2 == -1){
          width2 = 6;
        }
        if(width < 0){
          if(flagSharp){
            sprintf(format, "%%#.%d%c", width2, SpCharGetChar(*it));
          }else if(flagPlus){
            sprintf(format, "%%+.%d%c", width2, SpCharGetChar(*it));
          }else if(flagSpace){
            sprintf(format, "%% .%d%c", width2, SpCharGetChar(*it));
          }else if(flagMinus){
            sprintf(format, "%%-.%d%c", width2, SpCharGetChar(*it));
          }else if(flagZero){
            sprintf(format, "%%0.%d%c", width2, SpCharGetChar(*it));
          }else{
            sprintf(format, "%%.%d%c", width2, SpCharGetChar(*it));
          }
        }else{
          if(flagSharp){
            sprintf(format, "%%#%d.%d%c", width, width2, SpCharGetChar(*it));
          }else if(flagPlus){
            sprintf(format, "%%+%d.%d%c", width, width2, SpCharGetChar(*it));
          }else if(flagSpace){
            sprintf(format, "%% %d.%d%c", width, width2, SpCharGetChar(*it));
          }else if(flagMinus){
            sprintf(format, "%%-%d.%d%c", width, width2, SpCharGetChar(*it));
          }else if(flagZero){
            sprintf(format, "%%0%d.%d%c", width, width2, SpCharGetChar(*it));
          }else{
            sprintf(format, "%%%d.%d%c", width, width2, SpCharGetChar(*it));
          }
        }
        temp.setNewObject(new _format_make_real_string(format));
        temp.setNewObject(new SpSendMsg(temp, temp2));
        exps->append(temp);
        temp3.setNewObject(new SpArg(temp2, NilObject));
        args.push_back(temp3);
        break;
      default:
        throw SpException("illegal format string");
      }

    }else{
      vec.push_back(*it);
    }
  }
  if(vec.size() > 0){
    if(exps == NULL){
      temp.setNewObject(new SpString(vec));
      body = new SpCons(temp);
    }else{
      SpValue recv, msg;
      SpSendMsg* sm;
      temp.setNewObject(new SpString(vec));
      exps->append(temp);
      recv.setObject(exps);
      sm = new SpSendMsg(recv, SymFoldl);
      recv.setObject(sm);
      sm = new SpSendMsg(recv, PrimAdd);
      temp.setNewObject(sm);
      body = new SpCons(temp);
    }
  }else{
    if(exps != NULL){
      SpValue recv, msg;
      SpSendMsg* sm;
      recv.setObject(exps);
      sm = new SpSendMsg(recv, SymFoldl);
      recv.setObject(sm);
      sm = new SpSendMsg(recv, PrimAdd);
      temp.setNewObject(sm);
      body = new SpCons(temp);
    }
  }

  // make function
  if(args.size() != 0){
    vArgs.setNewObject(new SpTuple(args));
  }
  if(body != NULL){
    vBody.setObject(body);
  }

  func = new SpUsrFunc(NilObject, vArgs, NilObject, NilObject, vBody, NilObject, NilObject);

  return SpObjectResult(func);
}

// init Message Handler
void SpString::init()
{
  StringMsgHandler.append(SymIsString, PrimTrue);

  PrimAdd.setNewObject(new SpPrim2(_prim_add));

  SpValue PrimLength(new SpPrim1(str_length));
  StringMsgHandler.append(SymLength, PrimLength);

  SpValue PrimNth(new SpPrim2(prim_nth));
  StringMsgHandler.append(SymNth, PrimNth);

  SpValue SymSub(new SpSymbol("sub"));
  SpValue PrimSub(new SpPrim3(prim_sub));
  StringMsgHandler.append(SymSub, PrimSub);

  SpValue SymSymbol(new SpSymbol("symbol"));
  SpValue PrimSymbol(new SpPrim1(str2symbol));
  StringMsgHandler.append(SymSymbol, PrimSymbol);

  SpValue PrimSplit(new SpPrim2(prim_split));
  StringMsgHandler.append(SymSplit, PrimSplit);

  SpValue SymTr(new SpSymbol("tr"));
  SpValue PrimTr(new SpPrim3(prim_tr));
  StringMsgHandler.append(SymTr, PrimTr);

  SpValue PrimReader(new SpPrim1(prim_reader));
  StringMsgHandler.append(SymReader, PrimReader);

  SpValue PrimUpper(new SpPrim1(prim_upper));
  StringMsgHandler.append(SymUpper, PrimUpper);

  SpValue PrimLower(new SpPrim1(prim_lower));
  StringMsgHandler.append(SymLower, PrimLower);

  SpValue PrimReplace(new SpPrim3(prim_replace));
  StringMsgHandler.append(SymReplace, PrimReplace);

  SpValue PrimToInt(new SpPrim1(prim_toInt));
  StringMsgHandler.append(SymToInt, PrimToInt);

  SpValue PrimFind(new SpPrim3(prim_find));
  StringMsgHandler.append(SymFind, PrimFind);

  SpValue PrimSplit1(new SpPrim2(prim_split1));
  StringMsgHandler.append(SymSplit1, PrimSplit1);

  SpValue PrimReplace1(new SpPrim3(prim_replace1));
  StringMsgHandler.append(SymReplace1, PrimReplace1);

  SpValue SymFormat(new SpSymbol("format"));
  SpValue PrimFormat(new SpPrim1(prim_format));
  StringMsgHandler.append(SymFormat, PrimFormat);
}

