/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002,2003 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: SjisRW.cpp,v 1.28 2004/05/22 05:22:34 randy Exp $
 */

#ifdef __WIN32__
#include <io.h>
#else
#include <unistd.h>
#include <stdio.h>
#endif

#include "soopy.h"

/*
 * Class SjisReadEncoder
 *   TronCode -> Sjis
 */

SpChar SjisReadEncoder::readCh()
{
    SpChar c, ch, ch2, ku, ten;

    if(count >= 0){
        return buf[count--];
    }
    if(eof()){ return (SpChar)NULL; }
    c = reader->ReadChar();
//printf("w-code: %x\n", c);
    if(isEOF(c)){ return (SpChar)NULL; }

    if(!EqualCharCode(c, CodeJIS)){
        return c;
    }

    ch = SpCharGetChar(c);
    ku = (ch >> 8);
    ten = (ch & 0xFF);
    if(ku == 0){
        return ten;
    }else{
        ku  -= 0x20;
        ten -= 0x20;
        // first byte
        if(ku < 63){
//printf("ku1: %x\n", (unsigned char)(0x81 + ((ku-1)>>1)));
            ch2 = ((unsigned char)(0x81 + ((ku-1)>>1)));
        }else{
//printf("ku2: %x\n", (unsigned char)(0xE0 + ((ku-63)>>1)));
            ch2 = ((unsigned char)(0xE0 + ((ku-63)>>1)));
        }
        // second byte
        if(ku & 1){
            if(ten < 0x3F){
//printf("ten1: %x\n", (unsigned char)(ten + 0x3F));
                buf[0] = ((unsigned char)(ten + 0x3F));
            }else{
//printf("ten2: %x\n", (unsigned char)(ten + 0x40));
                buf[0] = ((unsigned char)(ten + 0x40));
            }
        }else{
//printf("ten3: %x\n", (unsigned char)(0x9E + ten));
            buf[0] = ((unsigned char)(0x9E + ten));
        }
        count = 0;
    }
    return ch2;
}

/*
 * Class SjisReadDecoder
 *   Sjis -> TronCode
 */

SpChar SjisReadDecoder::readCh()
{
    SpChar c;
    int ch1, ch2;
    int ku, ten, base1, base2;
    if(eof()){ return (SpChar)NULL; }
    c = reader->ReadChar();
    if(isEOF(c)){ return (SpChar)NULL; }
    ch1 = c & 0xff;
    if(ch1 > 0x7F){
        /*
        if(0xA1 <= ch1 && ch1 <= 0xDF){ // when hankaku KANA
            // return Zenkaku KANA
        }
        */
        if((0x81 <= ch1 && ch1 <= 0x9F) 
           || (0xE0 <= ch1 && ch1 <= 0xEF)) // JIS X 0208
        {
            if(eof()){ return (SpChar)NULL; }
            c = reader->ReadChar();
            ch2 = c & 0xff;
            base1 = ch1 < 0xA0 ? 0x80 : 0xDF;
            base2 = ch1 < 0xA0 ? 0 : 62;
            //if(ch2 < 0x9E){
            if(ch2 < 0x9F){
                ku = ((ch1 - base1)<<1) - 1 + base2;
                ten = ch2 - (ch2 < 0x7F ? 0x3F : 0x40);
            }else{
                ku = ((ch1 - base1)<<1) + base2;
                ten = ch2 - 0x9E;
            }
            ku += 0x20;
            ten += 0x20;
//printf("ch: %x\n", MakeSpChar(CodeJIS, (ku<<8) | ten));
            return MakeSpChar(CodeJIS, (ku<<8) | ten);
        }else{
            return MakeSpChar(CodeJIS, '?');
        }
    }
//printf("ch2: %x\n", MakeSpChar(CodeJIS, c));
    return MakeSpChar(CodeJIS, c);
}

/*
 * Class SjisWriteEncoder
 *   TronCode -> Sjis
 */

void SjisWriteEncoder::WriteChar(SpChar c)
{
    int ch, ku, ten;

//printf("w-code--: %x\n", c);
    if(!EqualCharCode(c, CodeJIS)){
        writeTextTronCode(c);
        return;
    }
    ch = SpCharGetChar(c);
    ku = (ch >> 8);
    ten = (ch & 0xFF);
    if(ku == 0){
        writer->WriteChar(ten);
    }else{
        ku  -= 0x20;
        ten -= 0x20;
        // put first byte
        if(ku < 63){
//printf("ku_1: %x\n", (unsigned char)(0x81 + ((ku-1)>>1)));
            writer->WriteChar((unsigned char)(0x81 + ((ku-1)>>1)));
        }else{
//printf("ku_2: %x\n", (unsigned char)(0xE0 + ((ku-63)>>1)));
             writer->WriteChar((unsigned char)(0xE0 + ((ku-63)>>1)));
        }
        // put second byte
        if(ku & 1){
            if(ten < 0x3F){
//printf("ten_1: %x\n", (unsigned char)(ten + 0x3F));
                writer->WriteChar((unsigned char)(ten + 0x3F));
            }else{
//printf("ten_2: %x\n", (unsigned char)(ten + 0x40));
                writer->WriteChar((unsigned char)(ten + 0x40));
            }
        }else{
//printf("ten_3: %x\n", (unsigned char)(0x9E + ten));
            writer->WriteChar((unsigned char)(0x9E + ten));
        }
    }
}


/*
 * Class SjisWriteDecoder
 *   Sjis -> TronCode
 */

void SjisWriteDecoder::WriteChar(SpChar c)
{
    int ch2;
    int ku, ten, base1, base2;

    if(count == 0){
        if(c > 0xFF){ // not sjis
            if(isEOL(c)){
                writer->WriteChar(c);
            }else{
                writeTextTronCode(c);
            }
        }else if(c > 0x7F){
            /*
            if(0xA1 <= ch1 && ch1 <= 0xDF){ // when hankaku KANA
                // write Zenkaku KANA
            }
            */
            ch1 = c;
            count++;
        }else{
            writer->WriteChar(c);
        }
        return;
    }
    // count == 1
    count = 0;
    if((0x81 <= ch1 && ch1 <= 0x9F) 
       || (0xE0 <= ch1 && ch1 <= 0xEF)) // JIS X 0208
    {
        ch2 = c;
        base1 = ch1 < 0xA0 ? 0x80 : 0xDF;
        base2 = ch1 < 0xA0 ? 0 : 62;
        if(ch2 < 0x9F){
            ku = ((ch1 - base1)<<1) - 1 + base2;
            ten = ch2 - (ch2 < 0x7F ? 0x3F : 0x40);
        }else{
            ku = ((ch1 - base1)<<1) + base2;
            ten = ch2 - 0x9E;
        }
        ku += 0x20;
        ten += 0x20;
        writer->WriteChar(MakeSpChar(CodeJIS, (ku<<8) | ten));
    }else{
        writer->WriteChar(MakeSpChar(CodeJIS, '?'));
    }
}


/*
 * SjisFile
 *   openIn, openOut
 */

SpValue& sjis_openIn(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (Sjis.openIn)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ifstream* fin = new ifstream;
#ifdef __WIN32__
    fin->open(filename, ios::binary);
#else
    fin->open(filename);
#endif
    if(fin->fail()){
        delete fin;
        throw SpException("can't open file");
    }
    FileStreamReader* sr = new FileStreamReader(fin, filename);
    ReadEncoder* encoder = new SjisReadDecoder(sr);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& sjis_openOut(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (Sjis.openOut)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ofstream* fout = new ofstream;
#ifdef __WIN32__
    fout->open(filename, ios::binary);
#else
    fout->open(filename);
#endif
    if(fout->fail()){
        delete fout;
        throw SpException("can't open file");
    }
    FileStreamWriter* sw = new FileStreamWriter(fout, filename);
    WriteEncoder* encoder = new SjisWriteEncoder(sw);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& sjis_openAppend(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (Sjis.openAppend)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ofstream* fout = new ofstream;
#ifdef __WIN32__
    fout->open(filename, ios::app | ios::binary);
#else
    fout->open(filename, ios::app);
#endif
    if(fout->fail()){
        delete fout;
        throw SpException("can't open file");
    }
    FileStreamWriter* sw = new FileStreamWriter(fout, filename);
    WriteEncoder* encoder = new SjisWriteEncoder(sw);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& sjis_encoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (Sjis.encoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new SjisReadEncoder(reader);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& sjis_decoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (Sjis.decoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new SjisReadDecoder(reader);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& sjis_encoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (Sjis.encoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new SjisWriteEncoder(writer);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& sjis_decoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (Sjis.decoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new SjisWriteDecoder(writer);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

// init
void initSjisFile()
{
    SpValue SymSjis(new SpSymbol("sjis"));
    SpNameSpace* ns = new SpNameSpace;
    SpValue NSSjis(ns);
    PMainNameSpace->internConst(SymSjis, NSSjis);

    SpValue PrimOpenIn(new SpPrim1(&sjis_openIn));
    ns->internConst(SymOpenIn, PrimOpenIn);
    SpValue PrimEncoderIn(new SpPrim1(&sjis_encoderIn));
    ns->internConst(SymEncoderIn, PrimEncoderIn);
    SpValue PrimDecoderIn(new SpPrim1(&sjis_decoderIn));
    ns->internConst(SymDecoderIn, PrimDecoderIn);

    SpValue PrimOpenOut(new SpPrim1(&sjis_openOut));
    ns->internConst(SymOpenOut, PrimOpenOut);
    SpValue PrimOpenAppend(new SpPrim1(&sjis_openAppend));
    ns->internConst(SymOpenAppend, PrimOpenAppend);
    SpValue PrimEncoderOut(new SpPrim1(&sjis_encoderOut));
    ns->internConst(SymEncoderOut, PrimEncoderOut);
    SpValue PrimDecoderOut(new SpPrim1(&sjis_decoderOut));
    ns->internConst(SymDecoderOut, PrimDecoderOut);
}

