/*
 * Programming SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: RW.cpp,v 1.46 2007/01/29 11:46:16 randy Exp $
 */

#ifndef __BORLANDC__
#include <stdio.h>
#endif
#include "soopy.h"

/*
 * class Reader
 */

/*
 * class StreamReader
 */

SpValue& StreamReader::Read(SpInt n)
{
    if(eof()){
        return NullString;
    }

    SpChar c;
    SpString* str = new SpString;

    unsigned char* buf = new unsigned char[n];
    stream->read((char*)buf, n);
    for(SpInt i=0; i < n; i++){
        str->addSpChar(MakeSpChar(CodeJIS, buf[i]));
    }
    delete buf;

    return SpObjectResult(str);
}

// n bytes ǂݍށB
// ۂɓǂݍ񂾃oCgԂ
SpValue& StreamReader::ReadByte(SpInt n, unsigned char* buf)
{
    if(eof()){
        n = 0;
    }else{
#ifdef NO_READSOME
      n = 0;
      for( ; ; n++, buf++){
        stream->read((char*)buf, 1);
        if(stream->fail()){
          break;
        }
      }
#else
        n = stream->readsome((char*)buf, n);
#endif
    }
    if(stream->fail()){
        throw SpFileException("can't readbyte");
    }

    return SpIntResult(n);
}

// n bytes ǂݔ΂B
// ۂɓǂݍ񂾃oCgԂ
SpValue& StreamReader::SkipByte(SpInt n)
{
    char* buf = new char[n];
    try {
        if(eof()){
            n = 0;
        }else{
#ifdef NO_READSOME
            n = 0;
            for( ; ; n++, buf++){
                stream->read((char*)buf, 1);
                if(stream->fail()){
                    break;
                }
            }
#else
            n = stream->readsome((char*)buf, n);
#endif
        }
        if(stream->fail()){
            throw SpFileException("can't skipbyte");
        }
    }catch(...){
        delete[] buf;
        throw;
    }
    delete[] buf;

    return SpIntResult(n);
}

// StreamReader::ScanByte
//   oCgPʂŁAtH[}bgɂēǂݍ
//   lԂBԂl̏ꍇɂ̓^vԂB
//  ioCg̎w͕K{j
//  ex.
//    fin scanbyte "%4d";  # int(little endian)
//    fin scanbyte "%4D";  # int(big endian)
//    fin scanbyte "%4b";  # bytes oCgzԂ
//    fin scanbyte "%4w";  # words (little endian) [hzԂ
//    fin scanbyte "%4W";  # words (big endian)    [hzԂ
//    fin scanbyte "?";    # skip a byte
//    fin scanbyte "%5*";  # skip 5 bytes
//    fin scanbyte "%8s";  # 8oCg𕶎Ƃēǂݍ
//                         # finGR[_̂Ƃɂ́ÃGR[_ɏ]ĉ߂
//
//  ȉA{c
//    fin scanbyte "%10{sjis}";  # 10oCgsjisGR[_gĕƂēǂݍ
//                               # {} ̂Ȃ͕̎]GR[_ƂĎg(??)
//
SpValue& StreamReader::ScanByte(SpString* format, Reader* self)
{
    unsigned char byte;
    SpValue result;
    SpValue temp;
    SpTuple* tpl = NULL;
    SpCharVector::iterator it;

    for(it = format->begin(); it != format->end(); it++){
        if(*it == MakeSpChar(CodeJIS, '?')){  // '?' skip a byte
            stream->seekg(1, ios::cur);
            if(stream->fail()){
                throw SpException("can't skip a byte (scanbyte)");
            }
        }else if(*it == MakeSpChar(CodeJIS, '%')){
            SpInt no;
            unsigned char* ptr;
            uintvec bvec;
            SpByteArray* bary;

            it++;
            if(!isDIGIT(*it)){
                throw SpException("illegal format. not int (scanbyte)");
            }
            // ǂݍރoCgvZ
            int count = SpCharGetChar(*it) - '0';
            for(it++; isDIGIT(*it); it++){
                count = (count * 10) + SpCharGetChar(*it) - '0';
            }
            switch(SpCharGetChar(*it)){
            case 'd':  // little endian
                no = 0;
                if(count > sizeof(SpInt)){
                    throw SpException("too big size. (scanbyte '%d')");
                }
                ptr = (unsigned char*)&no;
                for(int i=0; i < count; i++, ptr++){
		  //#if defined(LINUX) || defined(OSX)
                    stream->read((char*)ptr, 1);
		    //#else
		    //                    stream->read(ptr, 1);
		    //#endif
                    if(stream->fail()){
                        throw SpException("can't read byte");
                    }
                }
                // set result
                if(result.isNil()){
                    result.setInt(no);
                }else if(tpl == NULL){
                    tpl = new SpTuple(result);
                    result.setObject(tpl);
                    temp.setInt(no);
                    tpl->append(temp);
                }else{
                    temp.setInt(no);
                    tpl->append(temp);
                }
                break;
            case 'D':  // big endian
                no = 0;
                if(count > sizeof(SpInt)){
                    throw SpException("too big size. (scanbyte '%D')");
                }
                for(int i=0; i < count; i++){
		  //#if defined(LINUX) || defined(OSX)
                    stream->read((char*)&byte, 1);
		    //#else
		    //                    stream->read(&byte, 1);
		    //#endif
                    if(stream->fail()){
                        throw SpException("can't read byte");
                    }
                    no = (no << 8) | byte;
                }
                // set result
                if(result.isNil()){
                    result.setInt(no);
                }else if(tpl == NULL){
                    tpl = new SpTuple(result);
                    result.setObject(tpl);
                    temp.setInt(no);
                    tpl->append(temp);
                }else{
                    temp.setInt(no);
                    tpl->append(temp);
                }
                break;
            case 'b':  // byte
                bvec.push_back(count);
                bary = new SpByteArray(bvec);
                ptr = bary->getBuffer();
                for(int i=0; i < count; i++, ptr++){
		  //#if defined(LINUX) || defined(OSX)
                    stream->read((char*)&byte, 1);
		    //#else
		    //                    stream->read(&byte, 1);
		    //#endif
                    if(stream->fail()){
                        throw SpException("can't read byte");
                    }
                    *ptr = byte;
                }
                // set result
                if(result.isNil()){
                    result.setObject(bary);
                }else if(tpl == NULL){
                    tpl = new SpTuple(result);
                    result.setObject(tpl);
                    temp.setObject(bary);
                    tpl->append(temp);
                }else{
                    temp.setObject(bary);
                    tpl->append(temp);
                }
                break;
/*
            case 'w':  // word (little endian)
                break;
            case 'W':  // word (little endian)
                break;
*/
            case '*':  // '%n*' skip n bytes
                stream->seekg(count, ios::cur);
                if(stream->fail()){
                    throw SpException("can't seek stream (scanbyte)");
                }
                break;
            case 's':  // string
                if(result.isNil()){
                    result = self->ScanByteStr(count);
                }else if(tpl == NULL){
                    tpl = new SpTuple(result);
                    result.setObject(tpl);
                    temp = self->ScanByteStr(count);
                    tpl->append(temp);
                }else{
                    temp = self->ScanByteStr(count);
                    tpl->append(temp);
                }
                break;
//            case '{':
//                break;
            default:
                throw SpException("illegal format. (scanbyte)");
            }
        }else{
            throw SpException("illegal format (scanbyte).");
        }
    }

    return SpValueResult(result);
}

SpValue& StreamReader::ScanByteStr(int count)
{
    unsigned char byte;
    SpValue result;
    SpString* str = new SpString();
    result.setNewObject(str);

    for(int i=0; i < count; i++){
      //#if defined(LINUX) || defined(OSX)
      stream->read((char*)&byte, 1);
      //#else
      //        stream->read(&byte, 1);
      //#endif
        if(stream->fail()){
            throw SpException("can't read string (scanbyte)");
        }
        str->addSpChar((SpChar)byte);
    }
    return SpValueResult(result);
}

class ScanByteStrReader : public Reader {
private:
    Reader* reader;
    int index;
    int max;

public:
    ScanByteStrReader(Reader* r, int n) : reader(r), max(n) {
        index = 0;
    }
    SpChar readCh(){
        if(eof()){
            return (SpChar)NULL;
        }
        index++;
        return reader->readCh();
    }
    bool eof(){
        return reader->eof() || (index >= max);
    }
    bool is_open(){
        return reader->is_open();
    }
};

SpValue& ReadEncoder::ScanByteStr(int count)
{
    ScanByteStrReader* r = new ScanByteStrReader(reader, count);
    SpValue taker;
    taker.setNewObject(r);
    SpValue result;

    Reader* old_reader = reader;
    setReader(r);
    try{
        result = ReadLine();
    }catch(...){
        setReader(old_reader);
        throw;
    }
    setReader(old_reader);

    return SpValueResult(result);
}

SpValue& Reader::ReadLine()
{
    if(eof()){
        return NullString;
    }

    SpChar c;
    SpString* str = new SpString;

    while(true){
        c = ReadChar();
#ifdef __WIN32__
        if(c == JIS(0x0d)){
            SpChar c2 = ReadChar();
            if(c2 == JIS(0x0a)){
                break;
            }else{
                UnreadChar(c2);
            }
        }
#endif
        if(isEOL(c) || isEOF(c)){
            break;
        }
        str->addSpChar(c);
    }
    return SpObjectResult(str);
}

SpValue& Reader::ReadLineS()
{
    if(eof()){
        return NilObject;
    }
    SpValue line = ReadLine();

    SpCons* head = new SpCons(line);
    SpCons* ptr;
    for(ptr=head; !eof() ; ptr=(SpCons*)ptr->next()){
        line = ReadLine();
        ptr->append(line);
    }
    return SpObjectResult(head);
}

SpValue& Reader::toString()
{
    SpString* str;

    str = new SpString();
    *str = "#<reader: '";
    *str += getFilename();
    *str += "'>";
    return SpObjectResult(str);
}

SpValue& Reader::onMessage(SpValue& rec, SpValue& msg)
{
    return FileInMsgHandler(rec, msg);
}

// FileIn Primitives
SpValue& Reader::prim_close(SpValue& fin)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not file_in (close)");
    }
    ptr->close();
    return TrueObject;
}

SpValue& Reader::prim_readChar(SpValue& fin)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not reader (readChar)");
    }
    SpChar c = ptr->ReadChar();
    return SpCharResult(c);
}

SpValue& Reader::prim_peekChar(SpValue& fin)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not reader (peekChar)");
    }
    SpChar c = ptr->ReadChar();
    ptr->UnreadChar(c);
    return SpCharResult(c);
}

SpValue& Reader::prim_unreadChar(SpValue& fin, SpValue& ch)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not reader (unreadChar)");
    }
    if(!ch.isChar()){
        throw SpException("not char (unreadChar)");
    }
    ptr->UnreadChar(ch.getChar());
    return TrueObject;
}

SpValue& Reader::prim_readLine(SpValue& fin)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not reader (readLine)");
    }
    return SpValueResult(ptr->ReadLine());
}

SpValue& Reader::prim_readLineS(SpValue& fin)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not reader (readLineS)");
    }
    return SpValueResult(ptr->ReadLineS());
}

SpValue& Reader::prim_eof(SpValue& fin)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not reader (eof)");
    }
    if(ptr->eof()){
        return TrueObject;
    }
    return FalseObject;
}

SpValue& Reader::prim_closed(SpValue& fin)
{
    Reader* ptr = fin.asReader();
    if(ptr == NULL){
        throw SpException("not reader (closed)");
    }
    if(ptr->is_open()){
        return FalseObject;
    }
    return TrueObject;
}

SpValue& Reader::prim_read(SpValue& fin, SpValue& n)
{
    Reader* ptr = fin.asReader();
    if(!n.isInt()){
        throw SpException("not int (read)");
    }
    SpInt i = n.getInt();
    return SpValueResult(ptr->Read(i));
}

SpValue& Reader::prim_readbyte(SpValue& fin, SpValue& n, SpValue& byte_vector)
{
    Reader* ptr = fin.asReader();
    if(!n.isInt()){
        throw SpException("not int (readbyte)");
    }
    if(!byte_vector.isByteArray()){
        throw SpException("not byte array(readbyte)");
    }
    SpInt i = n.getInt();
    SpByteArray* ary = byte_vector.asByteArray();
    if(i > ary->length()){
        throw SpException("overflow (readbyte)");
    }
    unsigned char* buffer = ary->getBuffer();

    return ptr->ReadByte(i, buffer);
}

SpValue& Reader::prim_skipbyte(SpValue& fin, SpValue& n)
{
    Reader* ptr = fin.asReader();
    if(!n.isInt()){
        throw SpException("not int (skipbyte)");
    }
    SpInt i = n.getInt();

    return ptr->SkipByte(i);
}

SpValue& Reader::prim_scanbyte(SpValue& fin, SpValue& str)
{
    Reader* ptr = fin.asReader();
    if(!str.isString()){
        throw SpException("not string (scanbyte)");
    }
    SpString* s = str.asString();

    return ptr->ScanByte(s, ptr);
}

// init Message Handler
void Reader::init()
{
    SpValue PrimClose(new SpPrim1(prim_close));
    FileInMsgHandler.append(SymClose, PrimClose);

    SpValue SymReadChar(new SpSymbol("readchar"));
    SpValue PrimReadChar(new SpPrim1(prim_readChar));
    FileInMsgHandler.append(SymReadChar, PrimReadChar);

    SpValue SymPeekChar(new SpSymbol("peekchar"));
    SpValue PrimPeekChar(new SpPrim1(prim_peekChar));
    FileInMsgHandler.append(SymPeekChar, PrimPeekChar);

    SpValue SymUnreadChar(new SpSymbol("unreadchar"));
    SpValue PrimUnreadChar(new SpPrim2(prim_unreadChar));
    FileInMsgHandler.append(SymUnreadChar, PrimUnreadChar);

    SpValue SymReadLine(new SpSymbol("readline"));
    SpValue PrimReadLine(new SpPrim1(prim_readLine));
    FileInMsgHandler.append(SymReadLine, PrimReadLine);

    SpValue SymReadLineS(new SpSymbol("readlines"));
    SpValue PrimReadLineS(new SpPrim1(prim_readLineS));
    FileInMsgHandler.append(SymReadLineS, PrimReadLineS);

    SpValue PrimIsEOF(new SpPrim1(prim_eof));
    FileInMsgHandler.append(SymIsEOF, PrimIsEOF);

    SpValue SymClosed(new SpSymbol("closed?"));
    SpValue PrimClosed(new SpPrim1(prim_closed));
    FileInMsgHandler.append(SymClosed, PrimClosed);

    SpValue SymRead(new SpSymbol("read"));
    SpValue PrimRead(new SpPrim2(prim_read));
    FileInMsgHandler.append(SymRead, PrimRead);

    SpValue SymReadByte(new SpSymbol("readbyte"));
    SpValue PrimReadByte(new SpPrim3(prim_readbyte));
    FileInMsgHandler.append(SymReadByte, PrimReadByte);

    SpValue SymSkipByte(new SpSymbol("skipbyte"));
    SpValue PrimSkipByte(new SpPrim2(prim_skipbyte));
    FileInMsgHandler.append(SymSkipByte, PrimSkipByte);

    SpValue SymScanByte(new SpSymbol("scanbyte"));
    SpValue PrimScanByte(new SpPrim2(prim_scanbyte));
    FileInMsgHandler.append(SymScanByte, PrimScanByte);
}

/*
 * class Writer
 */

Writer& Writer::operator<<(const char* str)
{
  for(const char* ptr = str; *ptr; ptr++){
    writeCh(*ptr);
  }
  return *this;
}

Writer& Writer::operator<<(const SpInt i)
{
  char buf[50];
  sprintf(buf, "%d", i);
  operator<<(buf);
  return *this;
}

Writer& Writer::operator<<(const SpReal r)
{
  char buf[50];
  sprintf(buf, "%g", r);
  operator<<(buf);
  return *this;
}

void Writer::writeTextTronCode(SpChar c)
{
  char buf[30];
  sprintf(buf, "&T%06X;", c);
  operator<<(buf);
}

void Writer::write(SpList* ptr)
{
    SpValue next;
    SpList* list = ptr;

    writeCh('[');
    while(true){
        // Write(list->value());
        SpValue val;
        val = list->value();
        Write(val);
        next = list->nextList();
        if(next.isNil()){
            break;
        }
        list = (SpList*)next.getObject();
        writeCh(',');
        writeCh(' ');
    }
    writeCh(']');
}

void Writer::write(SpObject& obj)
{
    SpValue v;
    SpString* s;

    if(obj.isList()){
        SpList* list = (SpList*)&obj;
        write(list);
    }else{
        if(obj.isString()){
            s = (SpString*)&obj;
        }else{
            v = obj.toString();
            s = dynamic_cast<SpString*>(v.getObject());
        }
        SpCharVector::iterator it;
        for(it=s->str.begin(); it != s->str.end(); it++){
            WriteChar(*it);
        }
    }
}

SpValue& Writer::Write(SpValue& val)
{
    if(!val.isObject()){
        operator<<(val.toCStringWithEncoder());
    }else{
        SpObject* obj = val.getObject();
        SpString* s;
        if(obj->isList()){
            SpList* list = (SpList*)obj;
            write(list);
        }else{
            if(obj->isString()){
                s = (SpString*)obj;
            }else{
                SpValue v = obj->toString();
                s = dynamic_cast<SpString*>(v.getObject());
            }
            SpCharVector::iterator it;
            for(it=s->str.begin(); it != s->str.end(); it++){
                WriteChar(*it);
            }
        }
    }
    return TrueObject;
}

SpValue& Writer::WriteLine(SpValue& val)
{
    Write(val);
    Terpri();
    return TrueObject;
}

SpValue& Writer::WriteLineS(SpValue& list)
{
    if(list.isNil()){
        return TrueObject;
    }
    if(!list.isList()){
        throw SpException("not list(writer.writelines)");
    }
    SpList* ptr = list.asList();
    for(; ptr != NULL; ptr = ptr->next()){
        SpValue val;
        val = ptr->value();
        Write(val);
        Terpri();
    }
    return TrueObject;
}

SpValue& Writer::Terpri()
{
    WriteChar(MakeSpChar(CodeJIS,'\n'));
    flush();
    return TrueObject;
}

/*
 * class StreamWriter
 */
void StreamWriter::writeCh(unsigned char ch)
{
  *stream << (unsigned char)ch;
}

SpValue& StreamWriter::WriteByte(SpInt n, unsigned char* buf)
{
    stream->write((char*)buf, n);

    if(stream->fail()){
        throw SpFileException("can't writebyte");
    }

    return SpIntResult(n);
}

/*
 * class WriteEncoder
 */

WriteEncoder& WriteEncoder::operator<<(SpValue& v)
{
    switch(v.typ){
    case TypeNil:
        *writer << "nil";
        break;
    case TypeBool:
        if(v.Bool){
            *writer << "true";
        }else{
            *writer << "false";
        }
        break;
    case TypeInt:
        *writer << v.Int;
        break;
    case TypeChar:
        WriteChar(v.Char);
        break;
    case TypeReal:
        *writer << v.Real;
        break;
    case TypeObject:
        SpObject* obj = v.getObject();
        operator<<(*obj);
        break;
    }
    return *this;
}

static SpPrim2* ptrPrimWriteChar;
static SpPrim2* ptrPrimWrite;
static SpPrim2* ptrPrimWriteLine;
static SpPrim2* ptrPrimWriteLineS;

SpValue& Writer::toString()
{
    SpString* str;

    str = new SpString();
    *str = "#<output-file: '";
    *str += getFilename();
    *str += "'>";
    return SpObjectResult(str);
}

SpValue& Writer::onMessage(SpValue& rec, SpValue& msg)
{
    return FileOutMsgHandler(rec, msg);
}

// FileOut Primitives
SpValue& Writer::prim_close(SpValue& fout)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (close)");
    }
    ptr->close();
    return TrueObject;
}

SpValue& Writer::prim_writeChar(SpValue& fout, SpValue& ch)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (writeChar)");
    }
    if(!ch.isChar()){
        throw SpException("not char (writeChar)");
    }
    ptr->WriteChar(ch.getChar());
    return (*ptrPrimWriteChar)(fout);
}

SpValue& Writer::prim_write(SpValue& fout, SpValue& val)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (write)");
    }
    ptr->Write(val);
    return (*ptrPrimWrite)(fout);
}

SpValue& Writer::prim_writeLine(SpValue& fout, SpValue& val)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (writeLine)");
    }
    ptr->WriteLine(val);
    return (*ptrPrimWriteLine)(fout);
}

SpValue& Writer::prim_writeLineS(SpValue& fout, SpValue& val)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (writeLine)");
    }
    ptr->WriteLineS(val);
    return (*ptrPrimWriteLine)(fout);
}

SpValue& Writer::prim_flush(SpValue& fout)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (flush)");
    }
    ptr->flush();
    return TrueObject;
}

SpValue& Writer::prim_terpri(SpValue& fout)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (flush)");
    }
    ptr->Terpri();
    return TrueObject;
}

SpValue& Writer::prim_closed(SpValue& fout)
{
    Writer* ptr = fout.asWriter();
    if(ptr == NULL){
        throw SpException("not writer (closed)");
    }
    if(ptr->is_open()){
        return FalseObject;
    }
    return TrueObject;
}

SpValue& Writer::prim_writebyte(SpValue& fout, SpValue& n, SpValue& byte_vector)
{
    Writer* ptr = fout.asWriter();
    if(!n.isInt()){
        throw SpException("not int (writebyte)");
    }
    if(!byte_vector.isByteArray()){
        throw SpException("not byte array(writebyte)");
    }
    SpInt i = n.getInt();
    SpByteArray* ary = byte_vector.asByteArray();
    if(i > ary->length()){
        throw SpException("overflow (writebyte)");
    }
    unsigned char* buffer = ary->getBuffer();

    return  ptr->WriteByte(i, buffer);
}


// init Message Handler
void Writer::init()
{
    SpValue PrimClose(new SpPrim1(prim_close));
    FileOutMsgHandler.append(SymClose, PrimClose);

    ptrPrimWriteChar = new SpPrim2(prim_writeChar);
    SpValue PrimWriteChar(ptrPrimWriteChar);
    FileOutMsgHandler.append(SymWriteChar, PrimWriteChar);

    ptrPrimWrite = new SpPrim2(prim_write);
    SpValue PrimWrite(ptrPrimWrite);
    FileOutMsgHandler.append(SymWrite, PrimWrite);

    ptrPrimWriteLine = new SpPrim2(prim_writeLine);
    SpValue PrimWriteLine(ptrPrimWriteLine);
    FileOutMsgHandler.append(SymWriteLine, PrimWriteLine);

    SpValue SymWriteLineS(new SpSymbol("writelines"));
    ptrPrimWriteLineS = new SpPrim2(prim_writeLineS);
    SpValue PrimWriteLineS(ptrPrimWriteLineS);
    FileOutMsgHandler.append(SymWriteLineS, PrimWriteLineS);

    SpValue PrimFlush(new SpPrim1(prim_flush));
    FileOutMsgHandler.append(SymFlush, PrimFlush);

    SpValue PrimClosed(new SpPrim1(prim_closed));
    FileOutMsgHandler.append(SymClosed, PrimClosed);

    SpValue SymWriteByte(new SpSymbol("writebyte"));
    SpValue PrimWriteByte(new SpPrim3(prim_writebyte));
    FileOutMsgHandler.append(SymWriteByte, PrimWriteByte);
}

