/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002,2003 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: EucJPRW.cpp,v 1.27 2004/05/22 05:22:34 randy Exp $
 */

#include "soopy.h"

/*
 * Class EucJPReader
 */

#include <stdio.h>

SpChar EucJPReadEncoder::readCh()
{
    SpChar c;
    int ch, ku, ten;

    if(count >= 0){
        return buf[count--];
    }
    if(eof()){ return (SpChar)NULL; }
    c = reader->ReadChar();
    if(isEOF(c)){ return (SpChar)NULL; }

    if(!EqualCharCode(c, CodeJIS)){
        return c;
    }
    ch  = SpCharGetChar(c);
    ku  = (ch >> 8);
    ten = (ch & 0xFF);
    if(ku == 0){
        return ten;
    }else{
        // first byte
        ku  += 0xA0 - 0x20;
        // second byte
        ten += 0xA0 - 0x20;
        buf[0] = ten;
        count = 0;
    }
    return ku;
}

SpChar EucJPReadDecoder::readCh()
{
    char c;
    int ch1, ch2;

    if(eof()){ return (SpChar)NULL; }
    c = reader->ReadChar();
    if(isEOF(c)){ return (SpChar)NULL; }
    ch1 = c & 0xff;
    if(0xA1 <= ch1 && ch1 <= 0xFE){ // JIS X 0208
        c = reader->ReadChar();
        ch2 = c & 0xff;
        ch1 = ch1 - 0xA0 + 0x20;
        ch2 = ch2 - 0xA0 + 0x20;
        return MakeSpChar(CodeJIS, (ch1<<8) | ch2);
    }
    return MakeSpChar(CodeJIS, c);
}

/*
 * Class EucJPWriteEncoder
 */

void EucJPWriteEncoder::WriteChar(SpChar c)
{
    int ch, ku, ten;

    if(!EqualCharCode(c, CodeJIS)){
        writeTextTronCode(c);
        return;
    }
    ch  = SpCharGetChar(c);
    ku  = (ch >> 8);
    ten = (ch & 0xFF);
    if(ku == 0){
        writer->WriteChar((char)ten);
    }else{
        ku  += 0xA0 - 0x20;
        ten += 0xA0 - 0x20;
        // put first byte
        writer->WriteChar((char)ku);
        // put second byte
        writer->WriteChar((char)ten);
    }
}

/*
 * Class EucJPWriteDecoder
 */

void EucJPWriteDecoder::WriteChar(SpChar c)
{
    int ch2;

    if(count == 0){
        if(c > 0xFF){ // not jis
            if(isEOL(c)){
                writer->WriteChar(c);
            }else{
                writeTextTronCode(c);
            }
        }else if(0xA1 <= c && c <= 0xFE){ // JIS X 0208
            /*
            if(0xA1 <= ch1 && ch1 <= 0xDF){ // when hankaku KANA
                // write Zenkaku KANA
            }
            */
            ch1 = c;
            count++;
        }else{
            writer->WriteChar(c);
        }
        return;
    }
    // count == 1
    count = 0;
    ch2 = c;
    ch1 = ch1 - 0xA0 + 0x20;
    ch2 = ch2 - 0xA0 + 0x20;
    writer->WriteChar(MakeSpChar(CodeJIS, (ch1<<8) | ch2));
}

/*
 * EucJPFile
 *   openIn, openOut
 */

SpValue& eucjp_openIn(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (EucJP.openIn)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ifstream* fin = new ifstream;
#ifdef __WIN32__
    fin->open(filename, ios::binary);
#else
    fin->open(filename);
#endif
    if(fin->fail()){
        delete fin;
        throw SpException("can't open file");
    }
    FileStreamReader* sr = new FileStreamReader(fin, filename);
    ReadEncoder* encoder = new EucJPReadDecoder(sr);
    return SpObjectResult(encoder);
}

SpValue& eucjp_openOut(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (EucJP.openOut)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ofstream* fout = new ofstream;
#ifdef __WIN32__
    fout->open(filename, ios::binary);
#else
    fout->open(filename);
#endif
    if(fout->fail()){
        delete fout;
        throw SpException("can't open file");
    }
    FileStreamWriter* sw = new FileStreamWriter(fout, filename);
    WriteEncoder* encoder = new EucJPWriteEncoder(sw);
    return SpObjectResult(encoder);
}

SpValue& eucjp_openAppend(SpValue& v)
{
    SpValue name = v.eval();
    if(!name.isString()){
        throw SpException("filename is not string (EucJP.openAppend)");
    }
    SpString* str = name.asString();
    char* filename = (char*)str->toCStringWithEncoder();
    ofstream* fout = new ofstream;
#ifdef __WIN32__
    fout->open(filename, ios::app | ios::binary);
#else
    fout->open(filename, ios::app);
#endif
    if(fout->fail()){
        delete fout;
        throw SpException("can't open file");
    }
    FileStreamWriter* sw = new FileStreamWriter(fout, filename);
    WriteEncoder* encoder = new EucJPWriteEncoder(sw);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& eucjp_encoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (EucJP.encoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new EucJPReadEncoder(reader);
    return SpObjectResult(encoder);
}

SpValue& eucjp_decoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (EucJP.decoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new EucJPReadDecoder(reader);
    return SpObjectResult(encoder);
}

SpValue& eucjp_encoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (EucJP.encoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new EucJPWriteEncoder(writer);
    return SpObjectResult(encoder);
}

SpValue& eucjp_decoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (EucJP.decoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new EucJPWriteDecoder(writer);
    return SpObjectResult(encoder);
}

// init
void initEucJPFile()
{
    SpValue SymEucJP(new SpSymbol("eucjp"));
    SpNameSpace* ns = new SpNameSpace;
    SpValue NSEucJP(ns);
    PMainNameSpace->internConst(SymEucJP, NSEucJP);

    SpValue PrimOpenIn(new SpPrim1(&eucjp_openIn));
    ns->internConst(SymOpenIn, PrimOpenIn);
    SpValue PrimEncoderIn(new SpPrim1(&eucjp_encoderIn));
    ns->internConst(SymEncoderIn, PrimEncoderIn);
    SpValue PrimDecoderIn(new SpPrim1(&eucjp_decoderIn));
    ns->internConst(SymDecoderIn, PrimDecoderIn);

    SpValue PrimOpenOut(new SpPrim1(&eucjp_openOut));
    ns->internConst(SymOpenOut, PrimOpenOut);
    SpValue PrimOpenAppend(new SpPrim1(&eucjp_openAppend));
    ns->internConst(SymOpenAppend, PrimOpenAppend);
    SpValue PrimEncoderOut(new SpPrim1(&eucjp_encoderOut));
    ns->internConst(SymEncoderOut, PrimEncoderOut);
    SpValue PrimDecoderOut(new SpPrim1(&eucjp_decoderOut));
    ns->internConst(SymDecoderOut, PrimDecoderOut);
}

