/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: Base64.cpp,v 1.7 2004/03/24 13:45:51 randy Exp $
 */

#include <stdio.h>
#include "soopy.h"

const char* BASE64TBL = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

/*
 * Class Base64Reader
 */

SpChar Base64ReadEncoder::readCh()
{
    char c;
    int src;

    switch(count){
      case 1:
      case 2:
      case 3:
        return results[count++];
      default:
        // read 3 chars
        c = reader->ReadChar();
        if(eof()){ return (SpChar)NULL; }
        src = (c & 0xff) << 16;
        c = reader->ReadChar();
        if(eof()){
            c = 0;
        }
        src = src | ((c & 0xff) << 8);
        c = reader->ReadChar();
        if(eof()){
            c = 0;
        }
        src = src | (c & 0xff);
        // split to 4 chars
        results[0] = MakeSpChar(CodeJIS, BASE64TBL[(src >> 18) & 0x3F]);
        results[1] = MakeSpChar(CodeJIS, BASE64TBL[(src >> 12) & 0x3F]);
        results[2] = MakeSpChar(CodeJIS, BASE64TBL[(src >> 6) & 0x3F]);
        results[3] = MakeSpChar(CodeJIS, BASE64TBL[src & 0x3F]);
        if(eof()){
            if((src & 0x3F) == 0){
                results[3] = '=';
                if(((src >> 6) & 0x3F) == 0){
                    results[2] = '=';
                }
            }
        }
        // set count
        count = 1;
    }

    return results[0];
}

/*
 * Class Base64WriteEncoder
 */

//void Base64WriteEncoder::writeCh(unsigned char c)
void Base64WriteEncoder::WriteChar(SpChar c)
{
    switch(count){
      case 0:
        src = (c & 0xff) << 16;
        count++;
        break;
      case 1:
        src = src | ((c & 0xff) << 8);
        count++;
        break;
      case 2:
        src = src | (c & 0xff);
        count = 0;
        // split to 4 chars
        writer->WriteChar(BASE64TBL[(src >> 18) & 0x3F]);
        writer->WriteChar(BASE64TBL[(src >> 12) & 0x3F]);
        writer->WriteChar(BASE64TBL[(src >> 6) & 0x3F]);
        writer->WriteChar(BASE64TBL[src & 0x3F]);
        break;
    }
}

void Base64WriteEncoder::flush()
{
    if(count != 0){
        count = 0;
        // split to 4 chars
        writer->WriteChar(BASE64TBL[(src >> 18) & 0x3F]);
        writer->WriteChar(BASE64TBL[(src >> 12) & 0x3F]);
        if((src & 0x3F) == 0){
            if(((src >> 6) & 0x3F) == 0){
                writer->WriteChar('=');
                writer->WriteChar('=');
            }else{
                writer->WriteChar(BASE64TBL[(src >> 6) & 0x3F]);
                writer->WriteChar('=');
            }
        }else{
            writer->WriteChar(BASE64TBL[(src >> 6) & 0x3F]);
            writer->WriteChar(BASE64TBL[src & 0x3F]);
        }
    }
    writer->flush();
}

/*
 * Class Base64Reader
 */

static bool isBase64Char(char c)
{
    if(isdigit(c)){
        return true;
    }
    if(isalpha(c)){
        return true;
    }
    if((c == '+') ||
       (c == '/') ||
       (c == '='))
    {
        return true;
    }
    return false;
}

static int decodeChar(int c)
{
    int result;
    if(isupper(c)){
        result = c - 'A';
    }else if(islower(c)){
        result = c - 'a' + 26;
    }else if(isdigit(c)){
        result = c - '0' + 52;
    }else{
        switch(c){
          case '+':
            result = 62;
            break;
          case '/':
            result = 63;
            break;
          case '=':
            result = 0;
            break;
        }
    }
    return result;
}

SpChar Base64ReadDecoder::readCh()
{
    char c;
    int ch1, ch2, ch3, ch4;
    bool p = false;

    switch(count){
      case 1:
      case 2:
        return results[count++];
      default:
        // read 4 chars
        while(true){
            c = reader->ReadChar();
            if(eof()){ return (SpChar)NULL; }
            if(isBase64Char(c)){
                break;
            }
        }
        ch1 = c & 0xff;
        while(true){
            c = reader->ReadChar();
            if(eof()){ return (SpChar)NULL; }
            if(isBase64Char(c)){
                break;
            }
        }
        ch2 = c & 0xff;
        while(true){
            c = reader->ReadChar();
            if(eof()){ return (SpChar)NULL; }
            if(isBase64Char(c)){
                break;
            }
        }
        if(c == '='){
            p = true;
        }else{
            ch3 = c & 0xff;
        }
        while(true){
            c = reader->ReadChar();
            if(eof()){ return (SpChar)NULL; }
            if(isBase64Char(c)){
                break;
            }
        }
        if(c == '='){
            p = true;
        }else{
            ch4 = c & 0xff;
        }

        // decode
        int i;
        i = decodeChar(ch1) << 18;
        i = i | (decodeChar(ch2) << 12);
        i = i | (decodeChar(ch3) << 6);
        i = i | decodeChar(ch4);
        // set results
        results[0] = MakeSpChar(CodeJIS, (i >> 16) & 0xFF);
        if(p){
            if((i & 0xFF) == 0){
                if(((i >> 8) & 0xFF) == 0){
                    results[1] = (SpChar)NULL;
                    results[2] = (SpChar)NULL;
                }else{
                    results[1] = MakeSpChar(CodeJIS, (i >> 8) & 0xFF);
                    results[2] = (SpChar)NULL;
                }
            }else{
                results[1] = MakeSpChar(CodeJIS, (i >> 8) & 0xFF);
                results[2] = MakeSpChar(CodeJIS, i & 0xFF);
            }
        }else{
            results[1] = MakeSpChar(CodeJIS, (i >> 8) & 0xFF);
            results[2] = MakeSpChar(CodeJIS, i & 0xFF);
        }

        // set count
        count = 1;
    }

    return results[0];
}

/*
 * Class Base64WriteDecoder
 */

//void Base64WriteDecoder::writeCh(unsigned char c)
void Base64WriteDecoder::WriteChar(SpChar c)
{
    if(!isBase64Char(c)){
        return;
    }
    switch(count){
      case 0:
        src = decodeChar(c) << 18;
        count++;
        break;
      case 1:
        src = src | (decodeChar(c) << 12);
        count++;
        break;
      case 2:
        if(c != '='){
            src = src | (decodeChar(c) << 6);
        }else{
            eof = true;
        }
        count++;
        break;
      case 3:
        if(c != '='){
            src = src | decodeChar(c);
        }else{
            eof = true;
        }
        count = 0;
        // split to 3 chars
        writer->WriteChar((src >> 16) & 0xFF);
        if(eof){
            if((src & 0xFF) == 0){
                if(((src >> 8) & 0xFF) != 0){
                    writer->WriteChar((src >> 8) & 0xFF);
                }
            }else{
                writer->WriteChar((src >> 8) & 0xFF);
                writer->WriteChar(src & 0xFF);
            }
        }else{
            writer->WriteChar((src >> 8) & 0xFF);
            writer->WriteChar(src & 0xFF);
        }
        break;
    }
}

void Base64WriteDecoder::flush()
{
    if(count != 0){
        count = 0;
        // split to 3 chars
        writer->WriteChar((src >> 16) & 0xFF);
        if(eof){
            if((src & 0xFF) == 0){
                if(((src >> 8) & 0xFF) != 0){
                    writer->WriteChar((src >> 8) & 0xFF);
                }
            }else{
                writer->WriteChar((src >> 8) & 0xFF);
                writer->WriteChar(src & 0xFF);
            }
        }else{
            writer->WriteChar((src >> 8) & 0xFF);
            writer->WriteChar(src & 0xFF);
        }
    }
    writer->flush();
}

/*
 * Base64
 */

SpValue& base64_encoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (Base64.encoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new Base64ReadEncoder(reader);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& base64_encoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (Base64.encoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new Base64WriteEncoder(writer);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& base64_decoderIn(SpValue& v)
{
    SpValue r = v.eval();
    if(!r.isReader()){
        throw SpException("not reader (Base64.decoderIn)");
    }
    Reader* reader = r.asReader();
    ReadEncoder* encoder = new Base64ReadDecoder(reader);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

SpValue& base64_decoderOut(SpValue& v)
{
    SpValue w = v.eval();
    if(!w.isWriter()){
        throw SpException("not writer (Base64.decoderOut)");
    }
    Writer* writer = w.asWriter();
    WriteEncoder* encoder = new Base64WriteDecoder(writer);
    //    static SpValue result;
    //    result.setNewObject(encoder);
    //    return result;
    return SpObjectResult(encoder);
}

// init
void initBase64()
{
    SpValue SymBase64(new SpSymbol("base64"));
    SpNameSpace* ns = new SpNameSpace;
    SpValue NSBase64(ns);
    PMainNameSpace->internConst(SymBase64, NSBase64);

    SpValue PrimEncoderIn(new SpPrim1(&base64_encoderIn));
    ns->internConst(SymEncoderIn, PrimEncoderIn);
    SpValue PrimDecoderIn(new SpPrim1(&base64_decoderIn));
    ns->internConst(SymDecoderIn, PrimDecoderIn);

    SpValue PrimEncoderOut(new SpPrim1(&base64_encoderOut));
    ns->internConst(SymEncoderOut, PrimEncoderOut);
    SpValue PrimDecoderOut(new SpPrim1(&base64_decoderOut));
    ns->internConst(SymDecoderOut, PrimDecoderOut);
}

