﻿using System;
using System.Windows.Forms;
using System.IO;
using System.Xml.Linq;
using WeifenLuo.WinFormsUI.Docking;
using Somali.Base;

namespace Somali.EnvControl
{
    class LayoutSecretary : Somali.Base.BaseSecretary
    {
        // レイアウトファイル名
        private readonly string layoutFileName = global::Somali.Properties.Resources.LAYOUT_FILE;
        // Window情報ファイル名
        private readonly string windStateFileName = global::Somali.Properties.Resources.WINDOW_STATE_FILE;

        private IWindowPreservable _wsp;
        private DockPanel _dockPanel;

        public LayoutSecretary( IWindowPreservable wndStPrObj, DockPanel dockPanel )
        {
            if ( wndStPrObj.IsContainDockPanel( dockPanel ) == true )
            {
                this._wsp = wndStPrObj;
                this._dockPanel = dockPanel;
            }
            else
            {
                throw new ArgumentException( "指定されたDockPanelはメンバではありません。", dockPanel.Name );
            }
        }

        /// <summary>
        /// レイアウトの初期化
        /// </summary>
        public void InitializeLayout()
        {
            string dirPath = Util.GetAppSettingDir();

            // Window状態復元
            string filePath = String.Format( @"{0}\{1}", dirPath, windStateFileName );
            if ( File.Exists( filePath ) )
            {
                LoadWindowState( filePath );
            }

            // レイアウトファイル読み込み
            filePath = String.Format( @"{0}\{1}", dirPath, layoutFileName );
            if ( File.Exists( filePath ) )
            {
                LoadDockLayout( filePath );
            }
            else
            {
                // TODO:ファイルが存在しない場合、デフォルトファイルを生成

            }
        }

        /// <summary>
        /// レイアウトの保存
        /// </summary>
        public void PreserveLayout()
        {
            string dirPath = Util.GetDirPathWithCreate( new Util.FolderSelector( Util.GetAppSettingDir ) );

            // Window状態保存
            string filePath = String.Format( @"{0}\{1}", dirPath, windStateFileName );
            SaveWindowState( filePath );

            // DockPanelレイアウト保存
            filePath = String.Format( @"{0}\{1}", dirPath, layoutFileName );
            SaveDockLayout( filePath );
        }

        #region レイアウト保存
        /// <summary>
        /// Window状態の保存
        /// </summary>
        /// <param name="filePath"></param>
        private void SaveWindowState( string filePath )
        {
            XElement windXml = new XElement( "SomaliWindow",
                                    new XElement( "X",
                                        this._wsp.Location.X, new XAttribute( "Type", this._wsp.Location.X.GetType() ) ),
                                    new XElement( "Y",
                                        this._wsp.Location.Y, new XAttribute( "Type", this._wsp.Location.Y.GetType() ) ),
                                    new XElement( "WindowState",
                                        this._wsp.WindowState, new XAttribute( "Type", this._wsp.WindowState.GetType() ) ),
                                    new XElement( "Height",
                                        this._wsp.Height, new XAttribute( "Type", this._wsp.Height.GetType() ) ),
                                    new XElement( "Width",
                                        this._wsp.Width, new XAttribute( "Type", this._wsp.Width.GetType() ) ),
                                    new XElement( "NormalizeX",
                                        this._wsp.NormalizeLocation.X, new XAttribute( "Type", this._wsp.NormalizeLocation.X.GetType() ) ),
                                    new XElement( "NormalizeY",
                                        this._wsp.NormalizeLocation.Y, new XAttribute( "Type", this._wsp.NormalizeLocation.Y.GetType() ) ),
                                    new XElement( "NormalizeHeight",
                                        this._wsp.NormalizeSize.Height, new XAttribute( "Type", this._wsp.NormalizeSize.Height.GetType() ) ),
                                    new XElement( "NormalizeWidth",
                                        this._wsp.NormalizeSize.Width, new XAttribute( "Type", this._wsp.NormalizeSize.Width.GetType() ) )
                                    );

            SaveXmlFile( filePath, windXml );

            //IDictionary<string, object> valueMap = new Dictionary<string, object>();

            //valueMap.Add( "X", this._form.Location.X );
            //valueMap.Add( "Y", this._form.Location.Y );
            //valueMap.Add( "WindowState", this._form.WindowState );
            //valueMap.Add( "Height", this._form.Height );
            //valueMap.Add( "Width", this._form.Width );

            //SaveXmlFileByTag( filePath, valueMap );
        }

        /// <summary>
        /// レイアウトをXMLファイルに保存
        /// </summary>
        /// <remarks>User/%username%/AppData/Roaming/Somali配下に設定ファイルを書きだします。</remarks>
        private void SaveDockLayout( string filePath )
        {
            this._dockPanel.SaveAsXml( filePath );
        }
        #endregion

        #region レイアウト復元
        private void LoadWindowState( string filePath )
        {
            XElement windXml = LoadXmlFile( filePath );
            int x, y, h, w;

            Type t = typeof( FormWindowState );
            FormWindowState fws = (FormWindowState)Enum.Parse( t, GetValueByTag( windXml, "WindowState" ) );
            if ( fws.Equals( FormWindowState.Normal ) )
            {
                x = Int32.Parse( GetValueByTag( windXml, "X" ) );
                y = Int32.Parse( GetValueByTag( windXml, "Y" ) );

                h = Int32.Parse( GetValueByTag( windXml, "Height" ) );
                w = Int32.Parse( GetValueByTag( windXml, "Width" ) );

            }
            else
            {
                x = Int32.Parse( GetValueByTag( windXml, "NormalizeX" ) );
                y = Int32.Parse( GetValueByTag( windXml, "NormalizeY" ) );

                h = Int32.Parse( GetValueByTag( windXml, "NormalizeHeight" ) );
                w = Int32.Parse( GetValueByTag( windXml, "NormalizeWidth" ) );

            }
            this._wsp.Location = new System.Drawing.Point( x, y );
            this._wsp.Size = new System.Drawing.Size( w, h );
            this._wsp.NormalizeLocation = new System.Drawing.Point( x, y );
            this._wsp.NormalizeSize = new System.Drawing.Size( w, h );
            this._wsp.WindowState = fws;

        }

        /// <summary>
        /// XMLファイルからレイアウトを復元
        /// </summary>
        /// <remarks>User/%username%/AppData/Roaming/Somali配下にある設定ファイルから読みだします。</remarks>
        private void LoadDockLayout( string filePath )
        {
            //デリゲート生成
            DeserializeDockContent deserializeDockContent
                     = new DeserializeDockContent( GetDockContentFromPersistString );

            this._dockPanel.LoadFromXml( filePath, deserializeDockContent );
        }

        /// <summary>
        /// PersistStringからDockContentを返却
        /// </summary>
        /// <param name="persistString">DockContentの名前</param>
        /// <returns></returns>
        private IDockContent GetDockContentFromPersistString( string persistString )
        {
            //persistStringに応じて、対応するDockContentを生成し、それを返却
            if ( persistString.Equals( typeof( OutlineForm ).ToString() ) )
            {
                return new OutlineForm();
            }
            else if ( persistString.Equals( typeof( ExplorerForm ).ToString() ) )
            {
                return new ExplorerForm();
            }
            else
            {
                //ファイルパス取得
                string[] parsedStrings = persistString.Split( new char[] { '|' } );
                if ( parsedStrings.Length != 2 )
                    return null;

                if ( parsedStrings[0] != typeof( EditorForm ).ToString() )
                    return null;

                EditorForm frmEditor = new EditorForm();
                if ( parsedStrings[1] != string.Empty )
                    frmEditor.FilePath = parsedStrings[1];

                return frmEditor;

            }
        }
        #endregion

        /// <summary>
        /// 指定したTypeのDockContentを取得
        /// </summary>
        /// <param name="formType">DockContentを継承するオブジェクトのType</param>
        /// <returns>DockPanelに存在すればDockContentを返却します。存在しない場合は、nullを返却します。</returns>
        public IDockContent GetDockFormIfExists( Type formType )
        {
            DockContentCollection dc = this._dockPanel.Contents;
            foreach ( IDockContent d in dc )
            {
                if ( d.GetType().Equals( formType ) )
                {
                    return d;
                }
            }

            return null;
        }

        /// <summary>
        /// 指定したTypeのDockContentを取得
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <returns>DockPanelに存在すればDockContentを返却します。存在しない場合は、nullを返却します。</returns>
        public T GetDockFormIfExists<T>()
            where T : class, IDockContent
        {
            DockContentCollection dc = this._dockPanel.Contents;
            foreach ( IDockContent d in dc )
            {
                if ( d.GetType().Equals( typeof( T ) ) )
                {
                    return (T)d;
                }
            }
            return null;
            //return default( T );
        }

        /// <summary>
        /// IDockContentを継承するFormを取得
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="dockForm"></param>
        /// <returns>true:取得成功<br/>false:取得失敗</returns>
        /// <remarks>DockPanelに存在しないFormの場合、outにはnullを設定します。</remarks>
        public bool TryGetDockForm<T>( out T dockForm )
            where T : class, IDockContent
        {
            try
            {
                dockForm = GetDockFormIfExists<T>();
                //dockForm = (T)GetDockFormIfExists( typeof( T ) );
                if ( dockForm != null )
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
            catch ( Exception )
            {
                dockForm = default( T );
                return false;
            }
        }

        /// <summary>
        /// IDockContentを継承するFormの実装インスタンスを取得
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="dockForm"></param>
        /// <returns>true:取得成功<br/>false:取得失敗</returns>
        /// <remarks>DockPanelに存在しないFormの場合、outにはdockFormの新しいインスタンスを設定します。</remarks>
        public bool GetDockFormImpl<T>( out T dockForm )
            where T : class, IDockContent, new()
        {
            try
            {
                if ( TryGetDockForm( out dockForm ) )
                {
                    return true;
                }
                else
                {
                    dockForm = new T();
                    return true;
                }
            }
            catch ( Exception )
            {
                dockForm = default( T );
                return false;
            }
        }

    }
}
